(ns ingestion.api.client.ns-repo-impl
  (:require [clojure.tools.reader.edn :as edn]
            [ksql.gen.protocol :as p]
            [clojure.java.io :as io]
            [ksqldb.client :as client]
            [clojure.tools.logging :as log]))


;(def context (init-context))

(defn get-version [] "0.11.0-2-snapshot")


(defn get-file-list [dir-path]
  (let [grammar-matcher (.getPathMatcher
                          (java.nio.file.FileSystems/getDefault)
                          "glob:*.{edn}")]
    (->> dir-path
         clojure.java.io/file
         file-seq
         (filter #(.isFile %))
         (filter #(.matches grammar-matcher (.getFileName (.toPath %))))
         (map #(.toString (.getFileName (.toPath %))))
         (map (fn [v] (first (clojure.string/split v #"\.")))))))

(defn load-edn-file [dir-path file-name]
  (let [v (str dir-path "/" file-name ".edn")]
    (-> (slurp v)
        (edn/read-string)
        (p/edn-data->context))))


(defn persist-to-file [dir-path file-name file-v]
  (let [p (str dir-path "/" file-name ".edn")
        _ (log/debug "file save " p)
        _ (io/make-parents p)
        fv (-> file-v
               (p/context->edn-data)
               (pr-str))]
    (spit p fv)))


(comment

  (let [p (str (System/getProperty "user.home") "/.colle/" "flubber.edn")]
    (clojure.java.io/make-parents p)
    (spit p "test")
    )



  (let [p (str (System/getProperty "user.home") "/.colle")]

    (get-file-list p)
    )

  )


(defrecord INMeortyNSStore [mapping-context]
  p/NS-Store
  (-get-ns-names [this]
    (keys @mapping-context))
  (-get-ns-by-name [this ns-name]
    (get @mapping-context ns-name))
  #_(-open-ns [this ns-name ns-v]
              (swap! mapping-context (fn [m] (assoc m ns-name ns-v))))
  (-persist-ns [this ns-name ns-v closed?]
    (swap! mapping-context (fn [m] (assoc m ns-name ns-v))))
  (-ns-exits? [this ns-name]
    (get @mapping-context ns-name)))


(defn new-memory-store []
  (->INMeortyNSStore (atom {}))
  #_(->FileNSStore (str (System/getProperty "user.home") "/.colle") (atom {}))

  )


(defrecord FileNSStore [base-dir ns-context-m]
  p/NS-Store
  (-get-ns-names [this]
    (-> (get-file-list base-dir)
        (into (keys @ns-context-m))
        (distinct))
    #_(keys @mapping-context))
  (-get-ns-by-name [this ns-name]
    (if (contains? (into #{} (keys @ns-context-m)) ns-name)
      (get @ns-context-m ns-name)
      (let [ns-set (into #{} (get-file-list base-dir))]
        (when (contains? ns-set ns-name)
          (let [out (load-edn-file base-dir ns-name)]
            (swap! ns-context-m (fn [m] (assoc m ns-name out)))
            out)))))

  (-persist-ns [this ns-name ns-v closed?]
    (if closed?
      (swap! ns-context-m (fn [m] (dissoc m ns-name)))
      (swap! ns-context-m (fn [m] (assoc m ns-name ns-v))))

    (persist-to-file base-dir ns-name ns-v)
    #_(swap! mapping-context (fn [m] (assoc m ns-name ns-v))))
  (-ns-exits? [this ns-name]
    (let [out-set (-> (into #{} (get-file-list base-dir))
                      (into (keys @ns-context-m))
                      )]
      (contains? out-set ns-name))))




(defn new-file-store []
  (->FileNSStore (str (System/getProperty "user.home") "/.colle") (atom {})))


(defonce  ns-store-impl nil )

(defn set-ns-repo [ns-repo ]
  (alter-var-root #'ns-store-impl (constantly ns-repo  ) )
  )

(defn get-ns-repo [] ns-store-impl)



(defmethod p/invoke "df-ns-names"
  [{:keys [request]}]
  (p/get-ns-names (get-ns-repo)))



(defmethod p/invoke "df-ns-new"
  [{:keys [request]}]
;  (clojure.pprint/pprint request)
  (let [app-new-ns-name (get (or request {}) :app-new-ns-name)
        request (or app-new-ns-name "default")
        c (-> (p/get-default-context)
              (assoc :namespace request))]
    (p/persist-ns (get-ns-repo) request c)
    request))



(defmethod p/invoke "df-ns-save"
  [{:keys [app-ns-name]}]
  (let []
    (p/persist-ns (get-ns-repo) app-ns-name p/context false)
    app-ns-name))


(defmethod p/invoke "df-ns-save-as"
  [{:keys [request app-ns-name]}]
  (let [app-new-ns-name (get (or request {}) :app-new-ns-name)
        c (p/copy-context p/context)
        c (assoc c :namespace app-new-ns-name)]
    (p/persist-ns (get-ns-repo) app-ns-name p/context true)
    (p/persist-ns (get-ns-repo) app-new-ns-name c false)
    app-new-ns-name))


(defmethod p/invoke "df-ns-close"
  [{:keys [request app-ns-name]}]
  (let []
    (p/persist-ns (get-ns-repo) app-ns-name p/context true)
    ;(p/get-ns-names)
    ;(p/get (get-ns-repo) app-ns-name c false)
    app-ns-name))


(defmethod p/invoke "df-ns-open"
  [{:keys [request app-ns-name]}]
  (let [app-new-ns-name (get (or request {}) :app-new-ns-name)]
    (p/persist-ns (get-ns-repo) app-ns-name p/context true)
    (p/get-ns-by-name (get-ns-repo) app-new-ns-name )
    ;(p/get-ns-names)
    ;(p/get (get-ns-repo) app-ns-name c false)
    app-new-ns-name))



#_(defmethod p/invoke "df-ns"
  [{:keys [request app-ns-name]}]

  (let [app-new-ns-name (get (or request {}) :app-new-ns-name)

        req? (p/ns-exit? (get-ns-repo) app-new-ns-name)
        ns-name? (p/ns-exit? (get-ns-repo) app-ns-name)]
    (cond

      (and (= app-ns-name app-new-ns-name)
           ns-name?
           req?)

      (let [closed? (get (or request {}) :close false)]
        (p/persist-ns (get-ns-repo) app-ns-name p/context closed?)
        app-new-ns-name)

      (and (not= app-ns-name app-new-ns-name)
           ns-name?
           (= false req?))

      (let [c (p/copy-context p/context)
            c (assoc c :namespace app-new-ns-name)]
        (p/persist-ns (get-ns-repo) app-new-ns-name c)
        app-new-ns-name)

      :else
      (let [request (or app-new-ns-name "default")
            c (-> (p/get-default-context)
                  (assoc :namespace request))]
        (p/persist-ns (get-ns-repo) request c)
        request))))



(defmethod p/invoke "get-context"
  [{:keys [request]}]
  (-> p/context
      (select-keys p/context-user-key-list)
      ))


(defmethod p/invoke "update-context"
  [{:keys [request app-ns-name]}]
  (let [w (if (string? request)
            (edn/read-string request)
            request)
        out (merge p/context w)]
    (p/persist-ns (get-ns-repo) app-ns-name out)
    (-> out
        (select-keys p/context-user-key-list))))


;;;;;;;;;;;;;;;;;;;;;;; testing purpose ;;;;;;;;;;;;;;;;;;,


(defn load-app-config [app-file]
  (-> (slurp app-file)
      (edn/read-string)))


(defn init-context
  "Init app with new context, context contains kafka url, partition no etc. "
  ([]
   (init-context {}))
  ([m]
   (let [m (if (string? m)
             (load-app-config m)
             m)
         m (merge (p/get-default-context) m)
         m (if (get m :offline)
             m
             (client/init-client m))]
     (alter-var-root #'p/context (constantly m)))))


(defn get-context []
  (when (nil? p/context)
    (init-context))
  p/context)


#_(defn reset-md-repo []
    (-> (p/get-md-schema-coll)
        (p/-delete-all-metadata)))


(comment

  (edn/read-string "{:join-split false,\n :connector-url \"http://localhost:8083\",\n :schema-url \"http://localhost:8081\",\n :kafka-rest-url \"http://localhost:8082\",\n :distinct-duration \"10 MINUTES\",\n :partitions 3,\n :kafka-broker \"localhost:9092\",\n :join-window \"WITHIN 5 MINUTES\", \n :error-stream-name \"error\"}\n",)

  )



(defn show-api-doc []
  (->> [
        {:api "init-context" :doc "start compiler with kafka context" :usage "(init-context \" context file path \" )" :type "Ingestion API"}

        {:api "conn-show-connectors-status" :doc "Show connector status " :usage "(invoke-kafka \"show-connectors-status\" \"{connector_name}\"  ) " :type "Kafka API"}
        {:api "conn-show-connectors" :doc "Show connector list " :usage "(invoke-kafka \"show-connectors\"  )" :type "Kafka API"}
        {:api "conn-create-connector-batch" :doc "Create list of connector in kafka  " :usage "(invoke-kafka \"create-connector-batch\" \" {list of connector config} \"  )" :type "Kafka API"}
        {:api "conn-delete-connector" :doc "Delete connector " :usage "(invoke-kafka \"delete-connector\" \" {connector_name}\"  )" :type "Kafka API"}
        {:api "conn-delete-all-connector" :doc "Describe details of stream " :usage "(invoke-kafka \"delete-all-connector\"  )" :type "Kafka API"}
        {:api "conn-pause-connector" :doc "Pause connector " :usage "(invoke-kafka \"pause-connector\" \"{connector_name}\" )" :type "Kafka API"}
        {:api "conn-resume-connector" :doc "Resume connector " :usage "(invoke-kafka \"resume-connector\" \"{connector_name}\" )" :type "Kafka API"}
        {:api "df-create-connector" :doc "Create source or sink connector in kafka   " :usage "(invoke-kafka \"create-connector\" \"{connector_config}\" )" :type "Kafka API"}
        {:api "conn-describe-connector" :doc "Details of connector   " :usage "(invoke-kafka \"describe-connector\" \"{connector_name}\" )" :type "Kafka API"}

        {:api "ksql-show-streams" :doc "Display list of streams " :usage "(invoke-kafka \"show-streams\" )" :type "Kafka API"}
        {:api "ksql-show-streams-name" :doc "Display list of stream name " :usage "(invoke-kafka \"show-streams-name\" )" :type "Kafka API"}
        {:api "ksql-describe-stream" :doc "Describe details of stream " :about "TODO" :usage "(invoke-kafka \"describe-stream\" \"{stream_name}\" )" :type "Kafka API"}
        {:api "ksql-drop-stream" :doc "remove stream " :usage "(invoke-kafka \"drop-stream\" )" :type "Kafka API"}


        {:api "ksql-show-tables" :doc "Show tables list  " :usage "(invoke-kafka \"show-tables\"  )" :type "Kafka API"}
        {:api "show-table-name" :doc "List of kafka table " :about "TODO" :usage "(invoke-kafka \"show-table-name\")" :type "Kafka API"}
        {:api "ksql-describe-table" :doc "Details of table  " :about "(invoke-kafka \"describe-table\" \"{table_name}\" )" :type "Kafka API"}

        {:api "ksql-show-topics" :doc "Display list of topics " :usage "(invoke-kafka \"show-topics\" )" :type "Kafka API"}
        {:api "describe-topic" :doc "Describe details of topic " :usage "(invoke-kafka \"describe-topic\"  )" :type "Kafka API"}

        {:api "ksql-show-functions" :doc "Display list of functions in ksql " :usage "(invoke-kafka \"show-functions\" )" :type "Kafka API"}
        {:api "description-function" :doc "Display function documentation from kafka " :usage "(invoke-kafka \"description-function\" )" :type "Kafka API"}


        {:api "schema-register" :doc "Describe details of topic " :usage "(invoke-kafka \"register-schema\" \"{schema_details}\"  )" :type "Kafka API"}
        {:api "schema-delete" :doc "Delete schema for topics " :usage "(invoke-kafka \"delete-schema\" )" :type "Kafka API"}
        {:api "schema-show" :doc "List of topic schema   " :usage "(invoke-kafka \"show-schemas\" )" :type "Kafka API"}
        {:api "schema-describe" :doc "Show connector status " :usage "(invoke-kafka \"describe-schema\" \"{schema_name}\"  )" :type "Kafka API"}




        {:api "ksql" :doc "KSQL api adaptor, push ksql to ksqldb " :usage "(ksql \" ksql_statement \" )" :type "Kafka API"}
        {:api "query-by-total" :doc "Run query on ksqldb with time " :usage "(query \" ksql_query|stream \", total_row )" :type "Kafka API"}

        {:api "print-stream" :doc "Print stream from ksqldb " :usage "(print-stream \" stream name \" )" :type "Kafka API"}

        {:api "print-topic" :doc "print kafka topic in terminal, async operation " :usage "(print-topic \" topic name \" )" :type "data API"}
        {:api "print-query" :doc "Run query on ksqldb and print value, async operation " :usage "(print-query \" ksql_query|stream_name \" )
                                                                                         or (print-query \" ksql_query or stream_name \" earliest )" :type "data API"}


        {:api "terminate-query" :doc "Terminal query " :usage "(terminate-query \" query id \" )" :type "Kafka API"}

        {:api "get-ksqldb-schema" :doc "Get full schema from kafka  " :usage "(get-ksqldb-schema)" :type "Ingestion API"}


        {:api "get-schema" :doc "Return full schema from compiler  " :usage "(get-schema)" :type "Ingestion API"}

        ;; Generator API

        {:api "gen-ksql-by-mapping" :doc "Gen ksql statement based on mapping " :usage "(gen-ksql \" mapping or mapping file \" )" :type "Ingestion Generation API"}
        {:api "gen-ksql-by-file" :doc "Gen ksql statement based on mapping " :usage "(gen-ksql \" mapping or mapping file \" )" :type "Ingestion Generation API"}
        {:api "gen-schema" :doc "Gen compiler schema " :usage "(gen-schema \" mapping \" )" :type "Ingestion Generation API"}
        {:api "df-gen-source-connector-config" :doc "Gen source connector based on mapping " :usage "(gen-source-connector \" source connector mapping \" )" :type "Ingestion Generation API"}
        {:api "df-gen-sink-connector-config" :doc "Gen sink connector based on mapping " :usage "(gen-sink-connector \" sink connector mapping \" )" :type "Ingestion Generation API"}


        {:api "df-create-dataflow" :doc "Create ksql statement based on mapping, push to kafka " :usage "(invoke-dataflow create-dataflow \" {mapping } \")" :type "Ingestion API"}
        {:api "df-show-dataflow" :doc "show dataflow from kafka " :usage "(invoke-dataflow show-dataflow \" {mapping } \")" :type "Ingestion API"}
        {:api "df-create-source-connector" :doc "create source connector based on mapping " :usage "(invoke-dataflow \"create-source-connector\" \"{source_connector_mapping or file }\")" :type "Ingestion API"}
        {:api "pause-source-connector" :doc "pause source connector based on mapping " :usage "(invoke-dataflow \"pause-source-connector\" \"{source_connector_mapping or file }\")" :type "Ingestion API"}
        {:api "resume-source-connector" :doc "resume source connector based on mapping " :usage "(invoke-dataflow \"pause-source-connector\" \"{source_connector_mapping or file }\")" :type "Ingestion API"}
        {:api "df-create-sink-connector" :doc "create sink connector based on mapping " :usage "(invoke-dataflow \"create-source-connector\" \"{sink_connector_mapping or file }\")" :type "Ingestion API"}

        {:api "export-ksql-to-file" :doc "create ksql based on mapping and save as file " :usage "(export-ksql-to-file \"Export file name \" \"mapping file \"  )" :type "Pipeline Generator API"}
        {:api "import-ksql-from-file" :doc "Push ksql file to ksql server  " :usage "(import-ksql-from-file \"KSQL file name \" )" :type "Pipeline Generator API"}


        {:api "push-edn-event" :doc "create push edn event " :usage "(push-edn-event topic_name end_data_coll)" :type "data import api"}
        {:api "push-json-event" :doc "create push json event " :usage "(push-json-event topic_name json_data_coll)" :type "data import api"}
        {:api "push-file" :doc "push file, it will create model on fly " :usage "(push-file topic_name file_path)" :type "data import api"}
        {:api "push-meta-model" :doc "import meta model from directory  " :usage "(push-meta-model dir_name)" :type "data import api"}
        {:api "push-directory" :doc "push directory, internally it use push-file " :usage "(push-directory dir_name)" :type "data import api"}

        {:api "gen-dv-connector-mapping" :doc "generate connector config mapping for data vault" :usage "(gen-dv-connector-mapping \"Kakfa_connector_template\")" :type "Ingestion Generation API"}
        {:api "df-clean-system" :doc "clean kafka system, it include schema, topic, stream, table, source connector, sink connector " :usage "(clean-system)" :type "Kafka API"}

        ]


       (clojure.pprint/print-table)
       )

  )


