(ns ingestion.api.client.ksqldb-api
  (:require [clojure.tools.logging :as log]
            [clojure.java.io :as io]
            [ksqldb.client :as client]
            [ksql.gen :as gen]
            [ingestion.api.client.ns-repo-impl :as core]
            [ksql.gen.core-error-msg :as emsg]
            [clojure.data.csv :as csv]
            [ksqldb.client.broker-api :as bapi]
            [cheshire.core :as json]
            [ksql.gen.reader.file-reader :as r]
            [ksql.gen.protocol :as p]
            [clojure.core.async :as async]
            [ksqldb.client.broker-sedes-config :as bsc]
            [ingestion.api.client.dataflow-mapping-gen-api :as mg])
  (:import (java.util.zip ZipInputStream ZipFile ZipEntry)))


(defn ksql
  "Execute ksql statement to KSQL server. KSQL server URL must need to be configured in context  "
  [ksql-str]
  (client/ksql (core/get-context) ksql-str))



(defn- invoke-kafka
  "Invoke Kafka API, if there is no invoke API it will throw exception and return list of available API. "
  ([op-name] (client/invoke op-name))
  ([op-name request]
   (client/invoke op-name request)))


#_(defn get-ksqldb-schema
  "Return current schema from Kafka schema registry "
  []
  (md/get-ksqldb-schema (core/get-context)))



(defn build-dataflow-drop-model
  ([request-name-list] (build-dataflow-drop-model request-name-list []))
  ([request-name-list request-query-list]
   (let [request-name-set (into #{} (map clojure.string/upper-case) request-name-list)
         server-name-list (into (invoke-kafka "ksql-show-streams")
                                (invoke-kafka "ksql-show-tables"))
         drop-sname-list (into [] (filter (fn [m] (contains? request-name-set (get m :name)))) server-name-list)

         schema-list (into #{} (invoke-kafka "schema-show"))
         s-topic-list (mapv :topic drop-sname-list)
         schema-list (into [] (comp
                                (map (fn [v] (str (clojure.string/lower-case (get v :name)) "-value")))
                                (filter (fn [v] (contains? schema-list v)))
                                ) drop-sname-list)

         drop-statement (into [] (comp (map (fn [m] (str " DROP " (get m :type) " IF EXISTS " (get m :name) "")))
                                       ) drop-sname-list)
         terminate-quary-list (into []
                                    (comp
                                      (map (fn [n] (invoke-kafka (str "ksql-describe-" (clojure.string/lower-case (get n :type)))
                                                                 (get n :name))))
                                      (map (fn [v] (into (get v :readQueries)
                                                         (get v :writeQueries))))
                                      cat
                                      (map :id)) drop-sname-list)

         server-query-id-sets (->> (invoke-kafka "ksql-show-queries")
                                   (into #{} (map :id)))
         requested-query-id-list (into [] (map (fn [q-id]
                                                 (contains? server-query-id-sets q-id)
                                                 )) request-query-list)

         terminate-quary-list (->> (into terminate-quary-list requested-query-id-list)
                                   (distinct))
         terminate-query-statement (into [] (map (fn [v]
                                                   (str "TERMINATE " v)
                                                   )) terminate-quary-list)

         out (into terminate-query-statement drop-statement)]
     {:statement   out
      :topic-list  s-topic-list
      :schema-list schema-list})))


(defn get-drop-model [schema-m-coll]
  (let [drop-name-list (into [] (map :name) schema-m-coll)
        terminate-query-id-list (into []
                                      (comp
                                        (filter (fn [m] (= "mapping" (get m :type))))
                                        (map (fn [m] (get-in m [:with :commandStatus :queryId]))))
                                      schema-m-coll)]
    (build-dataflow-drop-model drop-name-list terminate-query-id-list)))


(defn do-dataflow-clean-up [m]
  (let [s (get m :statement)]
    (when-not (empty? s)
      (-> (clojure.string/join "; " s)
          (str " ; ")
          (ksql)))
    (invoke-kafka "broker-delete-topics" (get m :topic-list []))
    (doseq [w (get m :schema-list)]
      (invoke-kafka "schema-delete" w))))


(defn get-all-stream-list []
  (let [stream-list (->> (invoke-kafka "ksql-show-streams")
                         (mapv :name)
                         (mapv clojure.string/lower-case))
        table-list (->> (invoke-kafka "ksql-show-tables")
                        (mapv :name)
                        (mapv clojure.string/lower-case))
        w (into #{} stream-list)
        w (into w table-list)]
    (disj w "ksql_processing_log")))



(defn drop-all-stream-and-table-with-topic []
  (-> (get-all-stream-list)
      (build-dataflow-drop-model)
      (do-dataflow-clean-up)))



(defn delete-all-connector []
  (invoke-kafka "conn-delete-all-connector"))




(defn push-event-to-topic-impl [topic-name edn-data-coll]
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)]
    (doseq [v edn-data-coll]
      (let [w (json/generate-string v)
            w {:value w}]
        (async/>!! input-chan w)))
    (log/info "total msg send to kafka " (count edn-data-coll))
    (async/>!! input-chan "done!")))








(defmulti push-file-to-topic-impl2 (fn [_ file-path in]
                                     ;   (println "--" file-path)
                                     (let [w (last (clojure.string/split file-path #"\."))]
                                       (clojure.string/lower-case w))))


(defmethod push-file-to-topic-impl2 :default
  [topic-name file-path in]
  (let [w (last (clojure.string/split file-path #"\."))]
    (throw (emsg/ex-info-for-invalid-data-type w)  #_(ex-info "now support only json, xml and csv file " {:provided w
                                                                                                          :expected #{"json" "xml" "csv"}}))))


(defmethod push-file-to-topic-impl2 "json"
  [topic-name json-file-path in]
  (log/info " ---------start push file---------- " json-file-path)
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)]
    (try
      (with-open [rdr (clojure.java.io/reader in)]
        (doseq [line (line-seq rdr)]
          (async/>!! input-chan {:value line})
          (swap! total inc))
        (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (throw e) #_(ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path json-file-path :total @total})))



(defmethod push-file-to-topic-impl2 "csv"
  [topic-name csv-file-path in]
  (log/info " ---------start push file---------- " csv-file-path)
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)
        separator (r/get-separator in)
        header (mg/get-header in) #_(r/read-csv-header (clojure.string/replace csv-file-path "'" ""))
        header (r/format-csv-header header)]

    (try
      (with-open [rdr (clojure.java.io/reader in)   #_(clojure.java.io/reader csv-file-path)]
        (let [v (csv/read-csv  rdr :separator separator)]
          (doseq [line (rest v)]
            (let [w (zipmap header line)
                  w-json (json/generate-string w)]
              (async/>!! input-chan {:value w-json})
              (swap! total inc)))
          )

        (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (throw e)
          #_(ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path csv-file-path :total @total})))



(defmethod push-file-to-topic-impl2 "xml"
  [topic-name xml-file-path in]
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)]

    (try
      (let [w (r/convert-xml-file-to-edn in)
            ;_ (println "--msg")
            ;_ (clojure.pprint/pprint w)
            w1 (json/generate-string w)]

        (async/>!! input-chan {:value w1})
        (swap! total inc)
        (async/>!! input-chan "done!"))

      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          ;(ex-data e)
          (throw e)
          )))
    (log/info "--------- End push file to kafka with " {:file-path xml-file-path :total @total})))



(defn entries [zipfile]
  (enumeration-seq (.entries zipfile)))


(defmethod push-file-to-topic-impl2 "zip"
  [_ file-name file]
  (with-open [z (java.util.zip.ZipFile. file)]
    (reduce (fn [acc ^java.util.zip.ZipEntry v]
              (let [ext (mg/get-file-ext (.getName v))
                    f-name (mg/get-file-name (.getName v))
                    out (condp = ext
                          "xml"
                          (->> (.getInputStream z v)
                               (push-file-to-topic-impl2 f-name (str f-name ".xml"))
                               )
                          "json"
                          (->> (.getInputStream z v)
                               (push-file-to-topic-impl2 f-name (str f-name ".json")))
                          "csv"
                          (->> (.getInputStream z v)
                               (push-file-to-topic-impl2 f-name (str f-name ".csv")))
                          (throw (emsg/ex-info-for-invalid-data-type (mg/get-file-ext (.getName v)))))]
                (str acc "\n" out))
              ) "" (entries z))))

;;; Push file to data
(defmethod p/invoke "df-data-push-file"
  [ {:keys [request]}]
  (let [file-name (get request :file-name)

        in    (get request :file )
        in    (if (string? in )
                (io/file in)
                in
                )
        mapping  (mg/gen-mapping-by-file file-name in)

        topic-name (mg/get-file-name file-name)
        ]
    ;(p/log-v mapping)
    (p/invoke  {:op "df-create-dataflow" :request mapping})
    (push-file-to-topic-impl2 topic-name file-name in  )
    mapping
    ))












#_(defmulti push-file-to-topic-impl (fn [_ file-path]
                                      ;   (println "--" file-path)
                                      (let [w (last (clojure.string/split file-path #"\."))]
                                        (clojure.string/lower-case w))))


#_(defmethod push-file-to-topic-impl :default
    [topic-name file-path]
    (let [w (last (clojure.string/split file-path #"\."))]
      (throw (emsg/ex-info-for-invalid-data-type w)  #_(ex-info "now support only json, xml and csv file " {:provided w
                                                                                                            :expected #{"json" "xml" "csv"}}))))


#_(defmethod push-file-to-topic-impl "json"
    [topic-name json-file-path]
    (log/info " ---------start push file---------- " json-file-path)
    (let [context (core/get-context)
          topic-config (bsc/json-topic-config context topic-name)
          input-chan (bapi/publish-async context topic-config)
          total (atom 0)]
      (try
        (with-open [rdr (clojure.java.io/reader json-file-path)]
          (doseq [line (line-seq rdr)]
            (async/>!! input-chan {:value line})
            (swap! total inc))
          (async/>!! input-chan "done!"))
        (catch Exception e
          (do
            (async/>!! input-chan "done!")
            (ex-data e))))
      (log/info "--------- End push file to kafka with " {:file-path json-file-path :total @total})))



#_(defmethod push-file-to-topic-impl "csv"
    [topic-name csv-file-path]
    (log/info " ---------start push file---------- " csv-file-path)
    (let [context (core/get-context)
          topic-config (bsc/json-topic-config context topic-name)
          input-chan (bapi/publish-async context topic-config)
          total (atom 0)
          separator (r/get-separator csv-file-path)
          header (r/read-csv-header (clojure.string/replace csv-file-path "'" ""))
          header (r/format-csv-header header)]

      (try
        (with-open [rdr (csv/read-csv (clojure.java.io/reader csv-file-path) :separator separator)  #_(clojure.java.io/reader csv-file-path)]
          (doseq [line (rest rdr)]
            (let [w (zipmap header line)
                  w-json (json/generate-string w)]
              (async/>!! input-chan {:value w-json})
              (swap! total inc)))
          (async/>!! input-chan "done!"))
        (catch Exception e
          (do
            (async/>!! input-chan "done!")
            (ex-data e))))
      (log/info "--------- End push file to kafka with " {:file-path csv-file-path :total @total})))



#_(defmethod push-file-to-topic-impl "xml"
    [topic-name xml-file-path]
    (let [context (core/get-context)
          topic-config (bsc/json-topic-config context topic-name)
          input-chan (bapi/publish-async context topic-config)
          total (atom 0)]

      (try
        (let [w (r/convert-xml-file-to-edn xml-file-path)
              w1 (json/generate-string w)]

          (async/>!! input-chan {:value w1})
          (swap! total inc)
          (async/>!! input-chan "done!"))

        (catch Exception e
          (do
            (async/>!! input-chan "done!")
            (ex-data e))))
      (log/info "--------- End push file to kafka with " {:file-path xml-file-path :total @total})))

