(ns ingestion.api.client.dataflow-api
  (:require [clojure.tools.logging :as log]
            [ksql.gen :as gen]
            [ingestion.api.client.dataflow-connector-api]
            [ksqldb.client :as client]
            [cheshire.core :as json]
            [ksql.gen.protocol :as p]
            [ingestion.api.client.ns-repo-impl :as core]
            [ingestion.api.client.ksqldb-api :as kapi]))


(defn remote-handler [md-schema]
  (let [context {:ksql-url (get p/context :ksql-url)
                 :offset   (get-in md-schema [:with :offset])}
        statement (get-in md-schema [:with :statement])
        statement (first statement)]
    (-> (client/ksql context statement)
        (get :commandStatus))))


(defn try-remote-handler [handler]
  (fn [md-schema]
    (try
      [(handler md-schema) nil]
      (catch Exception e
        (let [v (ex-data e)
              v (if-let [w (get v :body)]
                  (json/parse-string w)
                  v)]
          [nil v])))

    )
  )


(defn warp-delta-middleware [handler]
  (let [stream-name-coll (->> (client/invoke "ksql-show-streams")
                              (into #{} (comp (map :name))))
        table-name-coll (->> (client/invoke "ksql-show-tables-name")
                             (into #{}))
        stream-name-coll (into stream-name-coll table-name-coll)
        kafka-insert-query (->> (client/invoke "ksql-show-queries")
                                (into #{} (comp (map :queryString)
                                                (remove nil?)
                                                (filter (fn [v]
                                                          (clojure.string/starts-with? v "INSERT")))
                                                (map (fn [v] (clojure.string/upper-case v))))))
        is-skip? (fn [md-schema]
                   (let [w (get md-schema :name)
                         sql-statement (first (get-in md-schema [:with :statement] [""]))]
                     (if (and (contains? stream-name-coll (clojure.string/upper-case w))
                              (not= "mapping" (get md-schema :type)))
                       true
                       (contains? kafka-insert-query (clojure.string/upper-case (str sql-statement ";"))))))]
    (fn [md-schema]
      (if (is-skip? md-schema)
        md-schema
        (handler md-schema)))))



(defn warp-batch-middleware [handler]
  (fn [md-schema-coll]
    (loop [[md-schema & md-schema-coll] md-schema-coll
           acc []]
      (if (nil? md-schema)
        [acc nil]
        (let [[out error] (handler md-schema)]
          (if error
            [acc error]
            (let [md-schema (update-in md-schema [:with] assoc :commandStatus out)
                  acc (conj acc md-schema)]
              (recur md-schema-coll acc))))))))



(defn remote-handler-batch [md-schema-coll]
  ((-> remote-handler
       (warp-delta-middleware)
       (try-remote-handler)
       (warp-batch-middleware))
   md-schema-coll))


(defn error? [v]
  (if (and (map? v)
           (get v :error))
    true false))



;; Keep remote separate, good for integration test
(defn ksql-batch [context ksql-coll]
  (->> ksql-coll
       ;(get-new-md-repo context)
       (mapv (partial client/ksql context))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defn map-ksql [schema-coll]
  (->> schema-coll
       (into [] (comp (map (fn [m]
                             (get-in m [:with :statement]))) cat))))


(defmethod p/invoke "df-do-compile"
  [{:keys [request]}]
  (let [new-md-schema-coll (gen/do-compile request)]
    (p/update-md-repo new-md-schema-coll)
    (p/reset-mapping! request)
    new-md-schema-coll))


(defmethod p/invoke "df-gen-ksql"
  [{:keys [request]}]
  (let [new-md-schema-coll (gen/do-compile request)]
    (p/update-md-repo new-md-schema-coll)
    (p/reset-mapping! request)
    (-> (p/get-md-schema-coll)
        (map-ksql )
        )


    ))



(defmethod p/invoke "df-create-dataflow"
  [{:keys [request]}]
  (let [new-md-schema-coll (gen/do-compile request)
        [out error] (remote-handler-batch new-md-schema-coll)]
    (p/update-md-repo out)
    (p/reset-mapping! request)
    (if error
      (throw (ex-info "error " (if (map? error) error {:error error})))
      {:status "ok"})))


(defmethod p/invoke "df-show-dataflow"
  [{:keys [request]}]
  (let [schema-m-coll (p/get-md-schema-coll)
        new-schema-m-coll (gen/do-compile schema-m-coll request)]
    (->> new-schema-m-coll
         (map-ksql))))


(defmethod p/invoke "df-clean-system"
  [{:keys [request]}]
  (kapi/delete-all-connector)
  (kapi/drop-all-stream-and-table-with-topic)
  (p/reset-md-repo!))



(defn- remove-impl [w]
  (let []
    (-> (kapi/get-drop-model w)
        (dissoc :topic-list :schema-list)
        (kapi/do-dataflow-clean-up))
    ;(p/pull-metadata md-repo)
    {:status "ok"})
  #_(try

    (catch Exception e
      (let [msg (-> (Throwable->map e)
                    (select-keys [:cause])
                    ) #_(or (-> (ex-data e)
                        (:body)
                        (json/parse-string))
                    (ex-message e))]

        (log/info "error: " e)
        (throw (ex-info msg {:error msg}))
        #_{:status "error"
         :error  msg}))))


#_(defmethod p/invoke "df-redu-dataflow"
    [{:keys [request]}]
    (->> (p/get-md-schema-coll)
         ;(p/peek-metadata)
         (remove-impl)))



(defmethod p/invoke "df-delete-dataflow"
  [{:keys [request]}]
  (->> (p/get-md-schema-coll)
       ;(p/get-all-metadata)
       (remove-impl)))



#_(defmethod p/invoke "df-update-dataflow"
    [{:keys [request]}]
    (try
      (let [new-md-schema-coll (gen/do-compile [] request)

            _ (->> (p/get-md-schema-coll)
                   (remove-impl p/context))

            new-md-schema-coll (invoke-remote p/context new-md-schema-coll)
            out (into [] (map (fn [m]
                                (str (get m :name) " " (get-in m [:with :commandStatus :message]))
                                )) new-md-schema-coll)]

        (if (error? new-md-schema-coll)
          (p/update-md-repo (get new-md-schema-coll :v))
          (p/update-md-repo new-md-schema-coll))


        (if (error? new-md-schema-coll)
          (-> (dissoc new-md-schema-coll :v)
              (assoc :status "error" :out out))
          {:status "ok"
           :out    out}))
      (catch Exception e
        {:status "error"
         :error  (ex-data e)})))


(defmethod p/invoke "df-delete-dataflow-with-data"
  [{:keys [request]}]
  (let [w (p/get-md-schema-coll)]
    (-> (kapi/get-drop-model w)
        (kapi/do-dataflow-clean-up))
    ;(p/pull-metadata w)
    {:status "ok"})
  #_(try

    (catch Exception e
      {:status "error"
       :error  (-> (ex-data e)
                   (:body)
                   (json/parse-string))})))



#_(defmethod p/invoke "df-update-dataflow"
    [op-name context mapping]
    (do (p/invoke "df-delete-dataflow" context mapping)
        (p/invoke "df-create-dataflow" context mapping)))


#_(defmethod p/invoke "req-handler"
    [op-name context mapping]
    #_(let [

            schema-m-coll (get-mapping-schema-repo context)
            new-schema-m-coll (gen/do-process context schema-m-coll mapping)
            delta-schema-m-coll (->> (split-at (count schema-m-coll) new-schema-m-coll)
                                     (second)
                                     (into []))
            ksql-coll (map-ksql delta-schema-m-coll)]
        (push-ksql-statement context ksql-coll)
        (update-mapping-schema-repo context delta-schema-m-coll)
        ksql-coll))



