(ns ingestion.api.client
  (:require [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [ingestion.api.client.ns-repo-impl :as ksql]
            [ksql.gen.core-error-msg :as emsg]
            [ksqldb.client :as client]
            [ksql.gen.reader.file-reader :as r]
            [ksql.gen.protocol :as p]
            [ksql.gen :as gen]
            [ksql.gen-connector :as conn]
            [ingestion.api.client.ksqldb-api :as kapi]
            [ingestion.api.client.dataflow-api :as iapi]
            [ingestion.api.client.dataflow-mapping-api]
            [ingestion.api.client.dataflow-mapping-gen-api :as gapi]
            [ingestion.api.client.ns-repo-impl :as core]
            [clojure.java.io :as io]
    #_[ingestion.api.client.gen-util :as gapi]))


;;;;;;;;;;;;;;;;;;, KSQLDB API adaptor ;;;;;;;;;;;;;;;;;

(defn ksql
  "Execute ksql statement to KSQL server. KSQL server URL must need to be configured in context  "
  [ksql-str]
  (client/ksql (core/get-context) ksql-str))



(defn invoke [req]
  (let [{:keys [app-ns-name]} req
        c (p/get-ns-by-name (core/get-ns-repo)  app-ns-name) ]
    ;(log/debug "invoke for " app-ns-name)
    ;(println "---mapping name " ans-name)
    (when (and (nil? c)
               (not (contains? #{"df-ns-names" "df-ns-new"} (get req :op))))
      (throw (ex-info "mapping ns not found " {:emsg        "mapping ns not found, set mapping"
                                               :e-ref       app-ns-name
                                               :e-available (p/get-ns-names (core/get-ns-repo)) })))
    (binding [p/context c]
      (let [out (p/invoke req)]
        out))))



(defn with-op
  ([op-name] (with-op {} op-name))
  ([req op-name] (assoc req :op op-name)))


(defn with-req
  ([req] (with-req {} req))
  ([req req-param] (assoc req :request req-param)))


(defn with-ns
  ([m-name] (with-ns {} m-name))
  ([req m-name] (assoc req :app-ns-name m-name)))


(defn invoke-old
  "Invoke dataflow API, if there is no api then it will throw exception with list of available API"
  ([op-name] (invoke-old op-name nil))
  ([op-name request]
   (p/invoke {:op op-name :request request})))


(defn terminate-query
  "Terminate any running query "
  [query-id]
  (client/terminate-push-query (client/get-client (core/get-context)) query-id))


(defn- as-query
  ([v total]
   (if (< 1 (count (clojure.string/split v #" ")))
     v
     (str "select * from " v " emit changes" " limit " total " ;")
     ))
  ([v]
   (if (< 1 (count (clojure.string/split v #" ")))
     v
     (str "select * from " v " emit changes;"))))


(defn query-by-duration
  "async execute query by duration, read data from latest offset, default call print, default timeout 3000  "
  ([^String query-str] (query-by-duration query-str println "latest" 3000))
  ([^String query-str callback] (query-by-duration query-str callback "latest" 3000))
  ([^String query-str callback offset] (query-by-duration query-str callback offset 3000))
  ([^String query-str callback offset duration]
   (client/stream-query-async-with-duration (core/get-context) (as-query query-str) callback offset duration)))


(defn query-by-total
  ([^String query-str total]
   (query-by-total query-str total "earliest"))
  ([^String query-str total offset]
   (client/stream-query (client/get-client (core/get-context)) (as-query query-str total) total offset)))



(defn print-query
  ([^String query-str] (print-query query-str "latest" 3000))
  ([^String query-str offset] (print-query query-str offset 3000))
  ([^String query-str offset duration]
   (client/stream-query-async-with-duration (core/get-context) (as-query query-str) println offset duration)))



(defn print-topic
  ([v] (print-topic v "latest" 3000))
  ([v offset] (print-topic v offset 3000))
  ([v offset duration]
   (client/print-topic-by-time (core/get-context) v offset duration println)))


(defn push-data-to-stream
  [strem-name rows]
  (client/push-data-batch (ksql/get-context) strem-name rows))

;;;;;;;;;;;;;;;;;;;;;ingestion api

(defn push-edn-event [topic-name edn-data-coll]
  (let [w (gapi/gen-napping topic-name edn-data-coll)]

    (p/invoke {:op "df-create-dataflow" :request w})
    (kapi/push-event-to-topic-impl topic-name edn-data-coll)))


(defn push-json-event [topic-name json-str]
  (push-edn-event topic-name (json/parse-string json-str true)))


(defn push-file [topic-name file-path]

  (p/invoke {:op      "df-data-push-file"
             :request file-path
             })

  #_(let [context (core/get-context)
          mapping (gapi/gen-mapping-by-file-name topic-name file-path)]
      ;  (clojure.pprint/pprint mapping)
      (p/invoke {:op "df-create-dataflow" :request mapping})
      ;(p/invoke-impl "df-create-dataflow" context mapping)
      (kapi/push-file-to-topic-impl2 topic-name file-path (io/file file-path))))







(defn push-meta-model [directory-name]
  (let [
        v ["_" "_" (str "(from_csv_folder \"" directory-name "\" )")
           "_" "_" "(value_format 'json' )" ";"
           "_" "_" (str "(from_json_folder \"" directory-name "\" )" ";")
           "_" "_" (str "(from_xml_folder \"" directory-name "\" )" ";")]]
    ;(p/invoke-impl "df-create-dataflow" context v)
    (p/invoke {:op "df-create-dataflow" :request v})
    ))


#_(defn push-directory [directory-name]
    (let [csv-file-list (r/get-all-file-name directory-name "glob:*.{csv}")
          json-file-list (r/get-all-file-name directory-name "glob:*.{json}")
          xml-file-list (r/get-all-file-name directory-name "glob:*.{xml}")]
      ;  (import-meta-model context directory-name)
      (doseq [[file-name file-path] csv-file-list]
        (push-file file-name file-path))
      (doseq [[file-name file-path] json-file-list]
        (push-file file-name file-path))
      (doseq [[file-name file-path] xml-file-list]
        (push-file file-name file-path))))

;;;;;;;;;;;;;;;;;;;;;;;; clean api ;;;;;;;;;;;;;;;;;;;;;;;;

(defn clean-system []
  (kapi/delete-all-connector)
  (kapi/drop-all-stream-and-table-with-topic)
  (p/reset-md-repo!))


;;;;;;;;;;;;;;;;;;;;;;;;;; gen api  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn do-compile [mapping]
  (gen/do-compile mapping))


(defn gen-ksql-by-mapping [mapping]
  (gen/gen-ksql-by-mapping mapping))


#_(defn gen-ksql-by-file [topic-name file-path]
    (gen/gen-ksql-by-mapping topic-name file-path))


(defn gen-ksql-by-data [type topic-name v]
  (let [v (condp = type
            "edn" (pr-str v)
            "json" (pr-str (json/parse-string v true))
            "xml" (pr-str (r/convert-xml-to-edn v))
            (throw (emsg/ex-info-for-invalid-data-type type)))
        v (clojure.string/replace v "'" "")
        mapping [topic-name "_" (str "(from_edn_data  " (pr-str v) "  )")
                 topic-name "_" "(value_format 'json' )" ";"]]
    (gen-ksql-by-mapping mapping)))


(defn gen-schema-by-date [type topic-name v]
  (let [v (condp = type
            "edn" (pr-str v)
            "json" (pr-str (json/parse-string (clojure.string/replace v "'" "") true))
            "xml" (pr-str (r/convert-xml-to-edn v))
            (throw (emsg/ex-info-for-invalid-data-type type)))
        mapping [topic-name "_" (str "(from_edn_data  " (pr-str v) "  )")
                 topic-name "_" "(value_format 'json' )" ";"]]
    (into [] (comp (filter (fn [v]
                             (contains? #{"stream" "table"} (get v :type))
                             ))) (do-compile mapping))))



;;;;;;;;;;;;;;;;;;;;;; Export import ksql

(defn export-ksql-to-file
  "Export generated KSQL statement to file
  file-name: Saving file name
  mapping: Mapping file or mapping as data
  "
  [file-name mapping]
  (let [o (gen-ksql-by-mapping mapping)
        o (clojure.string/join "\n" o)]
    (spit file-name o :append true)))


(defn import-ksql-from-file
  "Import KSQl to KSQl server after apply delta "
  [file-name]
  (->> (clojure.string/split (slurp file-name) #"\n")
       (remove (fn [v] (clojure.string/blank? v)))
       (vec)
       (run! ksql)
       #_(ksql-batch (core/get-context))))


