(ns azure.client.blob-storage
  (:import [com.azure.storage.blob BlobServiceClient BlobServiceClientBuilder BlobContainerClient BlobClient]
           [com.azure.storage.blob.models BlobStorageException BlobErrorCode]
           [com.azure.storage.common StorageSharedKeyCredential]
           [java.util Locale]
           [java.io File ByteArrayInputStream ByteArrayOutputStream]))


(defn ^StorageSharedKeyCredential storage-shared-key-credential [account-name account-key]
  (StorageSharedKeyCredential. account-name account-key))


(defn end-point [account-name]
  (format "https://%s.blob.core.windows.net" account-name))


;"testingazukqjys" "bmXte5EoS7KxXSBDgBYUAcCFZtLxpvK9JuaZx3cDdj2xEulejOEwlop0qgAjUYMnFq9yQ1mVwrENKpFfkv0M/A=="
(defn ^BlobServiceClient blob-service-client [storage-account-name storage-account-key]
  (let [^StorageSharedKeyCredential credential (storage-shared-key-credential storage-account-name storage-account-key)
        end-point-str (end-point storage-account-name)
        ^BlobServiceClientBuilder t (doto (BlobServiceClientBuilder.)
                                      (.endpoint end-point-str)
                                      (.credential credential))]
    (.buildClient t)))


(defn ^BlobContainerClient blob-container-client [^BlobServiceClient service-client container-name]
  (let [^BlobContainerClient container (-> (.getBlobContainerClient service-client container-name)
                                           #_(.getContainerReference container-name))]
    (try
      ; (.createIfNotExist container)
      (.create container)
      (catch BlobStorageException ex
        (when (not= (.getErrorCode ex) BlobErrorCode/CONTAINER_ALREADY_EXISTS)
          (throw ex))))
    container))


(defn blob-list-impl [^BlobContainerClient blob-container-client]
  (->> (.listBlobs blob-container-client)
       (map #(.getName %))))


(defn ^BlobClient blob-client [^BlobContainerClient blob-container-client file-name]
  (.getBlobClient blob-container-client file-name))


(defn upload-file-impl
  [^BlobClient blob-client ^File file-path]
  (.uploadFromFile blob-client (.getPath file-path)))


(defn upload-impl
  ([^BlobClient blob-client blob-v]
   (upload-impl blob-client blob-v true))
  ([^BlobClient blob-client blob-v override]
   (let [w (.getBytes blob-v "UTF-8")
         ^ByteArrayInputStream input-stream (ByteArrayInputStream. w)]
     (.upload blob-client input-stream (alength w) override))))


(defn download-impl
  [^BlobClient blob-client]
  (let [^ByteArrayOutputStream out-stream (ByteArrayOutputStream.)
        _ (.download blob-client out-stream)
        out (slurp (.toByteArray out-stream))]
    out))


(defn blob-list [^BlobServiceClient service-client container-name]
  (-> service-client
      (blob-container-client container-name)
      (blob-list-impl)))


(defn download [^BlobServiceClient service-client container-name blob-name]
  (-> service-client
      (blob-container-client container-name)
      (blob-client blob-name)
      (download-impl)))


(defn upload [^BlobServiceClient service-client container-name blob-name blob-v]
  (-> service-client
      (blob-container-client container-name)
      (blob-client blob-name)
      (upload-impl blob-v)))


;(defn download [])



