(ns test-utils.core
  (:use [org.httpkit.fake])
  (:require [ring.mock.request :as mock]
            [clojure.java.io :as io]
            [auth-utils.core :as auth-utils]
            [constants.core :as constants]
            [cacheing.core :as cache]))

(def bad-saml "PGp1bmtYbWw+PC9qdW5rWG1sPg==")
(def unsigned-saml "PD94bWwgdmVyc2lvbj0iMS4wIj8+DQo8c2FtbHA6UmVzcG9uc2UgeG1sbnM6c2FtbHA9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDpwcm90b2NvbCIgVmVyc2lvbj0iMi4wIiBJRD0icGZ4MzA0NTkyYmQtNmZjNC0xNmZmLTlmM2YtMTIwN2M3YjU0YzMxIiBJc3N1ZUluc3RhbnQ9IjIwMTYtMDgtMThUMjA6NTU6NDEuMTQ5WiIgRGVzdGluYXRpb249Imh0dHBzOi8vY3NhYS1pbnN1cmFuY2UuYWFhLmNvbS9zZWxmc2VydmljZS9kYXNoYm9hcmQiPjxzYW1sOklzc3VlciB4bWxuczpzYW1sPSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXNzZXJ0aW9uIj5CMkM6V01TUG9ydGFsOlNBTUw6SWRQPC9zYW1sOklzc3Vlcj4NCjxzYW1scDpTdGF0dXM+PHNhbWxwOlN0YXR1c0NvZGUgVmFsdWU9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDpzdGF0dXM6U3VjY2VzcyIvPjwvc2FtbHA6U3RhdHVzPjxzYW1sOkFzc2VydGlvbiB4bWxuczpzYW1sPSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXNzZXJ0aW9uIiBJRD0iaW9wQjJyYllQSDVEeUo5WGx6WXpUNkRuekdxIiBJc3N1ZUluc3RhbnQ9IjIwMTYtMDgtMThUMjA6NTU6NDEuMTQ5WiIgVmVyc2lvbj0iMi4wIj48c2FtbDpJc3N1ZXI+QjJDOldNU1BvcnRhbDpTQU1MOklkUDwvc2FtbDpJc3N1ZXI+PHNhbWw6U3ViamVjdD48c2FtbDpOYW1lSUQgRm9ybWF0PSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoxLjE6bmFtZWlkLWZvcm1hdDpYNTA5U3ViamVjdE5hbWUiPkFBQTAwMjEzMzk3MTwvc2FtbDpOYW1lSUQ+PHNhbWw6U3ViamVjdENvbmZpcm1hdGlvbiBNZXRob2Q9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDpjbTpiZWFyZXIiPjxzYW1sOlN1YmplY3RDb25maXJtYXRpb25EYXRhIFJlY2lwaWVudD0iaHR0cHM6Ly9jc2FhLWluc3VyYW5jZS5hYWEuY29tL3NlbGZzZXJ2aWNlL2Rhc2hib2FyZCIgTm90T25PckFmdGVyPSIyMDIwLTA4LTE4VDIxOjAwOjQxLjE0OVoiLz48L3NhbWw6U3ViamVjdENvbmZpcm1hdGlvbj48L3NhbWw6U3ViamVjdD48c2FtbDpDb25kaXRpb25zIE5vdEJlZm9yZT0iMjAxNi0wOC0xOFQyMDo1MDo0MS4xNDlaIiBOb3RPbk9yQWZ0ZXI9IjIwMjAtMDgtMThUMjE6MDA6NDEuMTQ5WiI+PHNhbWw6QXVkaWVuY2VSZXN0cmljdGlvbj48c2FtbDpBdWRpZW5jZT5CMkM6U0FNTDpTUDwvc2FtbDpBdWRpZW5jZT48L3NhbWw6QXVkaWVuY2VSZXN0cmljdGlvbj48L3NhbWw6Q29uZGl0aW9ucz48c2FtbDpBdXRoblN0YXRlbWVudCBTZXNzaW9uSW5kZXg9ImlvcEIycmJZUEg1RHlKOVhsell6VDZEbnpHcSIgQXV0aG5JbnN0YW50PSIyMDE2LTA4LTE4VDIwOjU1OjQxLjE0OVoiPjxzYW1sOkF1dGhuQ29udGV4dD48c2FtbDpBdXRobkNvbnRleHRDbGFzc1JlZj51cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YWM6Y2xhc3Nlczp1bnNwZWNpZmllZDwvc2FtbDpBdXRobkNvbnRleHRDbGFzc1JlZj48L3NhbWw6QXV0aG5Db250ZXh0Pjwvc2FtbDpBdXRoblN0YXRlbWVudD48c2FtbDpBdHRyaWJ1dGVTdGF0ZW1lbnQ+PHNhbWw6QXR0cmlidXRlIE5hbWU9IlppcCIgTmFtZUZvcm1hdD0idXJuOm9hc2lzOm5hbWVzOnRjOlNBTUw6Mi4wOmF0dHJuYW1lLWZvcm1hdDpiYXNpYyI+PHNhbWw6QXR0cmlidXRlVmFsdWUgeG1sbnM6eHM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hIiB4bWxuczp4c2k9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hLWluc3RhbmNlIiB4c2k6dHlwZT0ieHM6c3RyaW5nIj44NTM2MTwvc2FtbDpBdHRyaWJ1dGVWYWx1ZT48L3NhbWw6QXR0cmlidXRlPjxzYW1sOkF0dHJpYnV0ZSBOYW1lPSJjdXN0b21lclJlZ2lzdHJhdGlvbklEIiBOYW1lRm9ybWF0PSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXR0cm5hbWUtZm9ybWF0OmJhc2ljIj48c2FtbDpBdHRyaWJ1dGVWYWx1ZSB4bWxuczp4cz0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEiIHhtbG5zOnhzaT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEtaW5zdGFuY2UiIHhzaTp0eXBlPSJ4czpzdHJpbmciPjdFMEMwNjM1LTJGOEMtNDc4Qy05QTU3LTNDOENGMTkwM0Q5RTwvc2FtbDpBdHRyaWJ1dGVWYWx1ZT48L3NhbWw6QXR0cmlidXRlPjxzYW1sOkF0dHJpYnV0ZSBOYW1lPSJyb2xlIiBOYW1lRm9ybWF0PSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXR0cm5hbWUtZm9ybWF0OmJhc2ljIj48c2FtbDpBdHRyaWJ1dGVWYWx1ZSB4bWxuczp4cz0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEiIHhtbG5zOnhzaT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEtaW5zdGFuY2UiIHhzaTp0eXBlPSJ4czpzdHJpbmciPkNVU1RPTUVSPC9zYW1sOkF0dHJpYnV0ZVZhbHVlPjwvc2FtbDpBdHRyaWJ1dGU+PHNhbWw6QXR0cmlidXRlIE5hbWU9IkFkZHJlc3MiIE5hbWVGb3JtYXQ9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDphdHRybmFtZS1mb3JtYXQ6YmFzaWMiPjxzYW1sOkF0dHJpYnV0ZVZhbHVlIHhtbG5zOnhzPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYSIgeG1sbnM6eHNpPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYS1pbnN0YW5jZSIgeHNpOnR5cGU9InhzOnN0cmluZyIvPjwvc2FtbDpBdHRyaWJ1dGU+PHNhbWw6QXR0cmlidXRlIE5hbWU9Im1haWwiIE5hbWVGb3JtYXQ9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDphdHRybmFtZS1mb3JtYXQ6YmFzaWMiPjxzYW1sOkF0dHJpYnV0ZVZhbHVlIHhtbG5zOnhzPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYSIgeG1sbnM6eHNpPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYS1pbnN0YW5jZSIgeHNpOnR5cGU9InhzOnN0cmluZyI+dHJhdmlzNjIzQGljbG91ZC5jb208L3NhbWw6QXR0cmlidXRlVmFsdWU+PC9zYW1sOkF0dHJpYnV0ZT48c2FtbDpBdHRyaWJ1dGUgTmFtZT0iZ2l2ZW5OYW1lIiBOYW1lRm9ybWF0PSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXR0cm5hbWUtZm9ybWF0OmJhc2ljIj48c2FtbDpBdHRyaWJ1dGVWYWx1ZSB4bWxuczp4cz0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEiIHhtbG5zOnhzaT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEtaW5zdGFuY2UiIHhzaTp0eXBlPSJ4czpzdHJpbmciPlRSQVZJUzwvc2FtbDpBdHRyaWJ1dGVWYWx1ZT48L3NhbWw6QXR0cmlidXRlPjxzYW1sOkF0dHJpYnV0ZSBOYW1lPSJDaXR5IiBOYW1lRm9ybWF0PSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXR0cm5hbWUtZm9ybWF0OmJhc2ljIj48c2FtbDpBdHRyaWJ1dGVWYWx1ZSB4bWxuczp4cz0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEiIHhtbG5zOnhzaT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEtaW5zdGFuY2UiIHhzaTp0eXBlPSJ4czpzdHJpbmciLz48L3NhbWw6QXR0cmlidXRlPjxzYW1sOkF0dHJpYnV0ZSBOYW1lPSJPQXV0aCIgTmFtZUZvcm1hdD0idXJuOm9hc2lzOm5hbWVzOnRjOlNBTUw6Mi4wOmF0dHJuYW1lLWZvcm1hdDpiYXNpYyI+PHNhbWw6QXR0cmlidXRlVmFsdWUgeG1sbnM6eHM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hIiB4bWxuczp4c2k9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hLWluc3RhbmNlIiB4c2k6dHlwZT0ieHM6c3RyaW5nIj5leUpoYkdjaU9pSklVekkxTmlJc0ltdHBaQ0k2SWpFaWZRLmV5SjZhWEFpT2lJNE5UTTJNU0lzSW1OMWMzUnZiV1Z5YTJWNUlqb2lOMFV3UXpBMk16VXRNa1k0UXkwME56aERMVGxCTlRjdE0wTTRRMFl4T1RBelJEbEZJaXdpWm1seWMzUnVZVzFsSWpvaVZGSkJWa2xUSWl3aVlXUmtjbVZ6Y3lJNklpSXNJbU5zYVdWdWRFbEVJam9pYzJWc1puTmxjblpwWTJVdGNHOXlkR0ZzVmpJaUxDSmpiR2xsYm5ST1lXMWxJam9pYzJWc1puTmxjblpwWTJVdGNHOXlkR0ZzSWl3aWRYTmxjbWxrSWpvaVFVRkJNREF5TVRNek9UY3hJaXdpYkdGemRHNWhiV1VpT2lKVFYwVkZUaUlzSW5Cb2IyNWxJam9pSWl3aVkyeDFZaUk2SWpBd01pSXNJbk4wWVhSbElqb2lJaXdpWlcxaGFXd2lPaUowY21GMmFYTTJNak5BYVdOc2IzVmtMbU52YlNJc0ltMWxiV0psY21sa0lqb2lOREk1TURBeU5ESXdOVGczT1RBd01TSXNJbVY0Y0NJNk1UUTNNVFUyTURrME1Td2ljMk52Y0dVaU9sc2ljMk52Y0dVaVhTd2lZMnhwWlc1MFgybGtJam9pWTJ4cFpXNTBTVVFpZlEubjN6dnhKWThBR0FwUFh4elRuUmpmQktTV01CaXlzV25VUGVXWFF6WUsyczwvc2FtbDpBdHRyaWJ1dGVWYWx1ZT48L3NhbWw6QXR0cmlidXRlPjxzYW1sOkF0dHJpYnV0ZSBOYW1lPSJQaG9uZSIgTmFtZUZvcm1hdD0idXJuOm9hc2lzOm5hbWVzOnRjOlNBTUw6Mi4wOmF0dHJuYW1lLWZvcm1hdDpiYXNpYyI+PHNhbWw6QXR0cmlidXRlVmFsdWUgeG1sbnM6eHM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hIiB4bWxuczp4c2k9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hLWluc3RhbmNlIiB4c2k6dHlwZT0ieHM6c3RyaW5nIi8+PC9zYW1sOkF0dHJpYnV0ZT48c2FtbDpBdHRyaWJ1dGUgTmFtZT0iU3RhdGUiIE5hbWVGb3JtYXQ9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDphdHRybmFtZS1mb3JtYXQ6YmFzaWMiPjxzYW1sOkF0dHJpYnV0ZVZhbHVlIHhtbG5zOnhzPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYSIgeG1sbnM6eHNpPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYS1pbnN0YW5jZSIgeHNpOnR5cGU9InhzOnN0cmluZyIvPjwvc2FtbDpBdHRyaWJ1dGU+PHNhbWw6QXR0cmlidXRlIE5hbWU9Ik1lbWJlcnNoaXBJRCIgTmFtZUZvcm1hdD0idXJuOm9hc2lzOm5hbWVzOnRjOlNBTUw6Mi4wOmF0dHJuYW1lLWZvcm1hdDpiYXNpYyI+PHNhbWw6QXR0cmlidXRlVmFsdWUgeG1sbnM6eHM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hIiB4bWxuczp4c2k9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvWE1MU2NoZW1hLWluc3RhbmNlIiB4c2k6dHlwZT0ieHM6c3RyaW5nIj40MjkwMDI0MjA1ODc5MDAxPC9zYW1sOkF0dHJpYnV0ZVZhbHVlPjwvc2FtbDpBdHRyaWJ1dGU+PHNhbWw6QXR0cmlidXRlIE5hbWU9ImNsdWIiIE5hbWVGb3JtYXQ9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDphdHRybmFtZS1mb3JtYXQ6YmFzaWMiPjxzYW1sOkF0dHJpYnV0ZVZhbHVlIHhtbG5zOnhzPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYSIgeG1sbnM6eHNpPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYS1pbnN0YW5jZSIgeHNpOnR5cGU9InhzOnN0cmluZyI+MDAyPC9zYW1sOkF0dHJpYnV0ZVZhbHVlPjwvc2FtbDpBdHRyaWJ1dGU+PHNhbWw6QXR0cmlidXRlIE5hbWU9InNuIiBOYW1lRm9ybWF0PSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6YXR0cm5hbWUtZm9ybWF0OmJhc2ljIj48c2FtbDpBdHRyaWJ1dGVWYWx1ZSB4bWxuczp4cz0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEiIHhtbG5zOnhzaT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS9YTUxTY2hlbWEtaW5zdGFuY2UiIHhzaTp0eXBlPSJ4czpzdHJpbmciPlNXRUVOPC9zYW1sOkF0dHJpYnV0ZVZhbHVlPjwvc2FtbDpBdHRyaWJ1dGU+PC9zYW1sOkF0dHJpYnV0ZVN0YXRlbWVudD48L3NhbWw6QXNzZXJ0aW9uPjwvc2FtbHA6UmVzcG9uc2U+")
(def bad-jwt-no-sub "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJuYW1lIjoiSm9obiBEb2UiLCJhZG1pbiI6dHJ1ZX0.OLvs36KmqB9cmsUrMpUutfhV52_iSz4bQMYJjkI_TLQ")
(def customer-search (slurp (io/resource "search-results.json")))
(def rppd-response (slurp (io/resource "rppd.xml")))
(def rapd-response (slurp (io/resource "rapd.xml")))
(def document-list-response (slurp (io/resource "document-list.xml")))
(def document-response (slurp (io/resource "document.xml")))
(def claims-response (slurp (io/resource "claims.xml")))
(def soap-fault-response (slurp (io/resource "rppd-bad.xml")))
(def cq-response (slurp (io/resource "cq5-response.json")))
(def no-document (slurp (io/resource "no-document.xml")))
(def billing-history-response (slurp (io/resource "billing-history.xml")))
(def billing-detail-response (slurp (io/resource "billing-detail.xml")))
(def not-xml "Hi There!")
(def cq5-coverages-url "https://e3.pncie.com/bin/querybuilder.json?path=/content/aaa-ie/coverages/AZ/SIG/PAS/01-01-1970&p.properties=code%20label%20shortDesc%20twitterPitch%20riskLevel%20sequence%20shortName&p.hits=selective&p.limit=-1&property=cq:template&property.value=/apps/aaa-ie-common/templates/coverage")
(def auth-response "{\"access_token\":\"eyJhbGciOiJIUzI1NiIsImtpZCI6IjEifQ.eyJjbGllbnRfaWQiOiJjY19pZXdtcyIsImV4cCI6MTQ3NzQ3ODcyMCwic2NvcGUiOltdfQ.nb3jk2XgwxMRiG-34kzS75ZjrwUbm9v6psiCN7vj_To\",\"token_type\":\"Bearer\",\"expires_in\":43199}")

(defn create-mock-request
  [method path]
  (mock/request method path))

(defn create-mock-header
  [req header-name header-value]
  (mock/header req header-name header-value))

(defmacro with-fake-data
  [url-sets & assertions]
  `(with-redefs [cache/get-cache (fn [k#] nil)]
     (with-fake-http (concat ~url-sets [constants/auth-url auth-response]) ~@assertions)))

(defmacro with-good-fake-data
  [& assertions]
  `(with-redefs [cache/get-cache (fn [k#] nil)]
     (with-fake-http [constants/search-url customer-search
                      constants/property-policy-url rppd-response
                      constants/auto-policy-url rapd-response
                      constants/claims-url claims-response
                      constants/document-list-url document-list-response
                      constants/document-url document-response
                      "http://local-document-get-url.com" "Hi World!"
                      constants/auth-url auth-response
                      constants/billing-history-url billing-history-response
                      constants/billing-detail-url billing-detail-response
                      cq5-coverages-url cq-response] ~@assertions)))

(defmacro with-bad-fake-data
  [& assertions]
  `(with-redefs [cache/get-cache (fn [k#] nil)]
     (with-fake-http [constants/search-url customer-search
                      constants/property-policy-url soap-fault-response
                      constants/auto-policy-url soap-fault-response
                      constants/claims-url soap-fault-response
                      constants/document-list-url soap-fault-response
                      constants/auth-url auth-response
                      constants/document-url no-document
                      constants/billing-history-url soap-fault-response
                      constants/billing-detail-url soap-fault-response
                      cq5-coverages-url cq-response] ~@assertions)))

(defn set-prefix-url
  [path]
  (if (clojure.string/includes? path "http") path (str "http://localhost" path)))

(defn generate-saml
  [user-id location]
  (auth-utils/build-saml user-id location))

(defn generate-jwt
  [user-id]
  (auth-utils/build-jwt {:mdmid user-id}))

(defn generate-opentoken
  [attrs]
  (auth-utils/build-opentoken attrs))

(defn bad-auth-header-request
  [method path]
  (create-mock-header (create-mock-request method path) "x-digital-auth" "something-broken"))

(defn bad-jwt-header-request
  [method path]
  (create-mock-header (create-mock-request method path) "x-digital-auth" bad-jwt-no-sub))

(defn jwt-unauth-request
  [method path]
  (create-mock-request method path))

(defn saml-unauth-request
  [path]
  (mock/body
   (create-mock-request :post path)
   (str "SAMLResponse=" bad-saml)))

(defn jwt-auth-request
  [method path user-id]
  (create-mock-header (create-mock-request method path) "x-digital-auth" (generate-jwt user-id)))

(defn saml-auth-request
  [path user-id]
  (let [saml (auth-utils/build-saml user-id (set-prefix-url path))
        body {:SAMLResponse saml}
        request (create-mock-request :post path)]
    (mock/header (mock/body request body) "Content-Type" "application/x-www-form-urlencoded")))
