(ns soa.core
  (:require [clojure.tools.logging :as log]
            [clj-time.format :as time]
            [constants.core :as constants]
            [http.core :as http]
            [clostache.parser :as clostache]
            [cacheing.core :as cache])
  (:use [clj-xpath.core]))

(def date-fields ["termEffectiveDate" "termExpirationDate" "inceptionDate" "dateOfBirth" "statusDate" "effectiveDate" "issuedDate" "dateOfLoss" "reportedDate" "closeDate" "openDate" "agreementEffectiveDate" "policyExpirationDate" "policyEffectiveDate" "transactionDate"])

(defn utc
  [date]
  (try
    (str (time/parse (time/formatter "YYYY-MM-DD-HH:mm") date))
    (catch Exception e
      (str (time/parse (time/formatter "YYYY-MM-DD") date)))))

(defn modify-date
  [key value]
  (cond
      (some (partial = key) date-fields) {key (utc value)}
      :else {key value}))

(defn is-known-key?
  [key]
  (some (partial = key) ["coverage" "vehicle" "driver" "documentProfile" "claim"]))

(defn get-key
  [node]
  (try
    (last (clojure.string/split (name (:tag node)) #":"))
    (catch Exception e
      (log/error "Failed to get key: " node)
      nil)))

(defn build-children
  [f children]
  (let [tag (:tag (first children))
        key (get-key (first children))
        child-count (count children)
        all-same (every? (fn [i] (= tag (:tag i))) children)]
    (if (or (is-known-key? key) (and all-same (> child-count 1)))
      (vec (map (fn [n] (f n)) children))
      (into {} (map (fn [n] (f n)) children)))))

(defn builder
  [n]
  (let [key (get-key n)
        value (:text n)
        children (if (contains? n :children) @(:children n) nil)
        filtered-children (filter (fn [n] (not (= :#text (:tag n)))) children)]
    (if (> (count filtered-children) 0)
      {key (build-children builder filtered-children)}
      (modify-date key value))))

(defn parse-xml
  [xml]
  (try
    ($x "./*" xml)
    (catch Exception e
      (log/error "Failed to parse: " xml))))

(defn xml-is-fault
  [xml]
  (try
    (not (nil? ($x:tag "/Envelope/Body/Fault" xml)))
    (catch Exception e
      false)))

(defn auto-policy?
  [policy]
  (some #(= (:prodTypeCode policy) %) ["AU" "PA"]))

(defn home-policy?
  [policy]
  (= (:prodTypeCode policy) "HO"))

(defn make-policy
  [policy]
  {:number (:policyNumber policy) :type (:prodTypeCode policy) :dataSource (:sourceSystem policy)})

(defn soa-call
  [url request]
  (http/http-call url request :post))

(defn keywordize-xml
  [xml-doc]
  (clojure.walk/keywordize-keys (get (get (builder (first xml-doc)) "Envelope") "Body")))

(defn request-template
  [policy template]
  (clostache/render-resource template policy))

(defn soa-error
  [policy url]
  (log/error "Failed to retrieve information for the following policy: " policy)
  (log/error (str "SOA URL " constants/soa-base-url))
  (log/error (str "ENVOY URL " url))
  {:number (:number policy)
   :error "Policy information found from Search, but not from SOA."
   :envoyUrl url
   :soaUrl constants/soa-base-url})

(defn sanitize
  [policy url xml xml-doc model-fn]
  (if (or (xml-is-fault xml) (nil? xml-doc))
      (soa-error policy url)
      (model-fn policy (keywordize-xml xml-doc))))

(defn soa-xml
  [policy url template model-fn]
  (let [policy-key (str (:number policy) "-" url)
        request (request-template policy template)
        xml (or (cache/get-cache policy-key) (soa-call url request))
        xml-doc (parse-xml xml)]
    (cache/set-cache policy-key xml)
    (sanitize policy url xml xml-doc model-fn)))

(defn transformer
  [soa-url request-template model-fn]
  (fn [full-policy]
    (let [policy (make-policy full-policy)
          summary (soa-xml policy soa-url request-template model-fn)]
      summary)))

(defn model
  ([customer soa-url request-template model-fn]
   (let [policies (:policyList customer)
         autos (filter auto-policy? policies)
         homes (filter home-policy? policies)
         auto-response (map (transformer soa-url request-template model-fn) autos)
         home-response (map (transformer soa-url request-template model-fn) homes)]
     {:auto auto-response
      :home home-response}))

  ([customer auto-soa-url auto-request-template auto-model-fn home-soa-url home-request-template home-model-fn]
   (let [policies (:policyList customer)
         autos (filter auto-policy? policies)
         homes (filter home-policy? policies)
         auto-response (map (transformer auto-soa-url auto-request-template auto-model-fn) autos)
         home-response (map (transformer home-soa-url home-request-template home-model-fn) homes)]
     {:auto auto-response
      :home home-response})
   ))
