(ns coverages.core
  (:require [clojure.data.json :as json]
            [org.httpkit.client :as http]
            [constants.core :as constants]
            [soa.core :as soa]))

(defn- call-cq5
  [url]
  @(http/request {:url url
                  :method :get
                  :headers {"Content-Type" "application/json"}
                  :insecure? true}))

(defn get-coverage-meta-data
  [policy]
  (let [risk-state (:riskState policy)
        product-code (:productCode policy)
        data-source (:dataSource policy)
        effective-date  "01-01-1970"
        call-cq5-response (call-cq5 (str "https://e3.pncie.com/bin/querybuilder.json?path=/content/aaa-ie/coverages/" risk-state "/" product-code "/" data-source "/" effective-date "&p.properties=code%20label%20shortDesc%20twitterPitch%20riskLevel%20sequence%20shortName&p.hits=selective&p.limit=-1&property=cq:template&property.value=/apps/aaa-ie-common/templates/coverage"))
        response (json/read-str (:body call-cq5-response) :key-fn keyword)]
    (:hits response)))

(defn meta-data-filter-by-code
  [code meta-data]
  (first (filter (fn [m] (= (:code m) code)) meta-data)))

(defn coverage-model
  [coverages-from-meta-data]
  (fn
    [coverage]
    (let [code (:code (:coverage coverage))
          meta-data (meta-data-filter-by-code code coverages-from-meta-data)]
      {:code (:code meta-data)
       :twitterPitch (:twitterPitch meta-data)
       :label (:label meta-data)
       :description (:shortDesc meta-data)
       :premiums (:coveragePremium (:coverage coverage))
       :limit (:limit (:coverage coverage))})))

(defn coverages-model-mapper
  [rapd-summary meta-data]
  (map (coverage-model meta-data) (:coverages rapd-summary)))

(defn vehicle-coverages-model
  [vehicle-summary meta-data]
  (map (fn [v] {:coverages (vec (coverages-model-mapper (:vehicle v) meta-data))}) vehicle-summary))

(defn auto-coverages-getter
  [policy rapd-response]
  (let [summary (:policySummary (:autoPolicySummary (:retrieveAutoPolicyDetailResponse rapd-response)))
        meta-data (get-coverage-meta-data summary)
        vehicles-summary (:vehicles (:autoPolicySummary (:retrieveAutoPolicyDetailResponse rapd-response)))]
    {:number (:policyNumber summary)
     :policyCoverages (vec (coverages-model-mapper summary meta-data))
     :vehicles (vec (vehicle-coverages-model vehicles-summary meta-data))}))

(defn home-coverages-getter
  [policy rppd-response]
  (let [summary (:policySummary (:retrievePropertyPolicyDetailResponse rppd-response))]
    {:number (:policyNumber summary)
     :coverages (map (fn [c] {:code (:code (:coverage c))
                              :description (:description (:coverage c))
                              :premiums (:coveragePremium (:coverage c))
                              :limit (:limit (:coverage c))}) (:coverages summary))}))

(defn coverages-model
  [customer]
  (soa/model customer
             constants/auto-policy-url "templates/RetrieveAutoPolicyDetails.xml" auto-coverages-getter
             constants/property-policy-url "templates/RetrievePropertyPolicyDetails.xml" home-coverages-getter))

