(ns framework.core
  (:require [compojure.core :refer :all :as compojure]
            [compojure.route :as route]
            [ring.middleware.cors :refer [wrap-cors]]
            [ring.middleware.params :refer [wrap-params]]
            [clojure.data.json :as json]
            [customer.core :as customers]
            [policies.core :as policies]
            [identity-middleware.core :as identity]
            [auth-utils.core :as auth-utils]
            [http.core :as http]
            [claims.core :as claims]
            [coverages.core :as coverages]
            [constants.core :as constants]
            [documents.core :as documents]
            [discounts.core :as discounts]
            [billing-history.core :as billing-history]
            [billing-detail.core :as billing-detail]
            [clojure.tools.logging :as log]))

(defn- auth-cookie
  [value]
  (str "x-digital-auth=" value))

(defn- user-id
  [request]
  (get (:headers request) "x-digital-user-id"))

(defn- build-auth-cookie
  [request data]
  (let [override-user (:override-user data)
        existing-auth (get (:headers request) "x-digital-auth")
        existing-jwt (auth-utils/get-jwt-attrs existing-auth)]
    (if override-user
      (auth-utils/build-user-and-app-jwt override-user {:id override-user} (:app existing-jwt))
      existing-auth)))


(defn- respond-with-404-error
  [request]
  {:status 404
   :body (json/write-str {:error "Not Found"})
   :headers {"x-digital-error" "The request resulted in a nil return value."
             "Content-Type" "application/json"
             "Set-Cookie" (auth-cookie (get (:headers request) "x-digital-auth"))}})

(defn- respond-with-400-error
  [request data]
  {:status 400
   :body (json/write-str {:error (:error data)})
   :headers {"Content-Type" "application/json"
             "Set-Cookie" (auth-cookie (get (:headers request) "x-digital-auth"))}})

(defn- respond-with-200-success
  [request data]
  (let [cookie (build-auth-cookie request data)
        {response :response status :status :as all-response} data]
    {:status (if status status 200)
     :body (json/write-str (if response response all-response))
     :headers {"Content-Type" "application/json"
               "Set-Cookie" cookie}}))

(defn- execute-body-with
  [request body data]
  (log/debug (str "Executing service function for: " request))
  (if (contains? data :error)
   (respond-with-400-error request data)
   (if-let [handled-data (body data)]
      (respond-with-200-success request handled-data)
      (respond-with-404-error request))))

(defn handle-data-key
  [key customer request]
  (case key
    :customer {:customer (customers/customer-model customer)}
    :policies {:policies (policies/policies-model customer)}
    :coverages {:coverages (coverages/coverages-model customer)}
    :claims {:claims (claims/claims-model customer)}
    :documents {:documents (documents/documents-model customer)}
    :discounts {:discounts (discounts/discounts-model customer)}
    :billing-history {:billingHistory (billing-history/billing-history-model customer)}
    :billing-detail {:billingDetail (billing-detail/billing-detail-model customer)}
    {:error (str "Unknown data key: " (name key))}))

(defn handle-map-key
  [flag customer request]
  (let [key (first (keys flag))
        path (first (vals flag))
        response (http/http-call (str constants/cq5-base-url path ".infinity.json") nil :get)]
    {key (json/read-str response)}))

(defn load-data-from-keys
  [keys customer request]
  (map
    (fn [key]
      (if (map? key)
        (handle-map-key key customer request)
        (handle-data-key key customer request)))
    keys))

(defn- supply-data
  [request keys]
  (let [request-data (select-keys request [:headers :params])
        user-id (user-id request)
        response-map {:request request-data}]
    (if user-id
      (let [customer (customers/get-by-rcd user-id)]
        (into response-map (load-data-from-keys keys customer request)))
      response-map)))

(defmacro endpoint
  "Macro that abstracts compujure and sets up the endpoints. For Connect routes and services."
  [method path data & body]
  (if (= method :get)
    `(compojure/GET ~path request#
                    (#'execute-body-with request# ~@body (#'supply-data request# ~data)))
    `(compojure/POST ~path request#
                     (#'execute-body-with request# ~@body (#'supply-data request# ~data)))))

(defn service
  "Sets up a Connect Services."
  [& end-points]
  (let [routes (apply compojure/routes end-points)]
    (wrap-cors (wrap-params (identity/wrap-identity-middleware-service routes))
               :access-control-allow-origin [#".*"]
               :access-control-allow-headers ["cookie" "x-digital-auth"]
               :access-control-allow-credentials ["true"]
               :access-control-allow-methods [:get :put :post :delete :options])))
