(ns framework.core
  (:require [compojure.core :refer :all :as compojure]
            [compojure.route :as route]
            [ring.middleware.cors :refer [wrap-cors]]
            [ring.middleware.params :refer [wrap-params]]
            [clojure.data.json :as json]
            [customer.core :as customers]
            [policies.core :as policies]
            [identity-middleware.core :as identity]
            [auth-utils.core :as auth-utils]
            [http.core :as http]
            [claims.core :as claims]
            [coverages.core :as coverages]
            [constants.core :as constants]
            [documents.core :as documents]
            [discounts.core :as discounts]
            [billing-history.core :as billing-history]
            [billing-detail.core :as billing-detail]
            [clojure.tools.logging :as log]))

(def ^:private not-found-response {:status 404
                                   :body (json/write-str {:error "Not Found"})
                                   :headers {"x-digital-error" "The request resulted in a nil return value."
                                             "Content-Type" "application/json"}})

(defn- auth-cookie
  [request]
  (str "x-digital-auth=" (get (:headers request) "x-digital-auth")))

(defn- execute-body-with
  ([request body]
   (log/debug (str "Executing route function for: " request))
   (let [response (body request)]
     (if (nil? response)
       not-found-response
       (merge-with clojure.set/union response {:headers {"Set-Cookie" (auth-cookie request)}}))))

  ([request body data]
   (log/debug (str "Executing service function for: " request))
   (if (contains? data :error)
    {:status 400 :body (json/write-str data) :headers {"Content-Type" "application/json" "Set-Cookie" (auth-cookie request)}}
    (if-let [{response :response status :status :as all-response} (body data)]
       {:status (if status status 200) :body (json/write-str (if response response all-response)) :headers {"Content-Type" "application/json" "Set-Cookie" (auth-cookie request)}}
       not-found-response))))

(defn- user-id
  [request]
  (get(:headers request) "x-digital-user-id"))

(defn- handle-data-key
  [key customer request]
  (case key
    :customer {:customer (customers/customer-model customer)}
    :policies {:policies (policies/policies-model customer)}
    :coverages {:coverages (coverages/coverages-model customer)}
    :claims {:claims (claims/claims-model customer)}
    :documents {:documents (documents/documents-model customer)}
    :discounts {:discounts (discounts/discounts-model customer)}
    :billing-history {:billingHistory (billing-history/billing-history-model customer)}
    :billing-detail {:billingDetail (billing-detail/billing-detail-model customer)}
    {:error (str "Unknown data key: " (name key))}))

(defn- handle-map-key
  [flag customer request]
  (let [key (first (keys flag))
        path (first (vals flag))
        response (http/http-call (str constants/cq5-base-url path ".infinity.json") nil :get)]
    {key (json/read-str response)}))

(defn- load-data-from-keys
  [keys customer request]
  (map
    (fn [key]
      (if (map? key)
        (handle-map-key key customer request)
        (handle-data-key key customer request)))
    keys))

(defn- supply-data
  [request keys]
  (let [request-data (select-keys request [:headers :params])]
    (if-let [customer (customers/get-by-rcd (user-id request))]
      (into {:request request-data} (load-data-from-keys keys customer request))
      {:request request-data})))

(defmacro endpoint
  "Macro that abstracts compujure and sets up the endpoints. For Connect routes and services."
  (; ANY endpoint for routes
   [path body]
   `(seq [(compojure/GET ~path request# (#'execute-body-with request# ~body))
          (compojure/POST ~path request# (#'execute-body-with request# ~body))]))

  (; endpoint for routes by method
   [method path body]
   (if (= method :get)
     `(compojure/GET ~path request# (#'execute-body-with request# ~body))
     `(compojure/POST ~path request# (#'execute-body-with request# ~body))))

  (; endpoint for services
   [method path data & body]
   (if (= method :get)
     `(compojure/GET ~path request#
                     (#'execute-body-with request# ~@body (#'supply-data request# ~data)))
     `(compojure/POST ~path request#
                      (#'execute-body-with request# ~@body (#'supply-data request# ~data))))))

(defn service
  "Sets up a Connect Services."
  [& end-points]
  (let [routes (apply compojure/routes end-points)]
    (wrap-cors (wrap-params (identity/wrap-identity-middleware-service routes))
               :access-control-allow-origin [#".*"]
               :access-control-allow-headers ["cookie" "x-digital-auth"]
               :access-control-allow-credentials ["true"]
               :access-control-allow-methods [:get :put :post :delete :options])))

(defn route
  "Sets up Connect Routes."
  [redirect-url & end-points]
  (let [flattened (flatten end-points)
        routes (apply compojure/routes flattened)]
    (wrap-cors (wrap-params (identity/wrap-identity-middleware-route routes redirect-url))
               :access-control-allow-origin [#".*"]
               :access-control-allow-credentials ["true"]
               :access-control-allow-headers ["cookie" "x-digital-auth"]
               :access-control-allow-methods [:get :post :options])))
