(ns reporting.rollbar
  (:require [circleci.rollcage.core :as rollcage]
            [clojure.data.json :as json]
            [org.httpkit.client :as http-client]))


(def rb-key (System/getenv "ROLLBAR_KEY"))

(defn vcap [] (System/getenv "VCAP_APPLICATION"))

(defn app-name
  []
  (try
    (let [json (json/read-str (vcap) :key-fn keyword)]
      (:name json))
    (catch Exception e "no-vcap-app-name")))

(defn app_id
  []
  (try
    (let [json (json/read-str (vcap) :key-fn keyword)]
      (:application_id json))
    (catch Exception e "no-vcap-app-index")))

(def rollcage-client (rollcage/client rb-key {:environment (app-name)}))

(rollcage/setup-uncaught-exception-handler rollcage-client)

(defn rb-base
  [message level status error user uuid code]
  {:payload {
             :access_token rb-key
             :data {
                    :body {:message {:body message
                                     :error error
                                     :status status
                                     :user user
                                     :app_id (or (app_id) "no-app-id-found")}}
                    :level level
                    :timestamp (int (/ (System/currentTimeMillis) 1000))
                    :environment (or (app-name) "no-app-name-found")
                    :request {:url "/authenticate"
                              :method "POST"}
                    :notifier {:name "connect-framework"}
                    :person {:id uuid
                             :username user
                             :code code}
                    :language "clojure"}}})

(defn send-rollbar-event
  [body]
  (let [json-body (json/write-str body)]
    (http-client/request {:url "https://api.rollbar.com/api/1/item/"
                          :method :post
                          :body json-body
                          :insecure? true
                          :headers {"Contenet-Type" "application/json"}})))

(defn rb-error
  [message status error user uuid code]
  (let [body (rb-base message "error" status error user uuid code)]
    (send-rollbar-event body)))

(defn rb-info
  [message status error user uuid code]
  (let [body (rb-base message "info" status error user uuid code)]
    (send-rollbar-event body)))

(defn rb-warn
  [message status error user uuid code]
  (let [body (rb-base message "warning" status error user uuid code)]
    (send-rollbar-event body)))

(defn rb-debug
  [message status error user uuid code]
  (let [body (rb-base message "debug" status error user uuid code)]
    (send-rollbar-event body)))