(ns soa.core
  (:require [clojure.tools.logging :as log]
            [clj-time.format :as time]
            [constants.core :as constants]
            [http.core :as http]
            [clostache.parser :as clostache]
            [cacheing.core :as cache])
  (:use [clj-xpath.core]))

(def
  ^{:doc "These are field names of data from SOA services. Each of these names is checked for and then
          processed through the utc format. The Connect framework standardizes on UTC."}
  date-fields
  ["termEffectiveDate" "termExpirationDate" "inceptionDate" "dateOfBirth" "statusDate" "effectiveDate" "issuedDate" "dateOfLoss" "reportedDate" "closeDate" "openDate" "agreementEffectiveDate" "policyExpirationDate" "policyEffectiveDate" "transactionDate"])

(defn utc
  "This is the UTC formatter. Attempts to format in YYYY-MM-DD-HH:mm but defaults to YYYY-MM-DD."
  [date]
  (try
    (str (time/parse (time/formatter "YYYY-MM-DD-HH:mm") date))
    (catch Exception e
      (str (time/parse (time/formatter "YYYY-MM-DD") date)))))

(defn modify-date
  "This function is called on all SOA data fields to process the date-fields and format them for UTC."
  [key value]
  (cond
      (some (partial = key) date-fields) {key (utc value)}
      :else {key value}))

(defn is-known-key?
  [key]
  (some (partial = key) ["coverage" "vehicle" "driver" "documentProfile" "claim"]))

(defn get-key
  [node]
  (try
    (last (clojure.string/split (name (:tag node)) #":"))
    (catch Exception e
      (log/error "Failed to get key: " node)
      nil)))

(defn build-children
  [f children]
  (let [tag (:tag (first children))
        key (get-key (first children))
        child-count (count children)
        all-same (every? (fn [i] (= tag (:tag i))) children)]
    (if (or (is-known-key? key) (and all-same (> child-count 1)))
      (vec (map (fn [n] (f n)) children))
      (into {} (map (fn [n] (f n)) children)))))

(defn builder
  [n]
  (let [key (get-key n)
        value (:text n)
        children (if (contains? n :children) @(:children n) nil)
        filtered-children (filter (fn [n] (not (= :#text (:tag n)))) children)]
    (if (> (count filtered-children) 0)
      {key (build-children builder filtered-children)}
      (modify-date key value))))

(defn parse-xml
  [xml]
  (try
    ($x "./*" xml)
    (catch Exception e
      (log/error "Failed to parse: " xml))))

(defn xml-is-fault
  [xml]
  (try
    (not (nil? ($x:tag "/Envelope/Body/Fault" xml)))
    (catch Exception e
      false)))

(defn auto-policy?
  "Will determine if the policy provided is of product type code AU or PA."
  [policy]
  (some #(= (:prodTypeCode policy) %) ["AU" "PA"]))

(defn home-policy?
  "Will determine if the policy provided is of product type code HO."
  [policy]
  (= (:prodTypeCode policy) "HO"))

(defn make-policy
  [policy]
  {:number (:policyNumber policy) :type (:prodTypeCode policy) :dataSource (:sourceSystem policy)})

(defn soa-call
  [url request]
  (http/http-call url request :post))

(defn keywordize-xml
  [xml-doc]
  (clojure.walk/keywordize-keys (get (get (builder (first xml-doc)) "Envelope") "Body")))

(defn request-template
  [policy template]
  (clostache/render-resource template policy))

(defn soa-error
  "Returns a map of error information regarding a failed SOA service call."
  [policy url]
  (log/error "Failed to retrieve information for the following policy: " policy)
  (log/error (str "URL attempted: " url))
  {:number (:number policy)
   :error "Policy information found from Search, but not from SOA."
   :soaUrl url})

(defn sanitize
  "Performs an error check on the provided xml. If there is an error will
  return map with error information. Will otherwise call model-fn callback
  passing along the keywordized XML map."
  [policy url xml xml-doc model-fn]
  (if (or (xml-is-fault xml) (nil? xml-doc))
      (soa-error policy url)
      (model-fn policy (keywordize-xml xml-doc))))

(defn soa-xml
  "Combination function that creates the SOA request, calls for the XML and
  returns the map response."
  [policy url template model-fn]
  (let [policy-key (str (:number policy) "-" url)
        request (request-template policy template)
        xml (or (cache/get-cache policy-key) (soa-call url request))
        xml-doc (parse-xml xml)]
    (cache/set-cache policy-key xml)
    (sanitize policy url xml xml-doc model-fn)))

(defn transformer
  "Helper function used within the model function to map over Auto and Home policy.
  Returns a function that will accept the full policy model from Searcha and perform
  a SOA service call with the provided Request template."
  [soa-url request-template model-fn]
  (fn [full-policy]
    (let [policy (make-policy full-policy)
          summary (soa-xml policy soa-url request-template model-fn)]
      summary)))

(defn model
  "A multi-variate function that provided a customer and SOA url, Request and callback function,
  will return a model with Auto and Home keys loaded with relevant data from the SOA service."
  ([customer soa-url request-template model-fn]
   (let [policies (:policyList customer)
         autos (filter auto-policy? policies)
         homes (filter home-policy? policies)
         auto-response (map (transformer soa-url request-template model-fn) autos)
         home-response (map (transformer soa-url request-template model-fn) homes)]
     {:auto auto-response
      :home home-response}))

  ([customer auto-soa-url auto-request-template auto-model-fn home-soa-url home-request-template home-model-fn]
   (let [policies (:policyList customer)
         autos (filter auto-policy? policies)
         homes (filter home-policy? policies)
         auto-response (map (transformer auto-soa-url auto-request-template auto-model-fn) autos)
         home-response (map (transformer home-soa-url home-request-template home-model-fn) homes)]
     {:auto auto-response
      :home home-response})))
