(ns identity-middleware.core
  (:require [ring.middleware.params :as form]
            [clojure.tools.logging :as log]
            [auth-utils.core :as auth]
            [environ.core :as env]
            [ring.middleware.cookies :as cookies]
            [ring.util.request :only [request-url] :as request]
            [mdm-translation.core :as mdm-translate]
            [clojure.data.json :as json]
            [wms-customer-info.core :as wms]))

(defn- handle-form
  [request]
  (let [handler (fn [request] {:status 200 :headers (:params request)})
        wrapped (form/wrap-params handler :form-params)
        response (wrapped request)]
    (clojure.walk/keywordize-keys (:headers response))))

(defn- add-header
  [request header-name header-value]
  (assoc-in request [:headers] (merge (:headers request) {header-name header-value})))

(defn- remove-401
  [request]
  (update-in request [:headers] dissoc "x-digital-401"))

(defn- has-auth?
  [request]
  (let [in-header (contains? (:headers request) "x-digital-auth")
        in-cookie (get (:cookies (cookies/cookies-request request)) "x-digital-auth")]
    (or in-header in-cookie)))

(defn- has-saml?
  [request]
  (if-let [handled-request (handle-form request)]
    (:saml (:headers handled-request))
    false))

(defn- has-direct-access?
  [request]
  (contains? (:headers request) "direct_access_id"))

(defn- get-jwt-from-request
  [request]
  (let [cookies (cookies/cookies-request request)
        cookie-auth (:value (get (:cookies cookies) "x-digital-auth"))
        header-auth (get (:headers request) "x-digital-auth")]
    (or header-auth cookie-auth)))


(defn- error-check
  [response]
  (if (and (instance? clojure.lang.IFn response) (contains? response :error))
    nil
    response))

(defn- auth-failed-filter
  [request]
  (add-header request "x-digital-401" true))

(defn- process-successful-request
  [request user message]
  (log/info message)
  (log/debug user)
  (log/debug request)
  (let [user-id (or (:mdmid user) (:sub user) (:subject user))
        user-id-request (add-header request "x-digital-user-id" user-id)
        auth-request (add-header user-id-request "x-digital-auth" (#'auth/build-jwt (merge user {:application message})))
        auth-with-wms (add-header auth-request "x-digital-wms-customer-info" (json/write-str (merge user {:application message})))]
    (remove-401 auth-with-wms)))

;;; Pull user out and return new User map
(defn- get-user-by-digital-jwt
  [request]
  (let [attrs (#'auth/get-jwt-attrs (get-jwt-from-request request))]
    (when (:sub attrs) attrs)))

(defn- get-mdm-id-from-saml
  [attrs]
  (if-let [cust-key (first (.get attrs "customerRegistrationID"))]
      (error-check (mdm-translate/custkey->mdm cust-key))))

(defn- get-user-by-saml
  [request saml]
  (let [url (request/request-url request)
        attrs (#'auth/get-saml-attrs saml url)
        mdm-id (get-mdm-id-from-saml attrs)]
    (when mdm-id
      (wms/wms-saml mdm-id attrs))))

(defn- get-user-by-opentoken
  [token]
  (#'auth/get-opentoken-attrs token))

(defn- get-user-by-direct-access
  [request]
  (let [mdm-id (error-check (mdm-translate/direct-access->mdm (get (:headers request) "direct_access_id")))]
    (when mdm-id
      (wms/wms-direct-access mdm-id request))))

;;; Handle pulling out the user from the incoming request
(defn- jwt-request
  [request]
  (and (has-auth? request) (get-user-by-digital-jwt request)))

(defn- direct-access-request
  [request]
  (and (has-direct-access? request) (get-user-by-direct-access request)))

(defn- saml-request
  [params request]
  (if-let [saml (:SAMLResponse params)]
    (get-user-by-saml request saml)
    false))

(defn- opentoken-request
  [params]
  (if-let [opentoken (:iportalopentoken params)]
    (get-user-by-opentoken opentoken)
    false))

;;; Handle the user being put back into the request
(defn- saml-filter
  [params request]
  (if-let [user (saml-request params request)]
    (process-successful-request request user "Filtered user by SAML.")
    request))

(defn- jwt-filter
  [request]
  (if-let [user (jwt-request request)]
    (process-successful-request request user "Filtered user by JWT.")
    request))

(defn- direct-access-filter
  [request]
  (if-let [user (direct-access-request request)]
    (process-successful-request request user "Filtered user by Direct Access Header.")
    request))

(defn- opentoken-filter
  [params request]
  (if-let [user (opentoken-request params)]
    (process-successful-request request user "Filtered user by OpenToken")
    request))

(defn- is-unauthenticated?
  [request]
  (contains? (:headers request) "x-digital-401"))

(defn wrap-identity-middleware-service
  "Identity middleware for Connect services.
  Processes incoming requets by assigning them the x-digital-401 header, then processing the x-digital-auth header for a valid JWT. If one exists removes x-digital-401 header and call is a success. Otherwise returns status of 401."
  [handler]
  (fn [request]
    (let [filtered-request (jwt-filter (auth-failed-filter request))]
      (if (is-unauthenticated? filtered-request)
        {:status 401}
        (handler filtered-request)))))

(defn- build-redirect-url
  [request redirect-url]
  (let [host (get (:headers request) "host")
        path (:uri request)]
    (if (#'auth/is-test-cert?)
      (str redirect-url "&TargetResource=https://" host path)
      redirect-url)))

(defn wrap-identity-middleware-route
  "Identity middleware for Connect routes.
  Processes incoming requests by assigning them the x-digital-401 header, then through the following filters:
    jwt-filter, saml-filter, opentoken-filter and direct-access-filter. If one of these authentication types are a success the x-digital-401 header is removed and the page is delivered. Otherwise returns status of 302 to indicated redirect-url."
  [handler redirect-url]
  (fn [request]
    (let [paramed-request (handle-form request)
          filtered-request (direct-access-filter (opentoken-filter paramed-request (saml-filter paramed-request (jwt-filter (auth-failed-filter request)))))
          location (build-redirect-url request redirect-url)]
      (if (is-unauthenticated? filtered-request)
        {:status 302 :headers {"Location" location}}
        (handler filtered-request)))))
