(ns crisptrutski.boot-cljs-test
  (:require
   [boot.core :as core :refer [deftask]]
   [boot.util :refer [info dbug warn fail]]
   [clojure.java.io :as io]
   [clojure.string :as str]
   [crisptrutski.boot-cljs-test.utils :as u]
   [crisptrutski.boot-error.core :as err])
  (:import
   [java.io File]))

(def deps
  {:adzerk/boot-cljs "1.7.170-3"
   :doo              "0.1.7-SNAPSHOT"})

(def default-js-env :phantom)
(def default-ids    #{"cljs_test/suite"})

;; core

(defn no-op [& _])

(defn- scope-as
  "Modify dependency co-ords to have particular scope.
   Assumes not currently scoped"
  [scope deps]
  (for [co-ords deps]
    (conj co-ords :scope scope)))

(defn ensure-deps! [keys]
  (core/merge-env! :dependencies (scope-as "test" (u/filter-deps keys deps))))

(defn validate-cljs-opts! [js-env cljs-opts]
  ((u/r doo.core/assert-compiler-opts) js-env (assoc cljs-opts
                                                :output-to   "placeholder"
                                                :output-dir  "placeholder"
                                                :assert-path "placeholder")))

(defn- gen-suite-ns
  "Generate source-code for default test suite."
  [ns namespaces]
  (let [ns-spec `(~'ns ~ns (:require [doo.runner :refer-macros [~'doo-tests]] ~@(mapv vector namespaces)))
        run-exp `(~'doo-tests ~@(map u/normalize-sym namespaces))]
    (->> [ns-spec '(enable-console-print!) run-exp]
         (map #(with-out-str (clojure.pprint/pprint %)))
         (str/join "\n" ))))

(defn add-suite-ns!
  "Add test suite bootstrap script to fileset."
  [fileset tmp-main id namespaces verbosity]
  (ensure-deps! [:adzerk/boot-cljs])
  (let [relative   #(u/relativize (.getPath tmp-main) (.getPath %))
        out-main   (str id ".cljs")
        src-file   (doto (io/file tmp-main out-main) io/make-parents)
        edn-file   (io/file tmp-main (str out-main ".edn"))
        src-path   (relative src-file)
        edn-path   (relative edn-file)
        exists?    (into #{} (map core/tmp-path) (u/cljs-files fileset))
        suite?     (or (exists? src-path) (exists? (str/replace src-path ".cljs" ".cljc")))
        edn?       (exists? edn-path)
        edn        (when edn? (read-string (slurp (core/tmp-file (core/tmp-get fileset edn-path)))))
        ;; subset namespaces to those directly required by .cljs.edn
        ;; TODO: better would be to use transitive requires, once we have the dep graph
        namespaces (if edn (filter (set (:require edn)) namespaces) namespaces)
        suite-ns   (u/file->ns out-main)
        info       (if (pos? verbosity) info no-op)]
    (if suite?
      (info "Using %s...\n" src-path)
      (do
        (info "Writing %s...\n" src-path)
        (spit src-file (gen-suite-ns suite-ns namespaces))))
    (if edn?
      ;; ensure that .cljs file is required by .cljs.edn, if it's being created.
      (if suite?
        (info "Using %s...\n" edn-path)
        (do (info "Updating %s...\n" edn-path)
            (spit edn-file (update edn :require (fn [xs] (when-not (some #{suite-ns} xs)
                                                           (conj xs suite-ns)))))))
      (do (info "Writing %s...\n" edn-path)
          (spit edn-file {:require [suite-ns]})))
    (if (and suite? edn?)
      fileset
      (core/commit! (core/add-source fileset tmp-main)))))

(deftask prep-cljs-tests
  "Prepare fileset to compile main entry point for the test suite."
  [n namespaces NS ^:! #{str} "Namespaces whose tests will be run. All tests will be run if
                               ommitted.
                               Use symbols for literals.
                               Regexes are also supported.
                               Strings will be coerced to entire regexes."
   e exclusions NS ^:! #{str} "Namespaces or namesaces patterns to exclude."
   v verbosity VAL  int       "Log level"
   i id         VAL str       "TODO: WRITE ME"]
  (let [tmp-main  (core/tmp-dir!)
        verbosity (or verbosity boot.util/*verbosity*)]
    (core/with-pre-wrap fileset
      (let [namespaces (u/refine-namespaces fileset namespaces exclusions)]
        (core/empty-dir! tmp-main)
        (add-suite-ns! fileset tmp-main id namespaces verbosity)))))

(deftask run-cljs-tests
  "Execute test reporter on compiled tests"
  ;; TODO: perhaps vector rather than set, so you can control the running order
  [i ids        IDS  #{str} ""
   j js-env     VAL  kw     "Environment to execute within, eg. slimer, phantom, ..."
   c cljs-opts  OPTS edn    "Options to pass to the Clojurescript compiler."
   v verbosity  VAL  int    "Log level"
   x exit?           bool   "Exit process with runner's exit code on completion."]
  (validate-cljs-opts! js-env cljs-opts)
  (let [js-env    (or js-env default-js-env)
        ids       (if (seq ids) ids default-ids)
        verbosity (or verbosity boot.util/*verbosity*)
        info      (if (pos? verbosity) info no-op)]
    (ensure-deps! [:doo])
    (fn [next-task]
      (fn [fileset]
        (next-task
          (err/with-errors!
            (info "Running cljs tests...\n")
            ((u/r doo.core/print-envs) js-env)
            (doseq [id ids]
              (when (> (count ids) 1)
                (info "• %s\n" id))
              (let [file (str id ".js")
                    ;; TODO: perhaps use metadata on fileset from CLJS task rather
                    path (some->> (core/output-files fileset)
                                  (filter (comp #{file} :path))
                                  (sort-by :time)
                                  last
                                  core/tmp-file
                                  (#(.getPath ^File %)))
                    ;; TODO: perhaps get this logic exposed from boot-cljs rather than repeated
                    cljs (merge cljs-opts {:output-to  path
                                           :output-dir (str/replace path #".js\z" ".out")
                                           :asset-path (or (:asset-path cljs-opts) (str id ".out"))})]
                (if-not path
                  (do (warn "Test script not found: %s\n" file)
                      (err/track-error! {:exit 1 :out ""  :err (format "Test script not found: %s" file)})
                      (when exit? (System/exit 1)))
                  (let [dir (.getParentFile (File. ^String path))
                        {:keys [exit] :as result}
                        ((u/r doo.core/run-script) js-env cljs
                          {:exec-dir dir
                           :verbose  (>= verbosity 1)
                           :debug    (>= verbosity 2)})]
                    (when (pos? exit)
                      (err/track-error! result)
                      (when exit?
                        (System/exit exit)))))))
            fileset))))))

(deftask test-cljs
  "Run cljs.test tests via the engine of your choice.

   The --namespaces option specifies the namespaces to test. The default is to
   run tests in all namespaces found in the project."
  [j js-env        VAL   kw      "Environment to execute within, eg. slimer, phantom, ..."
   n namespaces    NS ^:! #{str} "Namepsaces or namespace patterns to run.
                                  If omitted uses all namespaces ending in \"-test\"."
   e exclusions    NS ^:! #{str} "Namespaces or namesaces patterns to exclude."
   O optimizations LEVEL kw      "The optimization level, defaults to :none."
   i ids           IDS   #{str}  ""
   o out-file      VAL   str     "DEPRECATED Output file for test script."
   c cljs-opts     OPTS  edn     "Options to pass to the Clojurescript compiler."
   u update-fs?          bool    "Enable fileset changes to be passed on to next task.
                                  By default hides all effects for suite isolation."
   v verbosity     VAL   int     "Verbosity level"
   x exit?               bool    "Exit process with runner's exit code on completion."]
  (ensure-deps! [:doo :adzerk/boot-cljs])
  (let [verbosity     (or verbosity @boot.util/*verbosity*)
        ids           (if (seq ids) ids default-ids)
        optimizations (or optimizations :none)
        js-env        (or js-env default-js-env)
        cljs-opts     (merge {:optimizations optimizations}
                             (when (= :node js-env) {:target :nodejs, :hashbang false})
                             cljs-opts)
        wrapper       (if update-fs?
                        identity
                        (fn [wrapped-handler]
                          (fn [handler]
                            (fn [fileset]
                              ((wrapped-handler
                                 (fn [-fs]
                                   (core/commit! fileset)
                                   ;; pass errors down pipeline
                                   (handler (err/track-errors fileset (err/get-errors -fs)))))
                                fileset)))))]
    (validate-cljs-opts! js-env cljs-opts)
    (wrapper
      (comp (reduce comp
                    (for [id ids]
                      (prep-cljs-tests
                        :id id
                        :namespaces namespaces
                        :exclusions exclusions
                        :verbosity verbosity)))
            ((u/r adzerk.boot-cljs/cljs)
              :ids ids
              :compiler-options cljs-opts)
            (run-cljs-tests
              :ids ids
              :cljs-opts cljs-opts
              :js-env js-env
              :exit? exit?
              :verbosity verbosity)))))

(deftask exit!
  "Exit with the appropriate error code"
  []
  (fn [_]
    (fn [fs]
      (let [errs (err/get-errors fs)]
        (System/exit (if (seq errs) 1 0))))))
