(ns craftybones.cricsheet
  (:require [clojure.spec.alpha :as s]))

(s/def ::date (s/or :inst inst? :str #(re-matches #"\d{4}-\d{2}-\d{2}" %)))
(s/def :meta/data_version (s/and double? pos?))
(s/def :meta/created ::date)
(s/def :meta/revision pos-int?)
(s/def ::meta (s/keys :req-un [:meta/data_version :meta/revision :meta/created]))

(s/def :bowl_out/bowler string?)
(s/def :bowl_out/outcome #{"hit" "miss"})
(s/def ::bowl_out (s/keys :opt-un [:bowl_out/bowler :bowl_out/outcome]))

(s/def ::city string?)
(s/def ::competition #{"IPL"
                       "Big Bash League"
                       "NatWest T20 Blast"
                       "Pakistan Super League"
                       "Women's Big Bash League"
                       "Caribbean Premier League"})

(s/def ::dates (s/coll-of ::date :min-count 1))
(s/def ::gender #{"male" "female"})
(s/def ::match_type #{"Test" "ODI", "T20", "IT20", "ODM", "MDM"})
(s/def ::neutral #{1})

(s/def :outcome.by/innings #{1})
(s/def :outcome.by/runs pos-int?)
(s/def :outcome.by/wickets pos-int?)
(s/def :outcome/by (s/or
                    :by-innings (s/keys :req-un [:outcome.by/innings
                                                 :outcome.by/runs])
                    :by-runs    (s/keys :req-un [:outcome.by/runs])
                    :by-wickets (s/keys :req-un [:outcome.by/wickets])))

(s/def :outcome/bowl_out string?)
(s/def :outcome/eliminator string?)
(s/def :outcome/method #{"D/L"})
(s/def :outcome/result #{"draw" "no result" "tie"})
(s/def :outcome/winner string?)

(s/def ::outcome (s/keys :opt-un [:outcome/by :outcome/bowl_out
                                  :outcome/eliminator :outcome/method
                                  :outcome/result :outcome/winner]))

(s/def ::overs pos-int?)
(s/def ::player_of_match (s/coll-of string?))
(s/def ::supersubs (s/map-of symbol? (s/coll-of string?)))
(s/def ::teams (s/coll-of string? :count 2))
(s/def :toss/winner string?)
(s/def :toss/decision #{"bat" "field"})
(s/def ::toss (s/keys :req-un [:toss/winner :toss/decision]))
(s/def ::umpires (s/coll-of string? :count 2))
(s/def ::venue string?)

(s/def ::info (s/keys :req-un [::dates ::gender ::match_type
                               ::outcome ::toss ::umpires]))

(s/def :inning/team string?)
(s/def :inning/absent_hurt (s/coll-of string? :min-count 1))
(s/def :inning.penalty_runs/pre pos-int?)
(s/def :inning.penalty_runs/post pos-int?)
(s/def :inning/penalty_runs (s/keys :opt-un [:inning.penalty_runs/post
                                             :inning.penalty_runs/pre]))
(s/def :inning/declared #{1})

(s/def :delivery/batsman string?)
(s/def :delivery/non_striker string?)
(s/def :delivery/bowler string?)

(s/def :delivery.runs/batsman int?)
(s/def :delivery.runs/extras int?)
(s/def :delivery.runs/non_boundary #{1})
(s/def :delivery.runs/total int?)
(s/def :delivery/runs (s/keys :req-run [:delivery.runs/batsman
                                        :delivery.runs/extras
                                        :delivery.runs/total]
                              :opt-un  [:delivery.runs/non_boundary]))
(s/def ::delivery (s/keys :req-un [:delivery/batsman
                                   :delivery/non_striker
                                   :delivery/bowler
                                   :delivery/runs]))

(s/def :delivery/over double?)
(s/def ::deliveries (s/coll-of (s/map-of :delivery/over ::delivery)))

(s/def :replacements/in string?)
(s/def :replacements/out string?)
(s/def :replacements/reason string?)
(s/def :replacements/team string?)
(s/def :replacements.match/reason #{"concussion_substitute" "supersub"})
(s/def :replacements.role/reason #{"excluded - high full pitched balls"
                                   "excluded - running on the pitch"
                                   "injury"
                                   "too many overs"
                                   "unknown"})
(s/def :replacements.role/role #{"batter" "bowler"})

(s/def :replacements/role (s/keys :req-un [:replacements/in
                                           :replacements/reason
                                           :replacements/role]
                                  :opt-un [:replacements/out]))

(s/def :replacements/match (s/keys :req-un [:replacements/in
                                            :replacements/reason
                                            :replacements/out
                                            :replacements/team]))

(s/def ::replacements (s/* (s/alt :role :replacements/role
                                    :match :replacements/match)))

(s/def :wicket/kind #{"bowled"
                      "caught"
                      "caught and bowled"
                      "lbw"
                      "stumped"
                      "run out"
                      "retired hurt"
                      "hit wicket"
                      "obstructing the field"
                      "hit the ball twice"
                      "handled the ball"
                      "timed out"})

(s/def :wicket/player_out string?)
(s/def :wicket/fielders (s/coll-of string?))

(s/def :extras/byes int?)
(s/def :extras/legbyes int?)
(s/def :extras/noballs int?)
(s/def :extras/penalty int?)
(s/def :extras/wides int?)

(s/def ::extras (s/keys :opt-un [:extras/byes
                                 :extras/legbyes
                                 :extras/noballs
                                 :extras/penalty
                                 :extras/wides]))

(s/def ::inning (s/keys :req-un [::deliveries
                                 :inning/team]
                        :opt-un [:inning/absent_hurt
                                 :inning/penalty_runs
                                 :inning/declared]))

(s/def ::innings (s/coll-of (s/map-of keyword? ::inning :count 1)))

(s/def ::sheet (s/keys :req-un [::meta ::info ::innings]))
