(ns leiningen.new.crafty
  (:require [clojure.java.io :as io]
            [leiningen.core.main :as main]
            [leiningen.new.templates :refer [renderer year project-name
                                             ->files sanitize-ns name-to-path]]))

(def render (renderer "crafty"))

(defn resource [name]
  (io/input-stream (io/resource (str "leiningen/new/crafty/" name))))

(defmulti profile-data  (fn [module name] module))
(defmulti profile-files (fn [module data] module))

(defmethod profile-data  :default [_ _] {})
(defmethod profile-files :default [_ _] [])

(defmethod profile-data :base [_ name]
  (let [main-ns (sanitize-ns name)]
    {:raw-name    name
     :name        (project-name name)
     :namespace   main-ns
     :dirs        (name-to-path main-ns)
     :year        (year)}))

(defmethod profile-files :base [_ data]
  [["project.clj"               (render "base/project.clj" data)]
   ["README.md"                 (render "base/README.md" data)]
   [".gitignore"                (render "base/gitignore" data)]
   ["config/defaults.edn"       (render "base/defaults.edn" data)]
   ["config/test.edn"           (render "base/test.edn" data)]
   ["config/dev.edn"            (render "base/dev.edn" data)]
   ["config/pro.edn"            (render "base/pro.edn" data)]
   ["resources/log4j2.xml"      (render "base/log4j2.xml" data)]
   ["dev/dev.clj"               (render "base/dev.clj" data)]
   ["dev/user.clj"              (render "base/user.clj" data)]
   ["src/{{dirs}}/main.clj"     (render "base/main.clj" data)]
   ["src/{{dirs}}/system.clj"   (render "base/system.clj" data)]
   "src/{{dirs}}/component"
   "src/{{dirs}}/service"
   "test/{{dirs}}"])

(defmethod profile-data :example [_ _]
  {:example? true})

(defmethod profile-files :example [_ data]
  [["src/{{dirs}}/service/example.clj"       (render "example/service.clj" data)]
   ["test/{{dirs}}/service/example_test.clj" (render "example/service_test.clj" data)]])

(defn profiles [hints]
  (for [hint hints :when (re-matches #"\+[A-Za-z0-9-]+" hint)]
    (keyword (subs hint 1))))

(defn crafty
  "Create a new Crafty API.
  
  Accepts the following profile hints:
    +example - adds an example service that uses swagger"
  [name & hints]
  (when (.startsWith name "+")
    (main/abort "Failed to create project: no project name specified."))
  (main/info (str "Generating a new Crafty project named " name "..."))
  (let [mods  (cons :base (profiles hints))
        data  (reduce into {} (map #(profile-data % name) mods))
        files (reduce into [] (map #(profile-files % data) mods))]
    (apply ->files data files)))
