(ns crafty.generate
  (:require
   [leiningen.core.project :as project]
   [stencil.core :as stencil]
   [clojure.java.io :as io]
   [clojure.string :as str]))

(def ^:dynamic *ns-prefix*
  "The common prefix all namespaces in the project share."
  nil)

(defn set-ns-prefix
  "Set the namespace prefix."
  [prefix]
  (alter-var-root #'*ns-prefix* (constantly (str prefix))))

(defn- name-to-path [name]
  (-> name
      (str/replace "-" "_")
      (str/replace "." java.io.File/separator)))

(defn camel-case [hyphenated]
  (->> (str/split hyphenated #"-")
       (map str/capitalize)
       (str/join)))

(defn- make-parent-dirs [path]
  (when-let [parent (.getParentFile (io/file path))]
    (.mkdirs parent)))

(defn- create-raw-file [data raw-string out-path]
  (let [out-path (stencil/render-string out-path data)]
    (println "Creating file" out-path)
    (make-parent-dirs out-path)
    (spit out-path raw-string)))

(defn- create-file [data in-path out-path]
  (create-raw-file data (stencil/render-file in-path data) out-path))

(defn- create-dir [data path]
  (let [path (stencil/render-string path data)]
    (println "Creating directory" path)
    (.mkdirs (io/file path))))

(defn service
  "Generate a new Crafty service with the supplied name."
  [name]
  (assert *ns-prefix* "crafty.generate/*ns-prefix* not set")
  (let [project   (project/read-raw "project.clj")
        namespace (str *ns-prefix* ".service." name)
        path      (name-to-path namespace)]
    (doto {:ns-prefix *ns-prefix* :name (str name) :namespace namespace :path path}
      (create-file "crafty/generate/templates/service/source.clj" "src/{{path}}.clj")
      (create-file "crafty/generate/templates/service/test.clj" "test/{{path}}_test.clj"))
    nil))

(defn component
  "Generate a new Crafty component with the supplied name."
  [name]
  (assert *ns-prefix* "crafty.generate/*ns-prefix* not set")
  (let [project   (project/read-raw "project.clj")
        namespace (str *ns-prefix* ".component." name)
        path      (name-to-path namespace)
        record    (camel-case (str name))]
    (doto {:name (str name) :namespace namespace :path path :record record}
      (create-file "crafty/generate/templates/component/source.clj" "src/{{path}}.clj")
      (create-file "crafty/generate/templates/component/test.clj" "test/{{path}}_test.clj"))
    nil))
