(ns ^{:doc "Macros to help building dc.js projects with Clojurescript"
      :author "Alexandre Cotarmanac'h"}
     cotarmanach.dcljs-lib
  )

(def ^:dynamic *debug* false)


(defmacro get-a [obj prop]
  """
    get-a : 'a -> str -> 'b
    gets property prop of object obj
    if *debug* throws error when property does not exist.
  """
  (let [
        test-form (if *debug*
                    `(if
                       (not (~'property? ~obj ~prop))
                       (throw (str "Error: property " ~prop " not defined "))))
        prop-accessor (symbol (str ".-" prop))
        body-form (if (string? prop)
                    `(~prop-accessor ~obj)
                    `(aget ~obj ~prop) ;; runtime only
                    )
        ]
    (if *debug*
      `(do ~test-form ~body-form)
       body-form)))

(defmacro set-a [obj prop expr]
  """
    set-a : 'a -> str -> 'b -> 'b
    sets property prop of object obj to be expr
  """
  (let [
        test-form (if *debug*
                    `(if
                       (not (~'property? ~obj ~prop))
                       (~'warning (str "Property " ~prop " was not defined "))))
        body-form `(~'aset ~obj ~prop ~expr)
        ]
  (if *debug*
        `(do ~test-form ~body-form)
        body-form)
    ))
;; (macroexpand-1 '(set-a d "date" 'bla))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                          Dataset macro

(defmacro additive-update 
  [target source fun field]
  `(set-a ~target ~field
         (~fun (get-a ~target ~field) (get-a ~source ~field)))
  )

(defmacro ratio [o a b]
  `(if (> (get-a ~o ~b) 0)
     (float (/ (get-a ~o ~a) (get-a ~o ~b)))
     0)
  )

(defmacro data-op-group-additive [op data-additive-fields]
  (concat '(do) (for [f data-additive-fields]
    `(additive-update ~'p ~'v ~op ~f)
    ))
  )

(defmacro data-op-group-calc [data-calc-spec]
  (concat '(do) (for [[field [fun-name field-1 field-2]] data-calc-spec]
    (cond 
      (= fun-name :ratio) `(set-a ~'p ~field (ratio ~'p ~field-1 ~field-2))
      (= fun-name :count) `(set-a ~'p ~field (get-a ~'p "n"))
      :else (throw (Exception. "Unknown method."))
      )
    ))
  )


(defmacro data-add-group [data-additive-fields data-calc-spec]
  `(fn [~'p ~'v]
     (data-op-group-additive + ["n"])
     (data-op-group-additive + ~data-additive-fields)
     (data-op-group-calc ~data-calc-spec)
     ~'p
     )
  )

(defmacro data-del-group [data-additive-fields data-calc-spec]
  `(fn [~'p ~'v]
     (data-op-group-additive - ["n"])
     (data-op-group-additive - ~data-additive-fields)
     (data-op-group-calc ~data-calc-spec)
     ~'p
     )
  )

(defmacro data-init-group [data-additive-fields data-calc-spec]
  (let [
        n-init ["n" 0]
        additive-init (interleave data-additive-fields  (repeat 0))
        data-calc-fields (keys data-calc-spec)
        extra-init (interleave data-calc-fields  (repeat 0))
        ]
    (concat '(fn []) [ (concat '(js-obj) n-init additive-init extra-init)])
    )
  )




;; (defmacro data-accessor [fields] 
;;   (concat 
;;     '(fn [d])
;;     [(concat 
;;       '(do)
;;        (for [[fname strfun] fields :when (not= "identity" (name strfun))]
;;          `(set-a ~'d ~fname (~(symbol strfun) (get-a ~'d ~fname)))
;;          )
;;        '(d)
;;       )])
;;   )



(defn parse-fields [fields-spec]
  (for [[fname {strfun :parse}] (:fields fields-spec)
        :when  (and strfun (not= "identity" (name strfun)))]
    `(set-a ~'d ~fname (~(symbol strfun) (get-a ~'d ~fname)))
    ) 
  )


;; (defn calc-fields [fields-spec]
;;   (for [[fname {fun :calc args :args}] (:fields fields-spec)
;;         :when fun
;;         ]
;;     (concat
;;       `(set-a ~'d ~fname )
;;       [
;;        (concat
;;          `(~(symbol fun))
;;          (for [a args] `(get-a ~'d ~a)))
;;        ]
;;       )
;;     ) 
;;   )



;; TODO : check that names have been already defined when applying function fun
(defn calc-fields [fields-spec]
  (for [[fname {fun :calc args :args}] (:fields fields-spec)
        :when fun
        ]
      `(set-a ~'d ~fname (~(symbol fun) ~@(for [a args] `(get-a ~'d ~a))))
    ) 
  )


(defmacro data-accessor [fields-spec]
  (concat
    '(fn [d])
    [`(do 
       (set-a ~'d "n" 1)
       ~@(parse-fields fields-spec)
       ~@(calc-fields fields-spec)
       ~'d
       )]
    ) 
  )

(defmacro dataset [spec]
 "outputs {accessor add del init}"
  (let [
        data-additive-fields (for [[field-name field-metadata] (:fields spec)
                                   :when (:additive field-metadata)]
                                field-name )
        data-calc-spec (get-in spec [ :group :calc-fields]) 
        calc-fields (into [] (keys data-calc-spec))
        grouping-fields (into [] (concat data-additive-fields calc-fields))
        ]
    `{ 
      :accessor (data-accessor ~spec)
      :add (data-add-group ~data-additive-fields ~data-calc-spec)
      :del (data-del-group ~data-additive-fields~data-calc-spec)
      :init (data-init-group ~data-additive-fields ~data-calc-spec) 
      :calc-fields ~calc-fields
      :grouping-fields ~grouping-fields
      } 
    ) 
  )

(macroexpand-1 '(dataset
    {
      :fields [
                    ["date"     {:parse parse-date :groupable true :additive false}]
                     ["open"     {:parse js/parseFloat :groupable false :additive false}]
                     ["high"     {:parse js/parseFloat :groupable false :additive false}]
                     ["low"      {:parse js/parseFloat :groupable false :additive false}]
                     ["close"    {:parse js/parseFloat :groupable false :additive false}]
                     ["volume"   {:parse js/parseInt :groupable false :additive true}]
                     ["oi"       {:parse js/parseInt :groupable false :additive true}]
                     ["month"        {:calc calc-month :args ["date"] :groupable true :additive false}]
                     ["gain"         {:calc -    :args ["close" "open"] :groupable false :additive true}]
                     ["fluctuation"  {:calc abs  :args ["gain"] :groupable false :additive true}]
                     ["index"        {:calc calc-index :args ["close" "open"] :groupable false :additive true}]
                     ["gain-or-loss" {:calc gain-or-loss :args ["close" "open"] :groupable true :additive false}]
                     ["quarter"      {:calc quarter :args ["month"] :groupable true :additive false}]
                     ["day"          {:calc day :args ["date"] :groupable true :additive false}]
               ]
     })
)






;; (defmacro dataset [spec]
;;  "outputs {accessor add del init}"
;;   (let [
;;         fields+accessors (for [[field-name field-metadata] (:fields spec)]
;;                           [field-name (:parse field-metadata)])
;;         data-additive-fields (for [[field-name field-metadata] (:fields spec)
;;                                    :when (:additive field-metadata)]
;;                                 field-name )
;;         data-calc-spec (get-in spec [ :group :calc-fields]) 
;;         ]
;;     `{ 
;;       :accessor (data-accessor ~fields+accessors)
;;       :add (data-add-group ~data-additive-fields ~data-calc-spec)
;;       :del (data-del-group ~data-additive-fields~data-calc-spec)
;;       :init (data-init-group ~data-additive-fields ~data-calc-spec) 
;;       } 
;;     ) 
;;   )

;; (macroexpand-1 '
;; (dataset {
;;   :fields {
;;            "SITE_ID"     {:parse identity :groupable true :additive false}
;;            "N_DISPLAYS"  {:parse js/parseInt :groupable false :additive true}
;;            "REVENUES"    {:parse js/parseFloat :groupable false :additive true}
;;            }
;;   :group {
;;          ;; optionally :fields [ fields to be included in the group methods ] 
;;          :calc-fields {
;;                        "conversion" [:ratio "N_PURCHASES" "N_SESSIONS"]
;;                        } 
;;           }}
;;   ))
;; 
;; 
