;
; Copyright (c) 2022 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.markdown
  (:require [clojure.java.io :as io]
            [clojure.pprint :as pprint]
            [clojure.string :as str]
            [itl.print-control :refer [attempt]]
            [itl.print-control :as pc]
            [itl.tags :as tags])
  (:import (com.vladsch.flexmark.ext.tables TablesExtension
                                            TableBlock
                                            TableHead
                                            TableBody
                                            TableCaption
                                            TableSeparator)
           (com.vladsch.flexmark.html HtmlRenderer)
           (com.vladsch.flexmark.parser Parser)
           (com.vladsch.flexmark.util.data MutableDataSet)
           (com.vladsch.flexmark.ext.admonition AdmonitionExtension
                                                AdmonitionBlock)
           (com.vladsch.flexmark.util.ast Node)
           (com.vladsch.flexmark.formatter Formatter)
           (java.io PrintWriter
                    FileWriter
                    FileReader
                    FileNotFoundException
                    Reader
                    File)
           (java.time ZonedDateTime)
           (java.util Arrays)))

(def status-names {:pass "pass"
                   :fail "fail"
                   :exception "exception"
                   :note "note"})

(def status-handlers {:exception #(str "E[" % "]")})

(defn- output-content [tag & content]
  (let [value (str/join content)]
    (if (str/blank? value)
      value
      (format "<%s class=\"output\">%s</%s>" tag value tag))))

(defn- status-element [status-name & content]
  (str "<span class=\"" status-name "\">" (str/join content) "</span>"))

(defn status
  "Mark some content as having a particular status"
  [given-status content]
  (status-element
   (status-names given-status)
   (->> content
        ((or (status-handlers given-status) identity))
        (output-content "strong"))))

(defn- arg [arg i]
  (let [[k v] (str/split arg #"=")]
    (if v {k v} {i k})))

(defn- add-args [l a]
  (concat l (map str/trim (str/split a #","))))

(defn make-args [args]
  (apply merge (map arg
                    (reduce add-args [] args)
                    (map str (rest (range))))))

(defn- children [n] (when n (iterator-seq (.getChildIterator n))))

(defn- text-content [n] (->> n .getChars str))

(defn content->elements [parser content]
  (->> content
       str
       (.parse parser)
       children
       first
       children))

(defn append-to-node!
  ([parser current-node content]
   (append-to-node! current-node (content->elements parser content)))
  ([current-node new-children]
   (doseq [n new-children] (.appendChild current-node n))))

(defn replace-node!
  ([parser current-node content]
   (replace-node! current-node (content->elements parser content)))

  ([current-node new-children]
   (.removeChildren current-node)
   (append-to-node! current-node new-children)))

(defn executed! [parser current-node op-name]
  (replace-node! parser current-node (output-content "em" op-name)))

(defn exception-occurred! [parser current-node op-name t exception]
  (pc/print-exception t)
  (append-to-node! parser
                   current-node
                   (exception (str "exception executing '" op-name "' -- "
                                   (or (.getMessage t)
                                       (-> t .getClass .getName))))))

(defn- cell-text [n]
  (let [t (-> n .getText str)] (when-not (str/blank? t) t)))

(defn discern-table-structure [table-node]
  (reduce (fn [m n]
            (cond (instance? TableHead n) (assoc m :table-head n)
                  (instance? TableBody n) (assoc m :table-body n)
                  (instance? TableSeparator n) (assoc m :table-sep n)
                  (instance? TableCaption n) (assoc m :table-caption n)
                  :else m))
          {}
          (children table-node)))

(defn nil->space [v] (or v " "))

(defn table-row [values]
  (str "|" (str/join "|" (map nil->space values)) "|\n"))

(defn replace-table-body! [parser table-body labels separators raw-rows]
  (let [new-table
        (.parse parser
                (str/join
                 (map table-row (into [labels separators] raw-rows))))

        new-rows
        (-> new-table
            children
            first
            discern-table-structure
            :table-body
            children)]
    (replace-node! table-body new-rows)))

(defn table [parser
             current-state
             exception
             {:keys [table-head table-body table-sep table-caption]}
             args
             f
             f-name]
  (let [header-cells (-> table-head children first children)
        labels (map cell-text header-cells)
        separators (map cell-text (-> table-sep children first children))
        table-rows (children table-body)
        row-text (map (comp (partial map cell-text) children)
                      table-rows)
        original-rows (map (partial zipmap labels) row-text)
        opargs [@current-state args {:labels labels
                                     :rows original-rows
                                     :raw-rows row-text}]]
    (try
      (let [[s {:keys [rows raw-rows]}] (apply f opargs)
            _ (reset! current-state s)]
        (cond
          ;; Tabular data
          rows
          (do
            (pc/attempt println "=== INPUT ===")
            (pc/attempt pprint/print-table original-rows)
            (flush)
            (doseq [[output table-cells]
                    (zipmap rows (map children table-rows))

                    [label cell] (zipmap labels table-cells)]
              (let [new-value (output label)]
                (replace-node! parser cell new-value)))
            (pc/attempt println "=== OUTPUT ===")
            (pc/attempt pprint/print-table rows))

          ;; Unstructured data in a vague grid-like thing
          raw-rows
          (do
            (pc/attempt println "=== RAW HEADERS ===")
            (pc/attempt println labels)
            (pc/attempt println "=== RAW INPUT ROWS ===")
            (doseq [row row-text] (pc/attempt println row))
            (replace-table-body! parser table-body labels separators raw-rows)
            (pc/attempt println "=== RAW OUTPUT ROWS ===")
            (doseq [row raw-rows] (pc/attempt println row)))))
      (catch Throwable t
        (exception-occurred! parser table-caption f-name t exception)))))

(def ^:private admonition-js (-> "admonition.js" io/resource slurp))

(def ^:private ^:dynamic css-content (-> "itl-md.css" io/resource slurp))

(defmacro with-css [css & body]
  `(if ~css
     (binding [css-content (slurp ~css)] ~@body)
     ~@body))

(def ^:private table-execution #"\((.*?):(.+?)?( (.*?))?\)")

(defmulti handle-node (fn [node _] (class node)))

(defmethod handle-node AdmonitionBlock
  [node {:keys [current-tags]}]
  (let [title (-> node
                  .getTitleChars
                  .unescape
                  (.replace "\"" "")
                  (str/split #":"))]
    (when (= 2 (count title))
      (reset! current-tags
              (-> title
                  first
                  (str/split #",")
                  ((partial map str/trim))
                  set
                  (disj "none"))))))

(defmethod handle-node TableBlock
  [node {:keys [current-tags
                table-types
                parser
                current-state
                exception]}]
  (let [{:keys [table-caption] :as table-structure}
        (discern-table-structure node)

        text (->> table-caption children (map text-content) (str/join ""))
        table-args (when text (re-find table-execution text))]
    (when table-args
      (tags/when-tags-allowed
       @current-tags
       (let [[_ table-type & args] (remove nil? table-args)
             args (make-args args)
             table-fn (@table-types table-type)]
         (if table-fn
           (table parser
                  current-state
                  exception
                  table-structure
                  args
                  table-fn
                  table-type)
           (attempt println
                    "WARNING: unknown table type"
                    (str "'" table-type "'"))))))))

(defmethod handle-node Node [_ _] nil)

(def options (doto (MutableDataSet.)
               (.set Parser/EXTENSIONS [(AdmonitionExtension/create)
                                        (TablesExtension/create)])))

(defn new-parser [] (.build (Parser/builder options)))

(defmulti parse type)

(defmethod parse String [filename]
  (let [f (io/file filename)]
    (if (.exists f)
      (with-open [reader (FileReader. f)] (parse reader))
      (throw (FileNotFoundException. (.getAbsolutePath f))))))

(defmethod parse Reader [reader] (.parseReader (new-parser) reader))

(defn reformat-md-file [^String filename]
  (let [formatter (.build (Formatter/builder options))
        doc (parse filename)]
    (with-open [writer (FileWriter. filename)]
      (.render formatter doc writer))
    "done"))

(defn execute [reader writer table-types current-state exception]
  (with-open [w (PrintWriter. writer)]
    (let [renderer (.build (HtmlRenderer/builder options))
          doc (parse reader)
          parser (new-parser)
          wr (fn [& s] (.println w (str/join s)))]
      (wr "<!DOCTYPE html><html><head><meta charset=\"UTF-8\">")
      (wr "<meta http-equiv=\"X-UA-Compatible\" content=\"IE=edge\">")
      (wr "<meta name=\"viewport\" content=\"width=device-width, "
          "initial-scale=1.0\">")
      (wr "<meta name=\"generator\" content=\"itl\">")
      (wr "<style>")
      (wr css-content)
      (wr "</style>")
      (wr "</head><body class=\"article\"><div id=\"content\">")

      (let [current-tags (atom #{})]
        (doseq [node (children doc)]
          (handle-node node {:parser parser
                             :current-tags current-tags
                             :table-types table-types
                             :current-state current-state
                             :exception exception})))
      (.render renderer doc w)
      (wr "</div>")
      (wr "<div id=\"footer\" class=\"md-footer\">")
      (wr "<div id=\"footer-text\">")
      (wr "Last updated " (ZonedDateTime/now))
      (wr "</div>")
      (wr "</div>")
      (wr "<script type=\"text/javascript\">")
      (wr admonition-js)
      (wr "</script>")
      (wr "</body></html>"))))

(defn die [exit-code & msg]
  (apply println msg)
  (System/exit exit-code))

(defn reformat [folder]
  (->> folder
       .listFiles
       Arrays/asList
       (map (fn [^File f] (.getAbsolutePath f)))
       (filter (fn [filename] (.endsWith filename ".md")))
       (map reformat-md-file)
       dorun))

(defn -main [& args]
  (if (= 1 (count args))
    (let [folder (io/file (first args))]
      (if (.exists folder)
        (reformat folder)
        (die 1 "Folder not found:" (.getAbsolutePath folder))))
    (die 1 "Please specify a folder to reformat")))