;
; Copyright (c) 2022 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.core
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [com.climate.claypoole :as cp]
            [itl.markdown :as md]
            [itl.print-control :as pc]
            [itl.worker-threads :as wt]
            [clojure.walk :as walk])
  (:import (java.io FileOutputStream FileNotFoundException)
           (java.time LocalDateTime LocalDate)
           (java.time.format DateTimeFormatter)
           (org.apache.commons.io FilenameUtils)
           (java.util UUID)
           (clojure.lang Keyword))
  (:gen-class))

;; # Integration Test Library

(def ^:dynamic ^:private max-table-threads 1)
(defn with-max-threads* [n f] (binding [max-table-threads n] (f)))
(defmacro with-max-threads [n & body] `(with-max-threads* ~n (fn [] ~@body)))

;; These functions are bound in `execute`. They mark an assertion
;; as "passed" or "failed."
(def ^:dynamic pass)
(def ^:dynamic fail)
(def ^:dynamic exception)
(def ^:dynamic note)

;; ## Document execution

(def table-types (atom {}))

(defn execute
  "Parse a document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out`. The `initial-state` is a map of bindings that the document can use."
  [execute-fn status-fn initial-state in out]
  (let [current-state (atom initial-state)
        local-stats (atom {:pass 0 :fail 0 :exception 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (status-fn given-status content))
        start-time (System/currentTimeMillis)]
    (binding [pass (partial local-status :pass)
              fail (partial local-status :fail)
              exception (partial local-status :exception)
              note (partial status-fn :note)]
      (with-open [reader (io/reader in)
                  writer (io/writer out)]
        (execute-fn reader writer table-types current-state exception))
      (assoc @current-state
        :stats
        (assoc @local-stats
          :elapsed-time (- (System/currentTimeMillis) start-time)
          :run-date (.format (LocalDateTime/now)
                             (DateTimeFormatter/ofPattern
                               "yyyy-MM-dd HH:mm")))))))

(defn- file-type [f]
  (-> f io/file .getName FilenameUtils/getExtension))

(defn- rename-to-html [f]
  (-> f io/file .getName FilenameUtils/getBaseName (str ".html") io/file))

(def executors {"md" [md/execute md/status]})

(defn- guarantee-file-structure [f]
  (-> f .getParentFile .mkdirs))

(defn run
  "Execute `file`. If `out-file` is specified, results will be written there.
   Otherwise they will be written to `file` replacing the extension with
   `.html`. AKA: `README.md -> README.html`"
  ([file]
   (let [in-file (io/file file)
         present? (.exists in-file)
         out-file (when present? (rename-to-html in-file))]
     (if present?
       (run in-file out-file)
       (throw (FileNotFoundException.
                (str "File not found in local dir or on classpath: " file))))))
  ([file out-file]
   (let [in-file (io/file file)
         out-file (doto (io/file out-file) guarantee-file-structure)
         [executor-fn status-fn] (-> in-file file-type executors)]
     (if-not out-file
       (run in-file)
       (if (and executor-fn status-fn)
         (with-open [fout (FileOutputStream. out-file)]
           (:stats (execute executor-fn status-fn {} in-file fout)))
         (throw (IllegalArgumentException.
                  (format "Unknown file type for '%s'. Supported extensions: %s"
                          in-file (->> executors keys (str/join ", "))))))))))

;; ## Utilities

;; ### Null-safe parsers. If null, just return null, else parse appropriately
(defmacro defparser [n f] `(defn ~n [s#] (when s# (~f s#))))

(defparser parse-short Short/parseShort)
(defparser parse-int Integer/parseInt)
(defparser parse-long Long/parseLong)
(defparser parse-float Float/parseFloat)
(defparser parse-double Double/parseDouble)
(defparser parse-bool Boolean/parseBoolean)
(defparser parse-uuid UUID/fromString)

(def ^:private date-formatter (DateTimeFormatter/ISO_LOCAL_DATE))
(defn parse-date* [s] (LocalDate/parse s date-formatter))

(defparser parse-date parse-date*)

(defn ->out
  "Converts a map into something you can use for assertions. Basically
  takes every keyword at the root that doesn't have a question mark
  at the end and adds a question mark to it."
  [m]
  (when m
    (reduce-kv (fn [m k v]
                 (let [kn (name k)
                       n (if (str/ends-with? kn "?") kn (str kn "?"))
                       k (keyword n)]
                   (assoc m k v)))
               {}
               m)))

;; ## Table type utilities

(defn stringify-values
  "Given a map `m`, cause all its non-nil values to become strings. If any
  given key also is present in `stringifers`, then invoke the function
  referred to by that key on the value instead. That function must also
  emit a string.

  If `m` is not a map, it is simply returned untouched."
  [stringifiers m]
  (if (map? m)
    (reduce-kv (fn [m k v]
                 (let [str-fn (k stringifiers)]
                   (assoc m
                     k
                     (if str-fn
                       (str-fn v)
                       (when v (str v))))))
               {}
               m)
    m))

(defmacro deftfn
  "Define a table processing function or helper. The structure of the `args`
  differs depending on the contexts in which it is intended to be used.

  If you give `args` as a map instead of a vector, then your function is assumed
  to take a single map. You can describe how that map's values get coerced upon
  invocation by declaring a `[symbol fn]` pair inside a vector bound to `:in`.
  The keyword form of the `symbol` is the key into the given map whereby its
  value can be retrieved. The `fn` will be invoked on that value, and its result
  bound to the given symbol. If you don't want a coercion to occur, simply
  put the `symbol` into the vector directly.

  Another thing you can do with this map-based table function declaration is
  declare a series of output coercions. All table functions must return
  a map, so you can specify how the values in those maps are converted to
  Strings. By default, we will invoke `str` on any non-string, non-nil values.
  However, you can force the issue with a map associated to `:out` which
  specifies which function to invoke on each value mapped to the given key.
  If a function is given, regardless of whether there is even a value present
  in the resultant map, that function will be invoked. If there is no value
  present, then that function will be invoked on `nil`.

  Note that, if your function returns something other than a map, (for example,
  the table table fixture requires the result to be a vector of maps), the
  `:out` keyword is simply ignored.

  You can see a complete example of this macro in action in the
  `itl.example.bank` namespace.

  Otherwise, `args` should be a vector and all the normal `defn` things apply.
  "
  [n args & body]
  ;;TODO Need a real test case for :out coercions.
  (let [fd
        (if (map? args)
          (let [{:keys [in out] :or {in []}} args
                in-direct-assignments (remove vector? in)
                in-coercion-pairs (filter vector? in)
                arg-map {:keys (vec (concat in-direct-assignments
                                            (map first in-coercion-pairs)))}

                coercion-bindings
                (->> (map (fn [[b f]] (when-not (= '_ f) `[~b (~f ~b)]))
                          in-coercion-pairs)
                     (remove nil?)
                     (reduce concat)
                     vec)]
            `(defn ~n [~arg-map]
               (stringify-values ~out (let ~coercion-bindings ~@body))))
          `(defn ~n ~args ~@body))]
    `(do ~fd (swap! table-types assoc (str '~n) ~n))))

(defn exec-file
  "Run a file on the classpath through the asciidoc parser/generator"
  [indir outdir file]
  (pc/attempt println "execute file" file)
  (let [in-file (io/file indir file)
        out-filename (io/file outdir (rename-to-html in-file))]
    (run in-file out-filename)))

(def ^:private input-assignment #"^:(.*)$")

(defn cell-value [current-state v]
  (when v
    (let [[_ n] (->> v str (re-matches input-assignment))]
      (if n ((keyword n) current-state) v))))

(def ^:private output-assignment #"^=:(.*)$")

(defn printable [v]
  (if v (format "'%s'" v) "nil"))

(defn assert-value
  ([current-state expected actual]
   (let [expected (cell-value current-state expected)
         [_ n] (re-matches output-assignment (str expected))]
     (if (and current-state n)
       [(assoc current-state (keyword n) actual)
        (str "{:" n " " actual "}")]
       [current-state
        (cond
          (or (nil? expected) (and (string? expected) (str/blank? expected)))
          actual

          (= expected actual) (pass expected)

          :else
          (fail (format "Expected %s but got %s"
                        (printable expected) (printable actual))))])))
  ([expected actual]
   (second (assert-value nil expected actual))))

(defn assert-cell [r n expected actual]
  (assoc r n (assert-value expected actual)))

(defn handle-execute-page-row [indir outdir
                               {:strs [File Pass Fail Result Exception] :as r}]
  (try
    (let [expected-pass (parse-int Pass)
          expected-fail (parse-int Fail)
          expected-exception (parse-int Exception)

          {:keys [pass fail exception]}
          (exec-file indir outdir File)]
      (-> r
          (assert-cell "Pass" expected-pass pass)
          (assert-cell "Fail" expected-fail fail)
          (assert-cell "Exception" expected-exception exception)
          (assoc "Result" (format "[output](%s)" Result))))
    (catch Throwable t
      (.printStackTrace t)
      (assoc r "File" (exception t)))))

; See <https://clojure.org/reference/reader#_literals>
(defmulti safe-keyword type)

(defmethod safe-keyword Keyword [k] k)

(defmethod safe-keyword String [s]
  (-> s
      str/lower-case
      (str/replace #"[^a-z0-9?*!_'<>=]" "-")
      (str/replace #"^([0-9]+)" "-$1")
      keyword))

(defn resolve-values [current-state m & [key-fn]]
  (reduce-kv (fn [m k v] (assoc m ((or key-fn identity) k)
                                  (cell-value current-state v)))
             {}
             m))

(defn -h->k [ks] (reduce (fn [m h] (assoc m h (safe-keyword h))) {} ks))
(def h->k (memoize -h->k))

(defn process-decision-row [current-state f f-args row]
  (let [headers-to-keys (h->k (keys row))

        input-row
        (reduce-kv (fn [m k v]
                     (assoc m
                       (headers-to-keys k)
                       (cell-value current-state v))) {} row)

        result (or (f (-> current-state
                          (merge input-row)

                          ;;TODO Add test coverage for this
                          (assoc :f-args
                                 (resolve-values current-state f-args))))
                   (constantly nil))]
    (reduce (fn [[current-state r] [h v]]
              (let [[current-state result]
                    (if (str/ends-with? h "?")
                      (let [actual (-> h headers-to-keys result)]
                        (assert-value current-state v actual))
                      (let [new-value (cell-value current-state v)]
                        [current-state
                         (if (= v new-value)
                           v
                           (str v " -> " new-value))]))
                    new-row (assoc r h result)]
                [current-state new-row]))
            [current-state row]
            row)))

(defn process-query-row [[expected-row actual-row]]
  (let [headers-to-keys (h->k (keys expected-row))]
    (reduce (fn [r [h v]]
              (assoc r
                h
                (let [actual (-> h headers-to-keys actual-row)]
                  (if-not (str/blank? v) (assert-value v actual) actual))))
            {}
            expected-row)))

(defmacro with-helper-fn [args b & body]
  `(let [helper-fn# (~args "1")
         ~b (@table-types helper-fn#)]
     (if ~b
       (do ~@body)
       (throw (ex-info (format "helper function not found: '%s'" helper-fn#)
                       {:args ~args})))))

(defmacro with-parallel-support [args b & body]
  `(let [~b (if (-> "parallel" ~args Boolean/parseBoolean)
              (partial cp/pmap max-table-threads) map)]
     ~@body))

(defn remove-nil-values [m]
  (reduce-kv (fn [m k v] (if v (assoc m k v) m)) {} m))

(defn decision-table [args {:keys [rows] :as t} process-row-fn]
  (with-helper-fn args f
    (with-parallel-support args mfn
      (let [answers (mfn (process-row-fn f) rows)
            new-rows (map second answers)
            new-state (->> answers (map first) (reduce merge))]
        [new-state (assoc t :rows new-rows)]))))

;; ## Table types

(deftfn usens [current-state _ {:keys [rows] :as t}]
  [current-state
   (assoc t
     :rows
     (map
       (fn [{:strs [ns] :as r}]
         (let [answer (wt/use-ns ns)]
           (assoc r
             "ns"
             (cond
               (instance? Throwable answer) (exception answer)

               (nil? answer)
               (exception (ex-info (str "Namespace not found:" ns) {}))
               :else ns))))
       rows))])

(deftfn execute-pages [current-state {:strs [indir outdir] :as args}
                       {:keys [rows] :as t}]
  (with-parallel-support args mfn
    [current-state
     (assoc t
       :rows
       (mfn (partial handle-execute-page-row indir outdir) rows))]))

;; ### Table-Table
(deftfn table [current-state args {:keys [raw-rows]}]
  (with-helper-fn args f [current-state {:raw-rows (f raw-rows)}]))

;; ### Query Table

(deftfn query [current-state args {:keys [rows] :as t}]
  (with-helper-fn args f
    (let [query-result (f (walk/keywordize-keys args))
          new-rows (map process-query-row
                        (partition 2 (interleave rows query-result)))]
      [current-state (assoc t :rows new-rows)])))

;; ### Decision Tables

;; Standard
(deftfn dt [current-state args t]
  (decision-table args t
                  (fn [f] (partial process-decision-row current-state f args))))

;; Baseline
(deftfn bdt [current-state args {:keys [rows] :as t}]
  (let [base-row (first rows)
        default-values (fn [row] (merge base-row (remove-nil-values row)))]
    (decision-table
      args t
      (fn [f] (comp
                (partial process-decision-row current-state f args)
                default-values)))))

;; ### Function Table

(defn execute-function
  [current-state common-params expected f-name f args out-name]
  (let [out-k (keyword out-name)
        original-args (->> args
                           (apply hash-map)
                           walk/keywordize-keys)
        args (resolve-values current-state original-args)
        f-result (f (merge common-params args))
        actual (out-k f-result)
        current-state (merge current-state f-result)
        [current-state v] (assert-value current-state expected actual)
        args (reduce-kv (fn [m k v]
                          (let [original (original-args k)]
                            (assoc m (name k)
                                     (if (= v original)
                                       v
                                       (format "%s => %s" original v)))))
                        {}
                        args)]
    [current-state (concat [(or v actual) f-name]
                           (reduce concat args)
                           [out-name])]))

;;todo We need to test we can run a function without looking at its results
;;todo (i.e. solely for side effects)
(defn handle-function-row [common-params
                           {:keys [current-state max-length] :as reduction}
                           [expected & rest-of-row]]
  (let [others (remove nil? rest-of-row)
        out-name (last others)
        [f-name & args] (if (> (count others) 1) (butlast others) others)
        f (@table-types f-name)]
    (try
      (cond
        f
        (let [[current-state new-row]
              (execute-function current-state
                                common-params
                                expected
                                f-name
                                f
                                args
                                out-name)]
          (-> reduction
              (update :current-state merge current-state)
              (update :rows conj new-row)
              (assoc :max-length (max max-length (count new-row)))))

        f-name
        (throw (ex-info "Function not found" {:f-name f-name}))

        :else
        (throw (ex-info "No function specified" {})))
      (catch Throwable t
        (pc/attempt pc/print-exception t)
        (update reduction :rows conj (concat [expected (exception t)] args))))))

(defn balance-row [desired-length row]
  (let [rowlen (count row)
        cells-to-add (- desired-length rowlen)
        blanks (repeat cells-to-add "")]
    (if (<= 2 rowlen)
      (concat row blanks)
      (let [left-side (butlast row), right-side (last row)]
        (concat left-side blanks [right-side])))))

(deftfn function [current-state args {:keys [raw-rows] :as t}]
  (let [common-params (resolve-values current-state args safe-keyword)

        {:keys [current-state rows max-length]}
        (reduce (partial handle-function-row common-params)
                {:current-state current-state
                 :rows []
                 :max-length 0}
                raw-rows)
        rows (map (partial balance-row max-length) rows)]
    [current-state (-> t (dissoc :rows) (assoc :raw-rows rows))]))

;; ## Run tests at the REPL

(comment

  @table-types

  (pc/with-printing-enabled true (run "README.md" "docs/index.html"))

  (pc/with-printing-enabled true (run "tests/tables.md" "docs/tables.html"))

  (pc/with-printing-enabled true (run "tests/table-table.md"
                                      "docs/tables/table-table.html"))

  (pc/with-printing-enabled true (run "tests/decision-table.md"
                                      "docs/tables/decision-table.html"))

  (pc/with-printing-enabled true
                            (run "tests/baseline-decision-table.md"
                                 "docs/tables/baseline-decision-table.html"))

  (pc/with-printing-enabled true (run "tests/function-table.md"
                                      "docs/tables/function-table.html"))

  (pc/with-printing-enabled true
                            (run "tests/query-table.md"
                                 "docs/tables/query-table.html"))

  *e

  )
