;
; Copyright (c) 2022 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.worker-threads
  (:require [clojure.core.async :as async]))

;; # Worker Threads
;;
;; This namespace is about fixed threads that should be doing the work
;; of the library that can't be multithreaded. They are separated from other
;; namespaces so we can freely reload those namespaces without worrying about
;; collecting more and more threads.

;; ## Thread for the `use` procedure

(def ^:private use-channel (async/chan))
(def ^:private used-namespaces (atom #{}))

(defn use-namespace [ns-to-use]
  (let [ns-to-use
        (when ns-to-use (->> ns-to-use (str "'") read-string eval))]
    (when ns-to-use
      (use ns-to-use :reload)
      (swap! used-namespaces conj ns-to-use)
      :success)))

(doto
  (Thread.
    (fn []
      (try
        (while true
          (let [[ns promise] (async/<!! use-channel)]
            (deliver promise (try
                               (if-not (@used-namespaces ns)
                                 (use-namespace ns)
                                 :already-used)
                               (catch Throwable t t)))))
        (catch Throwable t
          (.printStackTrace t)
          (System/exit 1)))))
  (.setDaemon true)
  (.start))

(defn use-ns [ns-to-use]
  (let [promise (promise)]
    (async/>!! use-channel [ns-to-use promise])
    @promise))
