;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.cli
  (:require [clojure.tools.cli :refer [parse-opts]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.pprint :as pprint]
            [itl.core :as itl]
            [itl.markdown :as md]
            [itl.print-control :as pc]
            [itl.tags :as tags])
  (:gen-class))

(defn- run-file [in out]
  (print (str (.getName in) "..."))
  (flush)

  (if-let [{:keys [fail exception] :as stats} (itl/run in out)]
    (do
      (pprint/print-table [stats])
      (System/exit (if (or (pos? fail) (pos? exception)) 1 0)))
    (do (println "Could not find: " in) (System/exit 1))))

(defn- parse-file [f] (when f (io/file f)))

(defn add-tag [opts _ tag] (update opts :tags conj tag))

(def cli-options
  [["-o" "--out OUTFILE"
    "File to write results to. Defaults to {FILE} =~ s/.adoc/.html/"
    :parse-fn parse-file]

   ["-q" "--quiet" "Don't print status information to *out*"]
   ["-t" "--tag TAG"
    (str "Run only page sections that are either tagged with 'global' or the "
         "given tag. Can specify multiple times")
    :id :tags
    :default #{}
    :assoc-fn add-tag]
   ["-n" "--num-threads THREADS"
    "Maximum number of table-processing threads"
    :default (+ 2 (.. Runtime getRuntime availableProcessors))
    :parse-fn itl/parse-int]
   [nil "--css CSSFILE" "Specify a CSS file for the Markdown renderer"]
   ["-h" "--help"]])

(defn- print-help [summary & [errors]]
  (when errors (println (str/join "\n" errors)))
  (printf (str "%n"
               "USAGE%n"
               "-----%n"
               "lein cli <FILE> <OPTIONS> %n"
               "lein cli -- -h|--help%n"
               "%n"
               "Options:%n"
               "%s%n")
          summary))

(defn -main
  "Execute `lein cli -- -h` for help"
  [& [file & args]]
  (let [{:keys [errors options summary] {:keys [help quiet]} :options}
        (parse-opts args cli-options)

        file (parse-file file)]
    (cond
      (or help (and file (#{"-h" "--help"} (str/lower-case file))))
      (print-help summary)

      (seq errors) (print-help summary errors)
      (nil? file) (print-help summary ["FILE not specified"])
      (not (.exists file))
      (print-help summary [(format "'%s' not found" (.getAbsolutePath file))])

      :default
      (let [{:keys [out tags css num-threads]} options
            file (parse-file file)]
        (itl/with-max-threads num-threads
          (md/with-css css
            (tags/with-allowed-tags tags
              (pc/with-printing-enabled (not quiet) (run-file file out)))))))))
