;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.example.bowling
  (:require [itl.core :refer [deftfn parse-int assert-value]]))

(defn roll->int [v & [pv]]
  (cond (= "X" v) 10
        (= "/" v) (- 10 pv)
        :else v))

(defn score-frame [prior-score frames fi]
  (let [[roll01 roll02 roll03] (nth frames fi)
        [roll11 roll12] (when (< fi 9) (nth frames (inc fi)))
        [roll21] (when (< fi 8) (nth frames (+ fi 2)))]
    (+ prior-score
       (cond
         ;;10th frame strike
         (and (= 9 fi) (= "X" roll01))
         (+ 10 (roll->int roll02) (roll->int roll03 roll02))

         ;;strike
         (= "X" roll01)
         (+ 10
            (roll->int roll11)
            (or (roll->int roll12 roll11) (roll->int roll21)))

         ;;spare
         (= "/" roll02)
         (+ 10 (roll->int (or roll03 roll11)))

         :else (+ roll01 roll02)))))

(defn roll-value [roll]
  (cond (= "-" roll) 0
        (and roll (re-find #"[1-9]" roll)) (parse-int roll)
        :else roll))

(defn translate-rolls [rolls]
  (let [first-9 (->> rolls (take 18) (partition-all 2) vec)
        last-frame (take-last 3 rolls)]
    (map (fn [frame] (map roll-value frame)) (conj first-9 last-frame))))

(def table-cell->score-index
  {1 0, 3 1, 5 2, 7 3, 9 4, 11 5, 13 6, 15 7, 17 8, 20 9})

(defn assert-frame [actual-scores [expected fi]]
  (let [si (table-cell->score-index fi)]
    (if si (assert-value expected (str (nth actual-scores si))) expected)))

(defn add-score [frames scores fi]
  (let [prior-score (or (last scores) 0)]
    (conj scores (score-frame prior-score frames fi))))

(deftfn bowling [[rolls scores]]
  (let [frames (translate-rolls rolls)
        actual-scores (reduce (partial add-score frames) [] (range 10))]
    [rolls (map (partial assert-frame actual-scores)
                (partition 2 (interleave scores (range))))]))
