;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.core
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [itl.asciidoc :as ad]
            [itl.markdown :as md]
            [itl.print-control :as pc]
            [itl.tags :as tags]
            [selmer.parser :as selmer])
  (:import (java.io FileOutputStream FileNotFoundException OutputStreamWriter)
           (java.time LocalDateTime)
           (java.time.format DateTimeFormatter)
           (org.apache.commons.io FilenameUtils)
           (java.util UUID))
  (:gen-class))

;; # Integration Test Library

;; These functions are bound in `execute`. They mark an assertion
;; as "passed" or "failed."
(def ^:dynamic pass)
(def ^:dynamic fail)
(def ^:dynamic exception)
(def ^:dynamic note)

(def ^:private ops (atom {}))
(def ^:private procs (atom {}))
(def ^:private generators (atom {}))

;; ## Fixture definition

(defn- add-to [a n f]
  (swap! a assoc n f))

;; Meant to be used primarily by `defop` and `defproc` and `defgenerator`
(def add-op! (partial add-to ops))
(def add-proc! (partial add-to procs))
(defn add-generator! [n f]
  (if (re-matches #"^[a-zA-Z\-]+$" n)
    (add-to generators n f)
    (throw (ex-info "Invalid generator name" {:given n}))))

(defmacro defop
  "Defines an operation that can be executed in your test. Accepts a
   name (`n`) that will be referred to in brackets in the document. The body
   will be of a function that accepts a single parameter `page-state`.
   That parameter will be filled in with the current state of the running tests.

   An example op reference in your doc could be: `op::[do something]`

   Your operation *must* return whatever parts of `page-state` your operation
   wishes to pass on to future operations and/or assertions."
  [^String n [page-state] & body]
  `(add-op! ~n (fn [~page-state] ~@body)))

(defmacro defproc
  "Defines a procedure that can be executed in your test. Accepts a
   name `n` and a set of bindings `page-state` and `aargs`. Just as in `defop`,
   the state will be bound to `page-state` and must be passed back out of your
   proc with whatever changes need to be made. `a` will be bound to the
   a map of arguments. Check out
   <http://www.methods.co.nz/asciidoc/chunked/ch21.html> for more info.

   Example proc reference in your doc: `exec:say[msg=hello]`"
  [^String n [page-state args] & body]
  `(add-proc! ~n (fn [~page-state ~args] ~@body)))

(defmacro deftfn
  "Defines a table processing function. Accepts a name `n` that will be
   referred to in brackets in the table's legend. The body will be of a
   function that accepts two parameters: `page-state`, the current state of the
   running tests, and `table-data`, a structure containing the parsed table
   structure.

   Your operation must return a vector containing an updated
   `[page-state table-data]`. The value of `page-state` will be sent to the next
   operation, whereas `table-data` will be re-constituted into a table and
   inserted into the document content."
  [^String n [page-state table-data] & body]
  `(add-op! ~n (fn [~page-state ~table-data] ~@body)))

(defmacro deftafn
  "Just like `deftfn`, except this table also accepts arguments passed in
   from the source document as `args`"
  [^String n [page-state table-data args] & body]
  `(add-op! ~n (fn [~page-state ~table-data ~args] ~@body)))

;; ## Variables

(def ^:private generator-expectation
  "< or /[a-zA-Z\\-]+>/ followed by a name of a keyword")

(defn vars
  "Retrieve a list of all vars in the given `page-state`"
  [page-state] (:_vars page-state))

(defn add-var
  "Add a new variable to the page state for use in column table assignments and
  expectations."
  [page-state k v]
  (update page-state :_vars assoc k v))

(defmacro defgenerator
  "Defines a function that generates a value, which will eventually be bound to
  a variable. The name may only contain letters and dashes. The function must
  not take any parameters. See `generated-var`"
  [^String n & body]
  `(add-generator! ~n (fn [] ~@body)))

(defn generated-var
  "If the var-code looks like `uuid>x` we will look for a generator bound to
  \"uuid\" and store its result in `page-state` at `[:generated :x]`. It will
  then return `[page-state assigned-var new-value]`.
  Otherwise, `[page-state nil var-code]` will be returned.

  See `defgenerator`"
  [page-state var-code]
  (if-let [s (and var-code (re-matches #"^([a-zA-Z\-]+)>(.*)$" var-code))]
    (let [[_ generator-name var-name] s
          k (keyword var-name)
          generator (@generators generator-name)
          new-value (if generator
                      (generator)
                      (throw (ex-info "Invalid generator"
                                      {:given generator-name})))]
      [(add-var page-state k new-value) k new-value])
    [page-state nil var-code]))

(defgenerator "uuid" (str (UUID/randomUUID)))

;; ## Document execution

(defn execute
  "Parse a document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out`. The `initial-state` is a map of bindings that the document can use."
  [execute-fn status-fn initial-state in out]
  (let [current-state (atom initial-state)
        local-stats (atom {:pass 0 :fail 0 :exception 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (status-fn given-status content))
        start-time (System/currentTimeMillis)]
    (binding [pass (partial local-status :pass)
              fail (partial local-status :fail)
              exception (partial local-status :exception)
              note (partial status-fn :note)]
      (with-open [reader (io/reader in)
                  writer (io/writer out)]
        (execute-fn reader writer current-state ops procs pass fail exception))
      (assoc @current-state
        :stats
        (assoc @local-stats
          :elapsed-time (- (System/currentTimeMillis) start-time)
          :run-date (.format (LocalDateTime/now)
                             (DateTimeFormatter/ofPattern
                               "yyyy-MM-dd HH:mm")))))))

(defn- file-type [f]
  (-> f .getName FilenameUtils/getExtension))

(defn- rename-to-html [f]
  (-> f .getName FilenameUtils/getBaseName (str ".html")))

(defn- in-file [file]
  (let [file-resource (when (string? file) (io/resource file))]
    (io/file (or file-resource file))))

(def executors {"adoc" [ad/execute ad/status]
                "md" [md/execute md/status]})

(defn run
  "Execute `file`. If `out-file` is specified, results will be written there.
   Otherwise they will be written to `file` replacing the extension with
   `.html`. AKA: `README.adoc -> README.html`"
  ([file]
   (let [in-file (in-file file)
         present? (.exists in-file)
         new-name (when present? (rename-to-html in-file))]
     (if present?
       (run in-file new-name)
       (throw (FileNotFoundException.
                (str "File not found in local dir or on classpath: " file))))))
  ([file out-file]
   (let [[executor-fn status-fn] (-> file file-type executors)]
     (if (and executor-fn status-fn)
       (run executor-fn status-fn file out-file)
       (throw (IllegalArgumentException.
                (format "Unknown file type for '%s'. Supported extensions: %s"
                        file (->> executors keys (str/join ", "))))))))
  ([execute-fn status-fn file out-file]
   (if-not out-file
     (run file)
     (with-open [fout (FileOutputStream. out-file)]
       (:stats (execute execute-fn status-fn {} file fout))))))

;; ## Utilities

;; ### Column Tables

(defn- replace-vars [page-state row k]
  (some-> k row (selmer/render (vars page-state))))

(defn- produce-replacement [page-state k v replacement-dir replacement-msg]
  (-> page-state
      (assoc k v)
      (assoc-in [:_replacements k] [replacement-dir replacement-msg])))

(defn- generate-vars [page-state row cell-title page-state-key]
  (let [original-value (row cell-title)
        replaced-value (replace-vars page-state row cell-title)
        replacement-happened? (not= original-value replaced-value)]
    (if replacement-happened?
      (produce-replacement page-state page-state-key replaced-value
                           "<" original-value)
      (let [[page-state k new-value] (generated-var page-state original-value)]
        (if k
          (produce-replacement page-state page-state-key new-value ">" k)
          (assoc page-state page-state-key original-value))))))

(defn- assert-cell [page-state e row k v]
  (let [expected (replace-vars page-state row k)
        actual (page-state v)]
    (assoc row k
               (if e
                 (exception e)
                 (if (= (str expected) (str actual))
                   (pass expected)
                   (fail (format "%s (got: '%s')" expected actual)))))))

(defn- reassign-cell [{:keys [_replacements] :as page-state} row k v]
  (let [new-value (or (v page-state) "")]
    (assoc row k (if-let [[dir repl] (and _replacements (_replacements v))]
                   (str new-value " " (note (str dir " " repl)))
                   new-value))))

(defn- handle-row [page-state assign exec asserts row]
  (let [page-state (reduce-kv (fn [page-state k v]
                                (when-not (associative? page-state)
                                  (throw
                                    (ex-info "page-state is not associative!"
                                             {:page-state page-state
                                              :k k
                                              :v v})))
                                (generate-vars page-state row k v))
                              page-state
                              assign)
        [page-state emsg] (try
                            [(exec page-state) nil]
                            (catch Throwable t
                              (pc/print-exception t)
                              [page-state (.getMessage t)]))
        row (reduce-kv (partial assert-cell page-state emsg) row asserts)]
    [(dissoc page-state :_replacements)
     (reduce-kv (partial reassign-cell page-state) row assign)]))

(defn- add-row-result [assign exec asserts [page-state rows] row]
  (let [[page-state row] (handle-row page-state assign exec asserts row)]
    [page-state (conj rows row)]))

(defn sync-column-table [page-state rows assign exec asserts]
  (reduce (partial add-row-result assign exec asserts) [page-state []] rows))

(defn async-column-table [page-state rows assign exec asserts]
  ;; The `doall` is because we don't want the output writers to close
  ;; before the results of the table execution are fully realized
  (let [rows (doall
               (pmap (comp second
                           (partial handle-row page-state assign exec asserts))
                     rows))]
    [page-state rows]))

(defn column-table
  "Take a table of data and assert a bunch of stuff on it. The
  `assign` map should contain mappings from labels in the table to
  keys they should be mapped to in the `page-state` collection. The `exec`
  should be a single function to be executed for each row. It will receive the
  current state after having all the assignments made. It should return
  the state to be passed to the assertions, and eventually, the next row.
  Finally, the `assert` should contain yet another map of labels to keys in the
  state that should match.

  If any content in the parsed document is either assigned or asserted against
  and contains `{{x}}` syntax, the value of `:x` will be pulled from
  page variables and inserted in its place. See `add-var` for more information.

  If you pass `:parallel? true` at the end, each row will be executed in
  parallel, and its page state will not be passed to the next row. The
  original state that was passed into this function will be returned out."
  [page-state {:keys [rows]} {:keys [assign exec asserts]} &
   {:keys [parallel?]}]
  (when (empty? assign) (println "WARNING: No assignments bound to :assign"))
  (when (empty? asserts) (println "WARNING: No assertions bound to :asserts"))
  (let [[page-state rows]
        (if parallel? (async-column-table page-state rows assign exec asserts)
                      (sync-column-table page-state rows assign exec asserts))]
    [page-state {:rows rows}]))

;; ### Generative tables

(defn generate-cell [declarations generate asserts label]
  (let [[gk gf] (generate label)
        [ak af] (asserts label)
        declaration (declarations label)
        [nk nv] (if gk [:value {gk (gf declaration)}]
                       (if ak
                         [:assert {ak (af declaration)}]
                         (throw (ex-info "Missing generator/assertion"
                                         {:label label}))))]
    {nk nv}))

(defn extract-by-key [cells k]
  (->> cells (filter k) (map k) (apply merge)))

(defn apply-generated-assertion [asserts page-state emsg k]
  (if emsg
    {k emsg}
    (let [assertion (asserts k)
          actual (page-state k)]
      {k (assertion actual)})))

(defn headers-to-keys [m]
  (apply merge (map (fn [k] {k (-> m (get k) first)}) (keys m))))

(defn assemble-generated-cell
  [declarations headers-to-keys is-assert? page-state emsg original label]
  (let [declaration (declarations label)
        assert (is-assert? label)
        k (headers-to-keys label)
        v (page-state k)]
    {label
     (if assert
       (if v
         (if emsg
           (exception emsg)
           (pass (str (original k) " was " declaration)))
         (fail (str (original k) " was not " declaration)))
       (str v))}))

(defn generate-row [page-state declarations generate exec asserts _]
  (let [labels (keys declarations)
        cells
        (map (partial generate-cell declarations generate asserts) labels)

        values (extract-by-key cells :value)
        generated-assertions (extract-by-key cells :assert)
        page-state (merge page-state values)
        [page-state emsg] (try
                            [(exec page-state) nil]
                            (catch Throwable t
                              (pc/print-exception t)
                              [page-state (.getMessage t)]))

        asserted
        (->> (keys generated-assertions)
             (map (partial apply-generated-assertion
                           generated-assertions
                           page-state
                           emsg))
             (apply merge))
        headers-to-keys (merge (headers-to-keys generate)
                               (headers-to-keys asserts))
        original page-state
        page-state (merge page-state asserted)]
    (->> labels
         (map
           (partial assemble-generated-cell declarations headers-to-keys
                    (into (hash-set) (keys asserts)) page-state emsg original))
         (apply merge))))

(defn generative-table
  "Take a table with two rows and generate a bunch more rows suitable for
  for assertion. The first row is the names of the columns of each of the newly
  generated rows. The second row is the specifications by which we generate or
  assert against values."
  [page-state {[declarations] :rows} {:keys [generate exec asserts]} num-rows]
  (let [new-rows
        (doall (map (partial generate-row
                             page-state
                             declarations
                             generate
                             exec
                             asserts)
                    (range num-rows)))]
    [page-state {:rows new-rows}]))

;; ### Generation Example functions

(defn- -integer-generator [v]
  (let [[l r] (->> #"\.\.\." (str/split v) (map #(Integer/parseInt %)))
        diff (- r l)]
    (when (<= diff 0)
      (throw
        (ex-info
          "The first number of an integer range must be smaller than the second"
          {:given v})))
    #(+ l (rand-int diff))))

(def ^:private integer-generator (memoize -integer-generator))

(defn random-integer
  "Parses a string that contains two integers separate by 3 dots
  (e.g. \"1...50\") and produces a random number within that range, inclusive of
  the first integer, and exclusive of the second."
  [v]
  ((integer-generator v)))

(defn random-ssn [_]
  (let [first-part (random-integer "100...1000")
        second-part (random-integer "10...100")
        third-part (random-integer "1000...10000")]
    (format "%s-%s-%s" first-part second-part third-part)))

(defn integer-comparison
  "Parses a string that contains an operator and an operand to be applied
  to two values. Produces a function which takes two values and produces
  the string value \"true\" if the operand applied successfully, else
  \"false\"."
  [v]
  (let [[op operand] (str/split v #" ")
        opfn (-> op read-string eval)
        operand (Integer/parseInt operand)]
    #(opfn % operand)))

(defn bit-bob-comparison
  "Parses a string that contains a regular expression to be applied to a value"
  [_]
  (let [p (re-pattern "(Bit|Bob)\\d{1,2}")]
    (partial re-find p)))

(defn convert-missing [v]
  (if (= "MISSING" v) :missing v))

(defn- -string-loop [v]
  (let [strs (map (comp convert-missing str/trim)
                  (str/split v #","))
        str-queue (atom (vec strs))]
    #(let [fv (first @str-queue)]
       (reset! str-queue (concat (rest @str-queue) [fv]))
       fv)))

(def ^:private string-loop (memoize -string-loop))

(defn loop-of-strings [v]
  ((string-loop v)))

(defn basic-equivalence [v] (partial = v))

(comment
  (loop-of-strings "hello,there, general kenobi          ")
  (loop-of-strings "     , MISSING, disturbance in the force     ")
  ((bit-bob-comparison "") "Bob12")
  ((basic-equivalence "thing") "thing")
  )

;; ## Globally accessible fixtures

(defn exec-file
  "Run a file on the classpath through the asciidoc parser/generator"
  [indir outdir {:keys [file] :as page-state}]
  (pc/attempt println "execute file" page-state)
  (let [in-file (in-file (io/file indir file))
        out-filename (io/file outdir (rename-to-html in-file))]
    (merge page-state (run in-file out-filename))))

(defn capture-output [logfile f]
  (with-open [out-file (FileOutputStream. (io/file logfile))
              new-out (OutputStreamWriter. out-file)]
    (binding [*out* new-out] (f))))

(deftafn "execute example files" [page-state table-data
                                  {:strs [indir outdir logfile parallel]}]
  (let [f (fn []
            (column-table page-state table-data
                          {:assign {"File" :file}
                           :exec (partial exec-file indir outdir)
                           :asserts {"Pass" :pass "Fail" :fail
                                     "Exception" :exception}}

                          :parallel?
                          (when parallel (Boolean/parseBoolean parallel))))]
    (if logfile (capture-output logfile f) (f))))

(defproc "use" [page-state {:strs [ns]}]
  (let [ns-to-use (when ns (->> ns (str "'") read-string eval))]
    (if ns-to-use (use ns-to-use :reload)
                  (throw (ex-info "Namespace not found" {:ns ns}))))
  page-state)

;; ## Run tests at the REPL

(comment
  (run "README.adoc")
  ;=>
  ;{:pass 6,
  ; :fail 0,
  ; :exception 0,
  ; :elapsed-time 1471,
  ; :run-date "2019-07-24 07:29"}

  (run "tests/simple-example.adoc")
  ;=>
  ;{:pass 7,
  ; :fail 3,
  ; :exception 0,
  ; :elapsed-time 125,
  ; :run-date "2019-07-24 07:28"}

  (run "tests/complete-example.adoc")
  ;=>
  ;{:pass 65,
  ; :fail 2,
  ; :exception 5,
  ; :elapsed-time 515,
  ; :run-date "2019-07-24 07:29"}

  (run "tests/simple-example.md")
  ;=>
  ;{:pass 7,
  ; :fail 3,
  ; :exception 0,
  ; :elapsed-time 57,
  ; :run-date "2019-07-27 19:16"}

  (run "tests/complete-example.md")
  ;=>
  ;{:pass 13,
  ; :fail 2,
  ; :exception 4,
  ; :elapsed-time 87,
  ; :run-date "2019-07-27 19:17"}

  (tags/with-allowed-tags #{"uuid" "checkbook"}
    (run "tests/complete-example.md"))
  ;=>
  ;{:pass 12,
  ; :fail 1,
  ; :exception 0,
  ; :elapsed-time 97,
  ; :run-date "2019-08-02 08:59"}

  (run "tests/bad-example.txt")
  ; Throws IllegalArgumentException
  )