;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.markdown
  (:require [clojure.pprint :as pprint]
            [clojure.string :as str]
            [itl.print-control :refer [attempt]]
            [itl.print-control :as pc])
  (:import (com.vladsch.flexmark.ast Paragraph)
           (com.vladsch.flexmark.ext.tables TablesExtension
                                            TableBlock
                                            TableHead
                                            TableBody)
           (com.vladsch.flexmark.html HtmlRenderer)
           (com.vladsch.flexmark.parser Parser)
           (com.vladsch.flexmark.util.data MutableDataSet)
           (java.io PrintWriter)
           (java.time ZonedDateTime)))

(def colors {:pass "green", :fail "red", :exception "yellow", :note "silver"})
(def status-handlers {:exception #(str "E[" % "]")})

(defn- status-update [& content]
  (let [value (str/join content)]
    (if (str/blank? value) value (str "**" value "**"))))

(defn- colored [color & content]
  (str "<span class=\"" color "\">" (str/join content) "</span>"))

(defn status
  "Mark some content as passed or failed"
  [given-status content]
  (let [color (colors given-status)
        handler (or (status-handlers given-status) identity)]
    (colored color (status-update (handler content)))))

(defn- arg [arg i]
  (let [[k v] (str/split arg #"=")]
    (if v {k v} {i k})))

(defn- add-args [l a]
  (concat l (map str/trim (str/split a #","))))

(defn make-args [args]
  (apply merge (map arg
                    (reduce add-args [] args)
                    (map str (rest (range))))))

(defn- children [n] (when n (iterator-seq (.getChildIterator n))))

(defn replace-node [parser current-node content]
  (.removeChildren current-node)
  (doseq [n (->> content str (.parse parser) children first children)]
    (.appendChild current-node n)))

(defn missing-op [parser current-node op-type op-name exception]
  (replace-node parser
                current-node
                (exception (str "Missing " op-type ": " op-name))))

(defn executed [parser current-node op-name]
  (replace-node parser current-node (status-update op-name)))

(defn exception-occurred [parser current-node op-name t exception]
  (pc/print-exception t)
  (replace-node parser
                current-node
                (exception (str "exception executing '" op-name "' -- "
                                (or (.getMessage t)
                                    (-> t .getClass .getName))))))

(defn bind [{:keys [parser current-state current-node]
             {s "1" v "2"} :args}]
  (let [k (keyword s)]
    (swap! current-state assoc k v)
    (pc/attempt printf "%s := %s%n" k v)
    (replace-node parser current-node (status-update k " := " v))))

(defn run-op [invoker o-or-p op-name nice-name op-type parser current-state
              current-node exception]
  (let [op (o-or-p op-name)]
    (if op
      (do
        (pc/attempt println op-type nice-name)
        (try
          (reset! current-state (invoker op))
          (executed parser current-node nice-name)
          (catch Throwable t
            (exception-occurred parser current-node nice-name t exception))))
      (do
        (pc/attempt println "missing " op-type op-name)
        (missing-op parser current-node op-type op-name exception)))))

(defn op [{:keys [parser current-state current-node ops exception]
           {op-name "1"} :args}]
  (run-op (fn [op] (op @current-state))
          ops
          op-name
          op-name
          "operation"
          parser
          current-state
          current-node
          exception))

(defn exec [{:keys [parser current-state current-node procs exception]
             {op-name "1" :as args} :args}]
  (let [args (dissoc args "1")]
    (run-op (fn [op] (op @current-state args))
            procs
            op-name
            (str op-name " " (pr-str args))
            "procedure"
            parser
            current-state
            current-node
            exception)))

(defn assertion [n c {:keys [parser current-state current-node pass fail]
                      {s "1" expected "2"} :args}]
  (let [k (keyword s)
        actual (k @current-state)
        matches? (c expected (str actual))
        out-status (if matches? pass fail)
        to-show (str n " be '" expected "', and was '" actual "'")]
    (pc/attempt printf
                "assertion: %s %s -> %s%n"
                k
                to-show
                (if matches? "PASS" "FAIL"))
    (replace-node parser current-node (out-status (str k " " to-show)))))

(defn- cell-text [n]
  (let [t (-> n .getText str)] (when-not (str/blank? t) t)))

(defn table [{:keys [parser current-state current-node previous-node ops
                     exception]
              {op-name "1" :as args} :args}]
  (try
    (let [opfn (ops op-name)
          args (dissoc args "1")]
      (if opfn
        (if (instance? TableBlock previous-node)
          (let [{:keys [table-head table-body]}
                (reduce (fn [m n]
                          (cond (instance? TableHead n) (assoc m :table-head n)
                                (instance? TableBody n) (assoc m :table-body n)
                                :else m))
                        {}
                        (children previous-node))

                header-cells (-> table-head children first children)
                labels (map cell-text header-cells)
                table-rows (children table-body)
                row-text (map (comp (partial map cell-text) children)
                              table-rows)
                original-rows (map (partial zipmap labels) row-text)
                opargs [@current-state {:labels labels :rows original-rows}]
                opargs (if (empty? args) opargs (conj opargs args))
                _ (do (pc/attempt println "=== INPUT ===")
                      (pc/attempt pprint/print-table original-rows)
                      (flush))
                [s {:keys [rows]}] (apply opfn opargs)
                _ (reset! current-state s)]

            (doseq [[output table-cells]
                    (zipmap rows (map children table-rows))

                    [label cell] (zipmap labels table-cells)]
              (let [new-value (output label)]
                (replace-node parser cell new-value)))

            (pc/attempt println "=== OUTPUT ===")
            (pc/attempt pprint/print-table rows)
            (executed parser current-node op-name))
          (replace-node parser
                        current-node
                        (exception "Could not find a table to operate on")))
        (missing-op parser current-node "table" op-name exception)))
    (catch Throwable t
      (exception-occurred parser current-node op-name t exception))))

(defn gentable [_]
  (pc/attempt println "gentable not yet implemented"))

(def ^:private commands
  {"should" (partial assertion "should" =)
   "shouldnot" (partial assertion "shouldnot" (complement =))
   "op" op
   "exec" exec
   "bind" bind
   "table" table
   "gentable" gentable})

(defn execute [reader writer current-state ops procs pass fail exception]
  (with-open [w (PrintWriter. writer)]
    (let [options (doto (MutableDataSet.)
                    (.set Parser/EXTENSIONS [(TablesExtension/create)]))
          parser (.build (Parser/builder options))
          renderer (.build (HtmlRenderer/builder options))
          doc (.parse parser (slurp reader))
          css (System/getenv "CSS_FILE")
          wr (fn [& s] (.println w (str/join s)))]
      (wr "<!DOCTYPE html><html><head><meta charset=\"UTF-8\">")
      (wr "<meta http-equiv=\"X-UA-Compatible\" content=\"IE=edge\">")
      (wr "<meta name=\"viewport\" content=\"width=device-width, "
          "initial-scale=1.0\">")
      (wr "<meta name=\"generator\" content=\"itl\">")
      (when css
        (wr "<style>")
        (wr (slurp css))
        (wr "</style>"))
      (wr "</head><body class=\"article\"><div id=\"content\">")
      (let [previous-node (atom nil)
            macro-expression #"(.*?)::(.+?)?\[(.*?)\]"]
        (doseq [node (iterator-seq (.getChildIterator doc))]
          (when (instance? Paragraph node)
            (let [text (-> node .getContentChars .unescape str/trim)
                  macro-matches (re-matches macro-expression text)]
              (when macro-matches
                (let [[_ cmd-name & args] (remove nil? macro-matches)
                      args (make-args args)
                      command (commands cmd-name)]
                  (if command
                    (command {:parser parser
                              :args args
                              :current-state current-state
                              :ops @ops
                              :procs @procs
                              :pass pass
                              :fail fail
                              :exception exception
                              :current-node node
                              :previous-node @previous-node})
                    (attempt println "WARNING: unknown command"
                             (str "'" cmd-name "'")))))))
          (reset! previous-node node)))
      (.render renderer doc w)
      (wr "</div>")
      (wr "<div id=\"footer\" class=\"md-footer\">")
      (wr "<div id=\"footer-text\">")
      (wr "Last updated " (ZonedDateTime/now))
      (wr "</div>")
      (wr "</div>")
      (wr "</body></html>"))))
