;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.asciidoc
  (:require [clojure.pprint :as pprint]
            [clojure.string :as str]
            [hiccup.core :refer :all]
            [itl.print-control :as pc])
  (:import (org.asciidoctor.ast Table)
           (org.asciidoctor.extension BlockMacroProcessor)
           (org.asciidoctor Asciidoctor$Factory SafeMode)))

(def colors {:pass "green", :fail "red" :exception "yellow"})
(def status-handlers {:pass identity :fail identity
                      :exception #(str "E[" % "]")})

; This is rebound in the code generated by `inline-macro`
(def ^:dynamic format-status)

(defn status-asciidoc [color content]
  (let [trimmed (str/trim content)]
    (if-not (str/blank? trimmed)
      (format "*[%s]#%s#*" color trimmed)
      trimmed)))

(defn status
  "Mark some content as passed or failed"
  [given-status content]
  (format-status (colors given-status)
                 ((status-handlers given-status) content)))

(defn respond [macro-processor parent r]
  (.createBlock macro-processor parent "paragraph" r))

(defn block-macro [exception registry n f]
  (.blockMacro
    registry
    (proxy [BlockMacroProcessor] [n]
      (process [parent target attributes]
        (let [a (into {} attributes)
              r (partial respond this parent)]
          (binding [format-status status-asciidoc]
            (try
              (f this parent target a r)
              (catch Throwable t
                (pc/print-exception t)
                (r (exception (.getMessage t)))))))))))

(defn- assertion [exception registry n op current-state pass fail]
  (block-macro exception registry n
    (fn [_ _ t a r]
      (let [expected (a "1")
            k (keyword t)
            actual (k @current-state)
            matches? (op expected (str actual))
            out-status (if matches? pass fail)
            to-show (str n " be '" expected "', and was '" actual "'")]
        (pc/attempt printf
                    "assertion: %s %s -> %s%n"
                    k
                    to-show
                    (if matches? "PASS" "FAIL"))
        (r (out-status (str k " " to-show)))))))

(defn- cell-values [labels r [result orig]]
  (concat r (zipmap (.getCells orig)
                    (map (partial get result) labels))))

(defn- cells [r]
  (if r (.getCells r)
        (throw (ex-info "No header row found in table" {}))))

(defn- labels [table]
  (when (and table (instance? Table table))
    (->> (.getHeader table)
         first
         cells
         (map #(.getText %)))))

(defn- body [table]
  (when table (.getBody table)))

(defn- table-rows [labels body]
  (map (fn [r] (zipmap labels (map (fn [c] (.getSource c))
                                   (.getCells r))))
       body))

(defn- table-to-str [labels v row]
  (str v "\n|" (str/join "|" (map #(get row %) labels))))

(defn table-title [table op]
  (let [title (.getTitle table)]
    (if (str/blank? title) op title)))

(defn set-table-title! [table op]
  (doto (table-title table op) #(.setTitle table %)))

(defn executor [exception missing-op registry current-state n c]
  (block-macro exception registry n
    (fn [_ _ t a r]
      (let [[op-name takes-args?] (if (str/blank? t) [(a "1") false]
                                                     [t true])
            proc (@c op-name)]
        (binding [format-status status-asciidoc]
          (if proc
            (do
              (pc/attempt println
                          "operation:"
                          op-name
                          (if takes-args? (pr-str a) ""))
              (reset! current-state
                      (if takes-args?
                        (proc @current-state a)
                        (proc @current-state)))
              (r (str "*" op-name ": " a "*")))
            (do
              (pc/attempt println "missing operation:" op-name)
              (r (missing-op op-name)))))))))

(defn table-handler [exception missing-op ops registry n current-state f]
  (block-macro exception registry n
    (fn [this parent _ a r]
      (binding [format-status status-asciidoc]
        (let [op (a "1")
              p (dissoc a "1")
              opfn (@ops op)]
          (if opfn
            (let [blocks (vec (.getBlocks parent))
                  tables (filter (partial instance? Table)
                                 blocks)

                  ;; We get all blocks in the given section. Get the last
                  ;; one -- hopefully the user put the macro right after
                  ;; the table they want to use as an example
                  table (last tables)
                  labels (labels table)
                  body (body table)
                  table-rows (table-rows labels body)
                  opargs
                  [@current-state {:labels labels
                                   :rows table-rows}]

                  opargs (if (empty? p) opargs (conj opargs p))
                  [s {:keys [rows]}] (apply opfn opargs)
                  _ (reset! current-state s)
                  result (f this parent op table body labels rows)]

              (pc/attempt println "=== INPUT ===")
              (pc/attempt pprint/print-table table-rows)
              (pc/attempt println "\n=== OUTPUT ===")
              (pc/attempt pprint/print-table rows)
              (pc/attempt println)
              result)
            (r (missing-op op))))))))

(defn- highlight-words [n]
  (str/replace n #"(\S+)" "*[blue]#$1#*"))

(defn register-macros
  "Registers all the asciidoc macros that accomplish the test library's work"
  [ad current-state ops procs pass fail exception]
  (let [registry (.javaExtensionRegistry ad)
        missing-op (fn [op] (exception (str "Missing op: " op)))]
    (assertion exception registry "should" = current-state pass fail)

    (assertion exception registry "shouldnot" (complement =) current-state
               pass fail)

    (executor exception missing-op registry current-state "op" ops)

    (executor exception missing-op registry current-state "exec" procs)

    (block-macro exception registry "bind"
      (fn [_ _ t a r]
        (let [k (keyword t)
              v (a "1")]
          (pc/attempt printf "%s := %s%n" k v)
          (swap! current-state assoc k v)
          (r (str "*" k " := " v "*")))))

    (table-handler
      exception missing-op ops registry "gentable" current-state
      (fn [this parent op table _ labels rows]
        (let [cells (-> table (.getBody) (.get 0) (.getCells))]
          (doseq [cell cells]
            (.setSource cell (highlight-words (.getSource cell)))))
        (let [new-block (.createBlock this parent "open" "")
              table-title (table-title table op)]
          (pc/attempt println "generated table:" table-title)
          (if (empty? rows)
            (do (.parseContent this new-block [(format "*%s*" op)])
                (pc/attempt println "No template row found to generate from"))
            (.parseContent this new-block
                           [(str "." table-title " (generated)")
                            "|==="
                            (str "|" (str/join "|" labels))
                            ""
                            (reduce (partial table-to-str labels) ""
                                    rows)
                            "|==="]))
          new-block)))

    (table-handler
      exception missing-op ops registry "table" current-state
      (fn [_ _ op table body labels rows]
        (pc/attempt printf "table: %s%n" (set-table-title! table op))
        (doseq [[cell v]
                (reduce (partial cell-values labels)
                        []
                        (zipmap rows body))]
          (.setSource cell v))
        nil))))

(defn execute [reader writer current-state ops procs pass fail exception]
  (let [ad (Asciidoctor$Factory/create)]
    (register-macros ad current-state ops procs pass fail exception)
    (.convert ad reader writer
              {"header_footer" true
               "safe" (.getLevel SafeMode/UNSAFE)
               "showtitle" true})))