;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.core
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [io.aviso.exception :as ioe]
            [itl.asciidoc :as ad]
            [itl.print-control :as pc])
  (:import (java.io FileOutputStream FileNotFoundException OutputStreamWriter)
           (java.time LocalDateTime)
           (java.time.format DateTimeFormatter)
           (org.apache.commons.io FilenameUtils)
           (org.asciidoctor Asciidoctor$Factory SafeMode))
  (:gen-class))

;; # Integration Test Library

;; These functions are bound in `execute-asciidoc`. They mark an assertion
;; as "passed" or "failed."
(def ^:dynamic pass)
(def ^:dynamic fail)
(def ^:dynamic exception)

(def ^:private ops (atom {}))
(def ^:private procs (atom {}))

;; ## Fixture definition

(defn- add-to [a n f] (swap! a assoc n f))

;; Meant to be used primarily by `defop` and `defproc`
(def add-op! (partial add-to ops))
(def add-proc! (partial add-to procs))

(defmacro defop
  "Defines an operation that can be executed in your test. Accepts a
   name (`n`) that will be referred to in brackets in the document. The body
   will be of a function that accepts a single parameter `page-state`.
   That parameter will be filled in with the current state of the running tests.

   An example op reference in your doc could be: `op::[do something]`

   Your operation *must* return whatever parts of `page-state` your operation
   wishes to pass on to future operations and/or assertions."
  [^String n [page-state] & body]
  `(add-op! ~n (fn [~page-state] ~@body)))

(defmacro defproc
  "Defines a procedure that can be executed in your test. Accepts a
   name `n` and a set of bindings `page-state` and `aargs`. Just as in `defop`,
   the state will be bound to `page-state` and must be passed back out of your
   proc with whatever changes need to be made. `a` will be bound to the
   a map of arguments. Check out
   <http://www.methods.co.nz/asciidoc/chunked/ch21.html> for more info.

   Example proc reference in your doc: `exec:say[msg=hello]`"
  [^String n [page-state args] & body]
  `(add-proc! ~n (fn [~page-state ~args] ~@body)))

(defmacro deftfn
  "Defines a table processing function. Accepts a name `n` that will be
   referred to in brackets in the table's legend. The body will be of a
   function that accepts two parameters: `page-state`, the current state of the
   running tests, and `table-data`, a structure containing the parsed table
   structure.

   Your operation must return a vector containing an updated
   `[page-state table-data]`. The value of `page-state` will be sent to the next
   operation, whereas `table-data` will be re-constituted into a table and
   inserted into the document content."
  [^String n [page-state table-data] & body]
  `(add-op! ~n (fn [~page-state ~table-data] ~@body)))

(defmacro deftafn
  "Just like `deftfn`, except this table also accepts arguments passed in
   from the source document as `args`"
  [^String n [page-state table-data args] & body]
  `(add-op! ~n (fn [~page-state ~table-data ~args] ~@body)))

;; ## Document execution

(defn execute-asciidoc
  "Parse an Asciidoc document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out`. The `initial-state` is a map of bindings that the document can use."
  [initial-state in out]
  (let [current-state (atom initial-state)
        local-stats (atom {:pass 0 :fail 0 :exception 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (ad/status given-status content))
        ad (Asciidoctor$Factory/create)
        start-time (System/currentTimeMillis)]
    (binding [pass (partial local-status :pass)
              fail (partial local-status :fail)
              exception (partial local-status :exception)]
      (ad/register-macros ad current-state ops procs pass fail exception)
      (.convert ad (io/reader in) (io/writer out)
                {"header_footer" true
                 "safe" (.getLevel SafeMode/UNSAFE)
                 "showtitle" true})
      (assoc @current-state
        :stats
        (assoc @local-stats
          :elapsed-time (- (System/currentTimeMillis) start-time)
          :run-date (.format (LocalDateTime/now)
                             (DateTimeFormatter/ofPattern
                               "yyyy-MM-dd HH:mm")))))))

(defn- rename-to-html [f]
  (-> f .getName FilenameUtils/getBaseName (str ".html")))

(defn- in-file [file]
  (let [file-resource (when (string? file) (io/resource file))]
    (io/file (or file-resource file))))

(defn run
  "Execute `file`. If `out-file` is specified, results will be written there.
   Otherwise they will be written to `file` without the `.adoc` and with
   `.html`. AKA: `README.adoc -> README.html`"
  ([file]
   (let [in-file (in-file file)
         present? (.exists in-file)
         new-name (when present? (rename-to-html in-file))]
     (if present?
       (run in-file new-name)
       (throw (FileNotFoundException.
                (str "File not found in local dir or on classpath: " file))))))
  ([file out-file]
   (if-not out-file
     (run file)
     (with-open [fout (FileOutputStream. out-file)]
       (:stats (execute-asciidoc {} file fout))))))

;; ## Utilities

;; ### Column Tables

(defn- assert-cell [page-state e row k v]
  (let [expected (row k)
        actual (v page-state)]
    (assoc row k
               (if e
                 (exception e)
                 (if (= (str expected) (str actual))
                   (pass expected)
                   (fail (format "%s (got: '%s')" expected actual)))))))

(defn- reassign-cell [page-state row k v] (assoc row k (or (v page-state) "")))

(defn- handle-row [assign exec asserts [page-state rows] row]
  (let [page-state (reduce-kv (fn [page-state k v] (assoc page-state v (row k)))
                              page-state
                              assign)
        [page-state emsg] (try
                            [(exec page-state) nil]
                            (catch Throwable t
                              (ioe/write-exception t)
                              [page-state (.getMessage t)]))
        row (reduce-kv (partial assert-cell page-state emsg) row asserts)
        row (reduce-kv (partial reassign-cell page-state) row assign)]
    [page-state (conj rows row)]))

(defn column-table
  "Take a table of data and assert a bunch of stuff on it. The
  `assign` map should contain mappings from labels in the table to
  keys they should be mapped to in the `page-state` collection. The `exec`
  should be a single function to be executed for each row. It will receive the
  current state after having all the assignments made. It should return
  the state to be passed to the assertions, and eventually, the next row.
  Finally, the `assert` should contain yet another map of labels to keys in the
  state that should match."
  [page-state {:keys [rows]} {:keys [assign exec asserts]}]
  (when (empty? assign) (println "WARNING: No assignments bound to :assign"))
  (when (empty? asserts) (println "WARNING: No assertions bound to :asserts"))
  (let [[page-state rows]
        (reduce (partial handle-row assign exec asserts) [page-state []] rows)]
    [page-state {:rows rows}]))

;; ### Generative tables

(defn generate-cell [declarations generate asserts label]
  (let [[gk gf] (generate label)
        [ak af] (asserts label)
        declaration (declarations label)
        [nk nv] (if gk [:value {gk (gf declaration)}]
                       (if ak
                         [:assert {ak (af declaration)}]
                         (throw (ex-info "Missing generator/assertion"
                                         {:label label}))))]
    {nk nv}))

(defn extract-by-key [cells k]
  (->> cells (filter k) (map k) (apply merge)))

(defn apply-generated-assertion [asserts page-state emsg k]
  (if emsg
    {k emsg}
    (let [assertion (asserts k)
          actual (page-state k)]
      {k (assertion actual)})))

(defn headers-to-keys [m]
  (apply merge (map (fn [k] {k (-> m (get k) first)}) (keys m))))

(defn assemble-generated-cell
  [declarations headers-to-keys is-assert? page-state emsg original label]
  (let [declaration (declarations label)
        assert (is-assert? label)
        k (headers-to-keys label)
        v (page-state k)]
    {label
     (if assert
       (if v
         (if emsg
           (exception emsg)
           (pass (str (original k) " was " declaration)))
         (fail (str (original k) " was not " declaration)))
       (str v))}))

(defn generate-row [page-state declarations generate exec asserts _]
  (let [labels (keys declarations)
        cells
        (map (partial generate-cell declarations generate asserts) labels)

        values (extract-by-key cells :value)
        generated-assertions (extract-by-key cells :assert)
        page-state (merge page-state values)
        [page-state emsg] (try
                            [(exec page-state) nil]
                            (catch Throwable t
                              (ioe/write-exception t)
                              [page-state (.getMessage t)]))

        asserted
        (->> (keys generated-assertions)
             (map (partial apply-generated-assertion
                           generated-assertions
                           page-state
                           emsg))
             (apply merge))
        headers-to-keys (merge (headers-to-keys generate)
                               (headers-to-keys asserts))
        original page-state
        page-state (merge page-state asserted)]
    (->> labels
         (map
           (partial assemble-generated-cell declarations headers-to-keys
                    (into (hash-set) (keys asserts)) page-state emsg original))
         (apply merge))))

(defn generative-table
  "Take a table with two rows and generate a bunch more rows suitable for
  for assertion. The first row is the names of the columns of each of the newly
  generated rows. The second row is the specifications by which we generate or
  assert against values."
  [page-state {[declarations] :rows} {:keys [generate exec asserts]} num-rows]
  (let [new-rows
        (doall (map (partial generate-row
                             page-state
                             declarations
                             generate
                             exec
                             asserts)
                    (range num-rows)))]
    [page-state {:rows new-rows}]))

;; ### Generation Example functions

(defn- -integer-generator [v]
  (let [[l r] (->> #"\.\.\." (str/split v) (map #(Integer/parseInt %)))
        diff (- r l)]
    (when (<= diff 0)
      (throw
        (ex-info
          "The first number of an integer range must be smaller than the second"
          {:given v})))
    #(+ l (rand-int diff))))

(def ^:private integer-generator (memoize -integer-generator))

(defn random-integer
  "Parses a string that contains two integers separate by 3 dots
  (e.g. \"1...50\") and produces a random number within that range, inclusive of
  the first integer, and exclusive of the second."
  [v]
  ((integer-generator v)))

(defn random-ssn [_]
  (let [first-part (random-integer "100...1000")
        second-part (random-integer "10...100")
        third-part (random-integer "1000...10000")]
    (format "%s-%s-%s" first-part second-part third-part)))

(defn integer-comparison
  "Parses a string that contains an operator and an operand to be applied
  to two values. Produces a function which takes two values and produces
  the string value \"true\" if the operand applied successfully, else
  \"false\"."
  [v]
  (let [[op operand] (str/split v #" ")
        opfn (-> op read-string eval)
        operand (Integer/parseInt operand)]
    #(opfn % operand)))

(defn bit-bob-comparison
  "Parses a string that contains a regular expression to be applied to a value"
  [_]
  (let [p (re-pattern "(Bit|Bob)\\d{1,2}")]
    (partial re-find p)))

(defn convert-missing [v]
  (if (= "MISSING" v) :missing v))

(defn- -string-loop [v]
  (let [strs (map (comp convert-missing str/trim)
                  (str/split v #","))
        str-queue (atom (vec strs))]
    #(let [fv (first @str-queue)]
       (reset! str-queue (concat (rest @str-queue) [fv]))
       fv)))

(def ^:private string-loop (memoize -string-loop))

(defn loop-of-strings [v]
  ((string-loop v)))

(defn basic-equivalence [v] (partial = v))

(comment
  (loop-of-strings "hello,there, general kenobi          ")
  (loop-of-strings "     , MISSING, disturbance in the force     ")
  ((bit-bob-comparison "") "Bob12")
  ((basic-equivalence "thing") "thing")
  )

;; ## Globally accessible fixtures

(defn exec-file
  "Run a file on the classpath through the asciidoc parser/generator"
  [outdir {:keys [file] :as page-state}]
  (pc/attempt println "execute file" page-state)
  (let [in-file (in-file file)
        out-filename (io/file outdir (rename-to-html in-file))]
    (merge page-state (run in-file out-filename))))

(defn capture-output [logfile f]
  (with-open [out-file (FileOutputStream. (io/file logfile))
              new-out (OutputStreamWriter. out-file)]
    (binding [*out* new-out] (f))))

(deftafn "execute example files" [page-state table-data {:strs [indir logfile]}]
  (let [f (fn []
            (column-table page-state table-data
                          {:assign {"File" :file}
                           :exec (partial exec-file indir)
                           :asserts {"Pass" :pass "Fail" :fail
                                     "Exception" :exception}}))]
    (if logfile (capture-output logfile f) (f))))

(defproc "use" [page-state {:strs [ns]}]
  (let [ns-to-use (when ns (->> ns (str "'") read-string eval))]
    (if ns-to-use (use ns-to-use)
                  (throw (ex-info "Namespace not found" {:ns ns}))))
  page-state)
