;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.cli
  (:require [clojure.tools.cli :refer [parse-opts]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.pprint :as pprint]
            [itl.core :as itl])
  (:gen-class))

(defn- run-file [in out]
  (if-let [{:keys [fail exception] :as stats} (itl/run in out)]
    (do
      (print (.getName in))
      (pprint/print-table [stats])
      (System/exit (if (or (pos? fail) (pos? exception)) 1 0)))
    (println "Could not find: " in)))

(defn- parse-file [f] (when f (io/file f)))

(def cli-options
  [["-n" "--namespace NAMESPACE" "Namespace to load (may contain middleware)"
    :parse-fn #(let [ns-to-use (when % (->> % (str "'") read-string eval))]
                 (when ns-to-use) (use ns-to-use) ns-to-use)]
   ["-o" "--out OUTFILE"
    "File to write results to. Defaults to %s{FILE}/.adoc/.html/"
    :parse-fn parse-file]
   ;; A boolean option defaulting to nil
   ["-h" "--help"]])

(defn- print-help [summary & [errors]]
  (when errors (println (str/join "\n" errors)))
  (printf (str "%n"
               "USAGE%n"
               "-----%n"
               "lein run <FILE> [-o <OUTFILE>] [-n <NAMESPACE>]%n"
               "%n"
               "Options:%n"
               "%s%n")
          summary))

(defn -main
  "Execute `lein run -h` for help"
  [& [file & args]]
  (let [{:keys [errors options summary] {:keys [help]} :options}
        (parse-opts args cli-options)

        file (parse-file file)]
    (cond
      help (print-help summary)
      (seq errors) (print-help summary errors)
      (nil? file) (print-help summary ["FILE not specified"])
      (not (.exists file))
      (print-help summary [(format "'%s' not found" (.getAbsolutePath file))])

      :default
      (let [{:keys [out]} options file (parse-file file)]
        (run-file file out)))))