;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.asciidoc
  (:require [clojure.spec.alpha :as s]
            [hiccup.core :refer :all]
            [io.aviso.exception :as ioe])
  (:import (org.asciidoctor.ast.impl TableImpl)
           (org.asciidoctor.extension InlineMacroProcessor)
           (java.util Map)))

(def colors {:pass "green", :fail "red" :exception "yellow"})

; This is rebound in the code generated by `inline-macro`
(def ^:dynamic format-status)
(defn status-asciidoc [color content] (format "*[%s]#%s#*" color content))
(defn status-html [color content] (html [:code {:class color} content]))

(defn status
  "Mark some content as passed or failed"
  [given-status content]
  {:pre [(s/valid? :itl.core/status given-status)]
   :post [(s/valid? string? %)]}
  (format-status (colors given-status) content))

(defn- completed-op [op] [:code [:strong op]])

(defn- missing-op [op] [:strong (str "Missing op: " op)])

(defmacro inline-macro [exception status-fn registry n b & body]
  `(.inlineMacro
     ~registry
     (proxy [InlineMacroProcessor] [~n]
       (process [~(first b) ^String ~(second b) ^Map ~(nth b 2)]
         (binding [format-status ~status-fn]
           (let [~(nth b 2) (into {} ~(nth b 2))]
             (try
               ~@body
               (catch Throwable t#
                 (ioe/write-exception t#)
                 (~exception (.getMessage t#))))))))))

(defmacro inline-html-macro [exception registry n b & body]
  `(inline-macro ~exception status-html ~registry ~n ~b ~@body))

(defmacro inline-asciidoc-macro [exception registry n b & body]
  `(inline-macro ~exception status-asciidoc ~registry ~n ~b ~@body))

(defn- assertion [exception registry n op current-state pass fail]
  (inline-html-macro exception registry n [p t a]
    (let [expected (a "1")
          k (keyword t)
          actual (k @current-state)
          matches? (op expected (str actual))
          out-status (if matches? pass fail)
          to-show (str n " be '" expected "', and was '" actual "'")]
      (out-status (str k " " to-show)))))

(defn register-macros
  "Registers all the asciidoc macros that accomplish the test library's work"
  [ad current-state ops pass fail exception]
  (let [registry (.javaExtensionRegistry ad)]
    (inline-html-macro exception registry "bind" [p t a]
      (let [k (keyword t)
            v (a "1")]
        (swap! current-state assoc k v)
        (html (completed-op (format "%s = %s" k v)))))

    (inline-html-macro exception registry "op" [p t a]
      (let [op (a "1")
            opfn (@ops op)]
        (html (if opfn
                (do (reset! current-state (opfn @current-state))
                    (completed-op op))
                (missing-op op)))))

    (assertion exception registry "should" = current-state pass fail)
    (assertion exception registry "shouldnot" (complement =) current-state
               pass fail)
    (inline-asciidoc-macro exception  registry "table" [p t a]
      (let [all-blocks (.blocks (.parent p))
            my-index (.indexOf all-blocks p)
            table (.get all-blocks my-index)
            table (if (instance? TableImpl table)
                    table
                    (.get all-blocks (inc my-index)))
            labels (->> (.getHeader table) first (.getCells)
                        (map #(.getText %)))
            body (.getBody table)
            rows (map (fn [r] (zipmap labels (map (fn [c] (.getText c))
                                                  (.getCells r))))
                      body)
            op (a "1")
            opfn (@ops op)]
        (html
          (if opfn
            (let [[s {:keys [rows]}] (opfn @current-state {:rows rows})]
              (reset! current-state s)
              (doseq [[cell v]
                      (reduce (fn [r [result orig]]
                                (concat r (zipmap (.getCells orig)
                                                  (map second result))))
                              []
                              (zipmap rows body))]
                (.setSource cell v))
              (.setTitle table op)
              nil)
            (missing-op op)))))))