;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.core
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [io.aviso.exception :as ioe]
            [itl.asciidoc :as ad])
  (:import (java.io FileOutputStream FileNotFoundException OutputStreamWriter)
           (java.time LocalDateTime)
           (java.time.format DateTimeFormatter)
           (org.apache.commons.io FilenameUtils)
           (org.asciidoctor Asciidoctor$Factory SafeMode))
  (:gen-class))

;; # Integration Test Library

;; These functions are bound in `execute-asciidoc`. They mark an assertion
;; as "passed" or "failed."
(def ^:dynamic pass)
(def ^:dynamic fail)
(def ^:dynamic exception)

(def ^:private ops (atom {}))
(def ^:private procs (atom {}))

;; ## Fixture definition

(defn- add-to [a n f] (swap! a assoc n f))

;; Meant to be used primarily by `defop` and `defproc`
(def add-op! (partial add-to ops))
(def add-proc! (partial add-to procs))

(defmacro defop
  "Defines an operation that can be executed in your document. Accepts a
   name (`n`) that will be referred to in brackets in the document. The body
   will be of a function that accepts a single parameter `s`. That parameter
   will be filled in with the current state of the running tests.

   An example op reference in your doc could be: `op::[do something]`

   Your operation *must* return whatever parts of `s` your operation wishes to
   pass on to future operations and/or assertions."
  [^String n [s] & body]
  `(add-op! ~n (fn [~s] ~@body)))

(defmacro defproc
  "Defines a procedure that can be executed in your document. Accepts a
   name `n` and a set of bindings `s` and `a`. Just as in `defop`, the
   state will be bound to `s` and must be passed back out of your proc
   with whatever changes need to be made. `a` will be bound to the
   a map of arguments. Check out
   <http://www.methods.co.nz/asciidoc/chunked/ch21.html> for more info.

   Example proc reference in your doc: `exec:say[msg=hello]`"
  [^String n [s a] & body]
  `(add-proc! ~n (fn [~s ~a] ~@body)))

(defmacro deftfn
  "Defines a table processing function. Accepts a name `n` that will be
   referred to in brackets in the table's legend. The body will be of a
   function that accepts two parameters: `s`, the current state of the running
   tests, and `t`, a structure containing the parsed table structure.

   Your operation must return a vector containing an updated `[s t]`. The
   value of `s` will be sent to the next operation, whereas `t` will be
   re-constituted into a table and inserted into the markdown stream."
  [^String n [s t] & body]
  `(add-op! ~n (fn [~s ~t] ~@body)))

(defmacro deftafn
  "Just like `deftfn`, except this table also accepts arguments passed in
   from the source document as `a`"
  [^String n [s t a] & body]
  `(add-op! ~n (fn [~s ~t ~a] ~@body)))

;; ## Document execution

(defn execute-asciidoc
  "Parse an Asciidoc document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out`. The `initial-state` is a map of bindings that the document can use."
  [initial-state in out]
  (let [current-state (atom initial-state)
        local-stats (atom {:pass 0 :fail 0 :exception 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (ad/status given-status content))
        ad (Asciidoctor$Factory/create)
        start-time (System/currentTimeMillis)]
    (binding [pass (partial local-status :pass)
              fail (partial local-status :fail)
              exception (partial local-status :exception)]
      (ad/register-macros ad current-state ops procs pass fail exception)
      (.convert ad (io/reader in) (io/writer out)
                {"header_footer" true
                 "safe" (.getLevel SafeMode/UNSAFE)
                 "showtitle" true})
      (assoc @current-state
             :stats
             (assoc @local-stats
                    :elapsed-time (- (System/currentTimeMillis) start-time)
                    :run-date (.format (LocalDateTime/now)
                                       (DateTimeFormatter/ofPattern
                                         "yyyy-MM-dd HH:mm")))))))

(defn- rename-to-html [f]
  (-> f .getName FilenameUtils/getBaseName (str ".html")))

(defn- in-file [file]
  (let [file-resource (when (string? file) (io/resource file))]
    (io/file (or file-resource file))))

(defn run
  "Execute `file`. If `out-file` is specified, results will be written there.
   Otherwise they will be written to `file` without the `.adoc` and with
   `.html`. AKA: `README.adoc -> README.html`"
  ([file]
   (let [in-file (in-file file)
         present? (.exists in-file)
         new-name (when present? (rename-to-html in-file))]
     (if present?
       (run in-file new-name)
       (throw (FileNotFoundException.
                (str "File not found in local dir or on classpath: " file))))))
  ([file out-file]
   (if-not out-file
     (run file)
     (with-open [fout (FileOutputStream. out-file)]
       (:stats (execute-asciidoc {} file fout))))))

;; ## Utilities

;; ### Column Tables

(defn- assert-cell [s e row k v]
  (let [expected (row k)
        actual (v s)]
    (assoc row k
           (if e
             (exception e)
             (if (= (str expected) (str actual))
               (pass expected)
               (fail (format "%s (got: '%s')" expected actual)))))))

(defn- reassign-cell [s row k v] (assoc row k (or (v s) "")))

(defn- handle-row [assign exec asserts [s rows] row]
  (let [s (reduce-kv (fn [s k v] (assoc s v (row k))) s assign)
        [s e] (try
                [(exec s) nil]
                (catch Throwable t
                  (ioe/write-exception t)
                  [s (.getMessage t)]))
        row (reduce-kv (partial assert-cell s e) row asserts)
        row (reduce-kv (partial reassign-cell s) row assign)]
    [s (conj rows row)]))

(defn column-table
  "Take a table of data and assert a bunch of stuff on it. The
  `assign` map should contain mappings from labels in the table to
  keys they should be mapped to in the `s` collection. The `exec` should be
  a single function to be executed for each row. It will receive the
  current state after having all the assignments made. It should return
  the state to be passed to the assertions, and eventually, the next row.
  Finally, the `assert` should contain yet another map of labels to keys in the
  state that should match."
  [s {:keys [rows]} {:keys [assign exec asserts]}]
  (when (empty? assign) (println "WARNING: No assignments bound to :assign"))
  (when (empty? asserts) (println "WARNING: No assertions bound to :asserts"))
  (let [[s rows]
        (reduce (partial handle-row assign exec asserts) [s []] rows)]
    [s {:rows rows}]))

;; ### Generative tables

(defn generate-cell [declarations generate asserts label]
  (let [[gk gf] (generate label)
        [ak af] (asserts label)
        declaration (declarations label)
        [nk nv] (if gk [:value {gk (gf declaration)}]
                       (if ak
                         [:assert {ak (af declaration)}]
                         (throw (ex-info "Missing generator/assertion"
                                         {:label label}))))]
    {nk nv}))

(defn extract-by-key [cells k]
  (->> cells (filter k) (map k) (apply merge)))

(defn apply-generated-assertion [asserts s e k]
  (if e
    {k e}
    (let [assertion (asserts k)
          actual (s k)]
      {k (assertion actual)})))

(defn headers-to-keys [m]
  (apply merge (map (fn [k] {k (-> m (get k) first)}) (keys m))))

(defn assemble-generated-cell
  [declarations headers-to-keys is-assert? s e original label]
  (let [declaration (declarations label)
        assert (is-assert? label)
        k (headers-to-keys label)
        v (s k)]
    {label
     (if assert
       (if v
         (if e
           (exception e)
           (pass (str (original k) " was " declaration)))
         (fail (str (original k) " was not " declaration)))
       (str v))}))

(defn generate-row [s declarations generate exec asserts _]
  (let [labels (keys declarations)
        cells
        (map (partial generate-cell declarations generate asserts) labels)

        values (extract-by-key cells :value)
        generated-assertions (extract-by-key cells :assert)
        s (merge s values)
        [s e] (try
                [(exec s) nil]
                (catch Throwable t
                  (ioe/write-exception t)
                  [s (.getMessage t)]))

        asserted
        (->> (keys generated-assertions)
             (map (partial apply-generated-assertion generated-assertions s e))
             (apply merge))
        headers-to-keys (merge (headers-to-keys generate)
                               (headers-to-keys asserts))
        original s
        s (merge s asserted)]
    (->> labels
         (map
           (partial assemble-generated-cell declarations headers-to-keys
                    (into (hash-set) (keys asserts)) s e original))
         (apply merge))))

(defn generative-table
  "Take a table with two rows and generate a bunch more rows suitable for
  passing to the `column-table` function for assertion. The first row is the
  names of the columns of each of the newly generated rows. The second row
  is the specifications by which we generate or assert against values."
  [s {[declarations] :rows} {:keys [generate exec asserts]} num-rows]
  (let [new-rows
        (doall (map
                 (partial generate-row s declarations generate exec asserts)
                 (range num-rows)))]
    [s {:rows new-rows}]))

;; ### Generation Example functions

(defn- -integer-generator [v]
  (let [[l r] (->> #"\.\.\." (str/split v) (map #(Integer/parseInt %)))
        diff (- r l)]
    (when (<= diff 0)
      (throw
        (ex-info
          "The first number of an integer range must be smaller than the second"
          {:given v})))
    #(+ l (rand-int diff))))

(def ^:private integer-generator (memoize -integer-generator))

(defn random-integer
  "Parses a string that contains two integers separate by 3 dots
  (e.g. \"1...50\") and produces a random number within that range, inclusive of
  the first integer, and exclusive of the second."
  [v]
  ((integer-generator v)))

(defn random-ssn [_]
  (let [f (random-integer "100...1000")
        s (random-integer "10...100")
        t (random-integer "1000...10000")]
    (format "%s-%s-%s" f s t)))

(defn integer-comparison
  "Parses a string that contains an operator and an operand to be applied
  to two values. Produces a function which takes two values and produces
  the string value \"true\" if the operand applied successfully, else
  \"false\"."
  [v]
  (let [[op operand] (str/split v #" ")
        opfn (-> op read-string eval)]
    #(opfn % (Integer/parseInt operand))))

(defn- -string-loop [v]
  (let [strs (map str/trim (str/split v #","))
        str-queue (atom (into [] strs))]
    #(let [fv (first @str-queue)]
       (reset! str-queue (concat (rest @str-queue) [fv]))
       fv)))

(def ^:private string-loop (memoize -string-loop))

(defn loop-of-strings [v]
  ((string-loop v)))

(comment
  (loop-of-strings "hello,there, general kenobi          "))

(defn basic-equivalence [v]
  #(= v %))

;; ## Globally accessible fixtures

(defn exec-file
  "Run a file on the classpath through the asciidoc parser/generator"
  [outdir {:keys [file] :as s}]
  (let [in-file (in-file file)
        out-filename (io/file outdir (rename-to-html in-file))]
    (merge s (run in-file out-filename))))

(deftafn "execute example files" [s t {:strs [outdir]}]
  (with-open [out-file
              (FileOutputStream. (io/file (or outdir ".") "example-output.log"))

              new-out (OutputStreamWriter. out-file)]
    (binding [*out* new-out]
      (column-table s t {:assign {"File" :file}
                         :exec (partial exec-file outdir)
                         :asserts {"Pass" :pass "Fail" :fail
                                   "Exception" :exception}}))))

(defproc "use" [s {:strs [ns]}]
  (let [ns-to-use (when ns (->> ns (str "'") read-string eval))]
    (if ns-to-use (use ns-to-use)
                  (throw (ex-info "Namespace not found" {:ns ns}))))
  s)