;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.example
  (:require [itl.core :refer :all]
            [clojure.pprint :as pprint]
            [clojure.java.io :as io]
            [clojure.set :as set])
  (:import (java.io ByteArrayOutputStream File FileNotFoundException
                    FileOutputStream)
           (org.apache.commons.io FilenameUtils)))

;; # itl example
;;
;; This is where we bind the functions that test itl itself.

(defn- run
  ([in out] (:stats (execute-asciidoc {} in out)))

  ([filename]
   (let [^File f (io/file filename)
         present? (and f (.exists f))
         new-name (when present?
                    (-> f .getName FilenameUtils/getBaseName (str ".html")))]
     (if present?
       (with-open [fout (FileOutputStream. new-name)]
         (run f fout))
       (println "File not found:" (.getAbsolutePath f))))))

(defn -main
  "Take a given asciidoc file and execute it in the example namespace, rendering
   its result to an HTML file on disk with the same name and the '.html'
   extension instead of 'adoc'"
  [& [filename]]
  (if-let [{:keys [fail] :as stats} (run filename)]
    (do
      (print filename)
      (pprint/print-table [stats])
      (System/exit (if (pos? fail) 1 0)))
    (println "Could not find: " filename)))

(comment
  ;; For running example files in a REPL
  (run "dev-resources/complete-example.adoc")
  (run "README.adoc")
  )

;; ## `complete-example.adoc` fixtures
(def server-status (atom {}))

(defn- update-server-status [s status]
  (let [server-name (:server s)]
    (swap! server-status assoc server-name status)
    s))

(defop "start the server" [s] (update-server-status s "up"))

(defop "stop the server" [s] (update-server-status s "down"))

(defop "check health" [s] (assoc s :status (@server-status (:server s))))

(deftfn "check register" [s t]
  (column-table s t {:assign {"Amount" :amount}
                     :exec (fn [{:keys [amount balance] :as s}]
                             (let [balance (Integer/parseInt balance)
                                   to-add (Integer/parseInt amount)
                                   actual-balance (+ balance to-add)]
                               (-> s
                                   (assoc :balance (str actual-balance))
                                   (dissoc :amount))))
                     :asserts {"Balance" :balance}}))

;; ## `README.adoc` fixtures

(defn exec-file
  "Run a file on the classpath through the asciidoc parser/generator"
  [{:keys [file] :as s}]
  (if file
    (let [out (ByteArrayOutputStream.)] (merge s (run (io/resource file) out)))
    (throw (FileNotFoundException.
             (str "File not found on classpath: " file)))))

(deftfn "execute example files"
  [s t]
  (column-table s t {:assign {"File" :file}
                     :exec exec-file
                     :asserts {"Pass" :pass "Fail" :fail}}))