;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.core
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [itl.asciidoc :as ad])
  (:import (java.time LocalDateTime)
           (java.time.format DateTimeFormatter)
           (org.asciidoctor Asciidoctor$Factory SafeMode))
  (:gen-class))

;; # Integration Test Library

(s/def ::status #{:pass :fail})

;; These functions are bound in `execute-asciidoc`. They mark an assertion
;; as "passed" or "failed."
(def ^:dynamic pass)
(def ^:dynamic fail)

(def ^:private ops (atom {}))

;; ## Fixture definition

(defn add-op!
  "Meant to be used primarily by `defop`."
  [n f] (swap! ops assoc n f))

(defmacro defop
  "Defines an operation that can be executed in your document. Accepts a
   name (`n`) that will be referred to in brackets. The body will be of a
   function that accepts a single parameter `s`. That parameter will be filled
   in with the current state of the running tests.

   Your operation *must* return whatever parts of `s` your operation wishes to
   pass on to future operations and/or assertions."
  [^String n [s] & body]
  `(add-op! ~n (fn [~s] ~@body)))

(defmacro deftfn
  "Defines a table processing function. Accepts a name (`n`) that will be
   referred to in brackets in the table's legend. The body will be of a
   function that accepts two parameters: `s`, the current state of the running
   tests, and `t`, a structure containing the parsed table structure.

   Your operation must return a vector containing an updated [`s` `t`]. The
   value of `s` will be sent to the next operation, whereas `t` will be
   re-constituted into a table and inserted into the markdown stream."
  [^String n [s t] & body]
  `(add-op! ~n (fn [~s ~t] ~@body)))

;; ## Document execution

(defn execute-asciidoc
  "Parse an Asciidoc document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out-stream`."
  [initial-state in out-stream]
  (let [current-state (atom initial-state)
        local-stats (atom {:pass 0 :fail 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (ad/status given-status content))
        ad (Asciidoctor$Factory/create)
        start-time (System/currentTimeMillis)
        pass (partial local-status :pass)
        fail (partial local-status :fail)]
    (ad/register-macros ad current-state ops pass fail)
    (binding [pass (partial local-status :pass)
              fail (partial local-status :fail)]
      (.convert ad (io/reader in) (io/writer out-stream)
                {"header_footer" true
                 "safe" (.getLevel SafeMode/UNSAFE)
                 "showtitle" true})
      (assoc @current-state
        :stats
        (assoc @local-stats
          :elapsed-time (- (System/currentTimeMillis) start-time)
          :run-date (.format (LocalDateTime/now)
                             (DateTimeFormatter/ofPattern
                               "yyyy-MM-dd HH:mm")))))))

;; ## Utilities

(defn- handle-cell [s row k v]
  (let [expected (row k)
        actual (v s)]
    (assoc row k (if (= (str expected) (str actual))
                   (pass expected)
                   (fail (format "%s (got: '%s')" expected actual))))))

(defn- handle-row [assign exec asserts [s rows] row]
  (let [s (reduce-kv (fn [s k v] (assoc s v (row k))) s assign)
        s (exec s)
        row (reduce-kv (partial handle-cell s) row asserts)]
    [s (conj rows row)]))

(defn column-table
  "Take a table of data and assert a bunch of stuff on it. The
  `assign` map should contain mappings from labels in the table to
  keys they should be mapped to in the `s` collection. The `exec` should be
  a single function to be executed for each row. It will receive the
  current state after having all the assignments made. It should return
  the state to be passed to the assertions, and eventually, the next row.
  Finally, the `assert` should contain yet another map of labels to keys in the
  state that should match."
  [s {:keys [labels sep rows]} {:keys [assign exec asserts]}]
  (when (empty? assign) (println "WARNING: No assignments bound to :assign"))
  (when (empty? asserts) (println "WARNING: No assertions bound to :asserts"))
  (let [[s rows]
        (reduce (partial handle-row assign exec asserts) [s []] rows)]
    [s {:labels labels :sep sep :rows rows}]))