;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.asciidoc
  (:require [clojure.string :as str]
            [hiccup.core :refer :all]
            [io.aviso.exception :as ioe])
  (:import (org.asciidoctor.ast.impl TableImpl)
           (org.asciidoctor.extension InlineMacroProcessor BlockMacroProcessor)
           (java.util Map)))

(def colors {:pass "green", :fail "red" :exception "yellow"})
(def status-handlers {:pass identity :fail identity
                      :exception #(str "E[" % "]")})

; This is rebound in the code generated by `inline-macro`
(def ^:dynamic format-status)

(defn status-asciidoc [color content]
  (let [trimmed (str/trim content)]
    (if-not (str/blank? trimmed)
      (format "*[%s]#%s#*" color trimmed)
      trimmed)))

(defn status-html [color content] (html [:code {:class color} content]))

(defn status
  "Mark some content as passed or failed"
  [given-status content]
  (format-status (colors given-status)
                 ((status-handlers given-status) content)))

(defn- completed-op [op] [:code [:strong op]])

(defmacro inline-macro [exception status-fn registry n b & body]
  `(.inlineMacro
     ~registry
     (proxy [InlineMacroProcessor] [~n]
       (process [~(first b) ^String ~(second b) ^Map ~(nth b 2)]
         (let [~(nth b 2) (into {} ~(nth b 2))]
           (try
             (binding [format-status ~status-fn] ~@body)
             (catch Throwable t#
               (ioe/write-exception t#)
               (binding [format-status status-html]
                 (~exception (.getMessage t#))))))))))

(defmacro inline-html-macro [exception registry n b & body]
  `(inline-macro ~exception status-html ~registry ~n ~b ~@body))

(defmacro inline-asciidoc-macro [exception registry n b & body]
  `(inline-macro ~exception status-asciidoc ~registry ~n ~b ~@body))

(defn- assertion [exception registry n op current-state pass fail]
  (inline-html-macro exception registry n [p t a]
    (let [expected (a "1")
          k (keyword t)
          actual (k @current-state)
          matches? (op expected (str actual))
          out-status (if matches? pass fail)
          to-show (str n " be '" expected "', and was '" actual "'")]
      (out-status (str k " " to-show)))))

(defmacro exec-op [current-state ops op completed-op missing-op b & body]
  `(if-let [~(first b) (@~ops ~op)]
     (html (do (reset! ~current-state (-> @~current-state ~@body))
               (~completed-op ~op)))
     (~missing-op ~op)))

(defn- cell-values [labels r [result orig]]
  (concat r (zipmap (.getCells orig)
                    (map (partial get result) labels))))

(defn- labels [table]
  (when (and table (instance? TableImpl table))
    (->> (.getHeader table)
         first
         (.getCells)
         (map #(.getText %)))))

(defn- body [table]
  (when table (.getBody table)))

(defn- table-rows [labels body]
  (map (fn [r] (zipmap labels (map (fn [c] (.getSource c))
                                   (.getCells r))))
       body))

(defn- table-to-str [labels v row]
  (str v "\n|" (str/join "|" (map #(get row %) labels))))

(defn register-macros
  "Registers all the asciidoc macros that accomplish the test library's work"
  [ad current-state ops procs pass fail exception]
  (let [registry (.javaExtensionRegistry ad)
        missing-op (fn [op] (exception (str "Missing op: " op)))]
    (inline-html-macro exception registry "bind" [p t a]
      (let [k (keyword t)
            v (a "1")]
        (swap! current-state assoc k v)
        (html (completed-op (format "%s = %s" k v)))))

    (inline-html-macro exception registry "op" [p t a]
      (exec-op current-state ops (a "1")
               completed-op
               missing-op [op] op))

    (inline-html-macro exception registry "exec" [p t a]
      (exec-op current-state procs t
               (fn [_] (completed-op (format "%s: %s" t a)))
               missing-op [proc] (proc a)))

    (assertion exception registry "should" = current-state pass fail)

    (assertion exception registry "shouldnot" (complement =) current-state
               pass fail)

    (inline-asciidoc-macro exception registry "table" [p t a]
      (let [all-blocks (.blocks (.parent p))
            my-index (.indexOf all-blocks p)
            table (.get all-blocks my-index)
            table (if (instance? TableImpl table)
                    table
                    (.get all-blocks (inc my-index)))
            labels (labels table)
            body (body table)
            table-rows (table-rows labels body)
            op (a "1")
            p (dissoc a "1")
            opfn (@ops op)]
        (if opfn
          (html
            (let [opargs
                  [@current-state {:labels labels
                                   :rows table-rows}]

                  opargs (if (empty? p) opargs (conj opargs p))
                  [s {:keys [rows]}] (apply opfn opargs)]
              (reset! current-state s)
              (doseq [[cell v]
                      (reduce (partial cell-values labels)
                              []
                              (zipmap rows body))]
                (.setSource cell v))
              (.setTitle table op)
              nil))
          (missing-op op))))

    (.blockMacro
      registry
      (proxy [BlockMacroProcessor] ["exec"]
        (process [parent target attributes]
          (let [a (into {} attributes)
                proc (@procs target)]
            (binding [format-status status-asciidoc]
              (if proc
                (do (reset! current-state (proc @current-state a))
                    (.createBlock this parent "paragraph"
                                  (str "*" target ": " a "*")))
                (.createBlock this parent "paragraph"
                              (missing-op target))))))))

    (.blockMacro
      registry
      (proxy [BlockMacroProcessor] ["gentable"]
        (process [parent _ attributes]
          (binding [format-status status-asciidoc]
            (let [a (into {} attributes)
                  op (a "1")
                  p (dissoc a "1")
                  opfn (@ops op)]
              (if opfn
                (let [blocks (into [] (.getBlocks parent))
                      tables (filter (partial instance? TableImpl)
                                     blocks)

                      ;; We get all blocks in the given section. Get the last
                      ;; one -- hopefully the user put the macro right after
                      ;; the table they want to use as an example
                      table (last tables)
                      labels (labels table)
                      body (body table)
                      table-rows (table-rows labels body)
                      opargs
                      [@current-state {:labels labels
                                       :rows table-rows}]

                      opargs (if (empty? p) opargs (conj opargs p))
                      [s {:keys [rows]}] (apply opfn opargs)

                      new-table-content
                      [(str "." "generated " op)
                       "|==="
                       (str "|" (str/join "|" labels))
                       ""
                       (reduce (partial table-to-str labels) "" rows)
                       "|==="]

                      new-block (.createBlock this parent "open" nil)]
                  (.parseContent this new-block new-table-content)
                  (reset! current-state s)
                  new-block)
                (.createBlock this parent "paragraph" (missing-op op))))))))))