(ns itl.md
  (:require [instaparse.core :as insta]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s])
  (:import (com.vladsch.flexmark.ast Document Paragraph Code SoftLineBreak
                                     Heading)
           (com.vladsch.flexmark.ext.gfm.strikethrough StrikethroughExtension)
           (com.vladsch.flexmark.ext.tables TablesExtension TableBlock TableHead
                                            TableSeparator TableBody
                                            TableCaption TableRow)
           (com.vladsch.flexmark.parser Parser ParserEmulationProfile)
           (com.vladsch.flexmark.util.options MutableDataSet)
           (itl AssertionResultExtension)))

;; # Markdown Processing

(defn status
  "Mark some content as passed or failed."
  [given-status content]
  {:pre [(s/valid? :itl.core/status given-status)]
   :post [(s/valid? string? %)]}
  (format "~!%s/%s/" given-status content))

(def parse-code
  (insta/parser
    "<S> = B | O;
     B = (<'`{'>, #'[^\\}]+', <'}`'>);
     O = (<'`['>, #'[^\\]]+', <']`'>);"))

(def md-options
  (-> (MutableDataSet.)
      (.set Parser/EXTENSIONS
            [(StrikethroughExtension/create) (TablesExtension/create)
             (AssertionResultExtension.)])
      (.setFrom (ParserEmulationProfile/MULTI_MARKDOWN))))

(defn parse
  "Parse a markdown document into a flexmark AST"
  [in]
  (let [options md-options]
    (-> (Parser/builder options) (.build) (.parseReader (io/reader in)))))

(defn- children-seq [b] (iterator-seq (.iterator (.getChildren b))))

(defn- process-chars [t] (-> t (.getChars) str))

(defn- process-text [t] (-> t (.getText) str))

(defmulti process-table-record class)

(defmethod process-table-record TableHead [th]
  {:labels (->> th children-seq first children-seq (map process-text))})

(defmethod process-table-record TableSeparator [ts]
  {:sep (process-chars ts)})

(defmethod process-table-record TableBody [tb]
  {:rows (map process-table-record (children-seq tb))})

(defmethod process-table-record TableRow [tr]
  (map process-text (children-seq tr)))

(defmethod process-table-record TableCaption [tc]
  (let [code (process-text tc)
        parsed (parse-code code)]
    (if-not (insta/failure? parsed)
      {:op (-> parsed first second)}
      {:caption code})))

(defmulti process-node class)

(defn- process-block [b] (map process-node (children-seq b)))

(defmethod process-node Document [d] [:D (process-block d)])

(defmethod process-node Paragraph [p] [:P (process-block p)])

(defmethod process-node Code [c]
  (-> c process-chars parse-code first))

(defmethod process-node TableBlock [t]
  [:T (apply merge (map process-table-record (children-seq t)))])

(defmethod process-node Heading [h]
  [:P [(process-chars h)]])

(defmethod process-node SoftLineBreak [_] "\n")

(defmethod process-node :default [p] (process-chars p))