;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.core
  (:require [clojure.java.io :as io]
            [clojure.pprint :as pprint]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [itl.asciidoc :as ad]
            [itl.md :as md])
  (:import (java.time LocalDateTime)
           (java.time.format DateTimeFormatter)
           (java.util List)
           (org.asciidoctor Asciidoctor$Factory SafeMode)
           (java.io File FileOutputStream)
           (org.apache.commons.io FilenameUtils))
  (:gen-class))

;; # Integration Test Library

(s/def ::status #{"P" "F"})

;; These functions are bound in `execute-document`. They mark an assertion
;; as "passed" or "failed."
(def ^:dynamic pass)
(def ^:dynamic fail)

(def ^:private ops (atom {}))

(defn- to-stream [out-stream s]
  (.write out-stream (.getBytes s "UTF-8")) (.flush out-stream))

(defn- to-streamf [out-stream s & args]
  (to-stream out-stream (apply format (into [s] args))))

(defn- unhandled
  ([s out-stream tt tv]
   (to-stream out-stream (if tt (str "_`" tt "/" tv "`_") tv)) s)
  ([s out-stream tv] (unhandled s out-stream nil tv)))

(defn- unhandled-bracketed [s out-stream tv]
  (unhandled s out-stream (str "[" tv "]")))

(defn- row-to-stream
  ([out-stream row]
   (to-stream out-stream (str "|" (str/join "|" row) "|\n")))
  ([out-stream labels row]
   (row-to-stream out-stream (map row labels))))

(defn- write-table [{:keys [labels sep rows]} out-stream]
  (row-to-stream out-stream labels)
  (to-stream out-stream (str sep "\n"))
  (doall (map (partial row-to-stream out-stream labels) rows)))

(defn- unhandled-table [s out-stream t op caption]
  (write-table t out-stream)
  (unhandled-bracketed s out-stream
                       (if op (str "`[" op "]`") caption)))

(defn- state-key [label]
  (-> label str/trim (str/replace #" " "-") keyword))

(defn- handle-check [s out-stream l r op match-op nomatch-op]
  (let [actual (s l)
        matches? (op r actual)
        operator (if matches? match-op nomatch-op)
        out-status (if matches? pass fail)
        expected-to-show (if matches? ""
                                      (str " (got '" actual "')"))]
    (to-streamf out-stream (out-status "%s %s '%s'%s")
                l operator r expected-to-show)
    s))

(def ^:private cmd-exec-fmt "**`%s`**")

(def ^:private table-exec-fmt (str "[" cmd-exec-fmt "]"))

(defn- executed
  ([fmt s out-stream e] (to-streamf out-stream fmt e) s)
  ([s out-stream e] (executed cmd-exec-fmt s out-stream e)))

(defmulti handle-token (fn [t _ _ _] (type t)))

(defn- handle-block [block out-stream s last?]
  (reduce (fn [s t] (handle-token t out-stream s last?)) s block))

(defn- block-separator [out-stream s last?]
  (when-not last? (to-stream out-stream "\n\n")) s)

(defmethod handle-token String [cur-token out-stream s _]
  (to-stream out-stream cur-token) s)

(defmethod handle-token List [[tt tv] out-stream s last?]
  (case tt
    :D (let [s (reduce (fn [s t] (handle-token t out-stream s false)) s
                       (butlast tv))]
         (handle-token (last tv) out-stream s true))
    :P (block-separator out-stream (handle-block tv out-stream s last?) last?)
    :B (let [[_ l o r] (re-matches #"^(.*?)([:\!]?=)(.*)$" tv)
             l (when l (state-key l))
             r (when r (str/trim r))]
         (case o
           ":=" (executed (assoc s l r) out-stream (str l " = " r))
           "=" (handle-check s out-stream l r = "=" "!=")
           "!=" (handle-check s out-stream l r (complement =) "!=" "=")
           (unhandled s out-stream tt tv)))
    :O (if-let [f (@ops tv)]
         (executed (f s) out-stream tv)
         (unhandled-bracketed s out-stream tv))
    :T (let [{:keys [labels sep rows op caption]} tv
             t {:labels labels
                :sep sep
                :rows (map (partial zipmap labels) rows)}]
         (block-separator out-stream
                          (if op
                            (if-let [f (@ops op)]
                              (let [[s t]
                                    (f s t)]
                                (write-table t out-stream)
                                (executed table-exec-fmt s out-stream op))
                              (unhandled-table s out-stream t op caption))
                            (unhandled-table s out-stream t op caption)) last?))
    (unhandled s out-stream tt tv)))

;; ## Fixture definition

(defn add-op!
  "Meant to be used primarily by `defop`."
  [n f] (swap! ops assoc n f))

(defmacro defop
  "Defines an operation that can be executed in your document. Accepts a
   name (`n`) that will be referred to in brackets. The body will be of a
   function that accepts a single parameter `s`. That parameter will be filled
   in with the current state of the running tests.

   Your operation *must* return whatever parts of `s` your operation wishes to
   pass on to future operations and/or assertions."
  [^String n [s] & body]
  `(add-op! ~n (fn [~s] ~@body)))

(defmacro deftfn
  "Defines a table processing function. Accepts a name (`n`) that will be
   referred to in brackets in the table's legend. The body will be of a
   function that accepts two parameters: `s`, the current state of the running
   tests, and `t`, a structure containing the parsed table structure.

   Your operation must return a vector containing an updated [`s` `t`]. The
   value of `s` will be sent to the next operation, whereas `t` will be
   re-constituted into a table and inserted into the markdown stream."
  [^String n [s t] & body]
  `(add-op! ~n (fn [~s ~t] ~@body)))

;; ## Document execution

(defn execute-markdown
  "Parse a Markdown document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out-stream`."
  [initial-state in out-stream]
  (let [local-stats (atom {"P" 0 "F" 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (md/status given-status content))
        start-time (System/currentTimeMillis)]
    (binding [pass (partial local-status "P")
              fail (partial local-status "F")]
      (let [result (handle-token
                     (-> in md/parse md/process-node) out-stream initial-state
                     true)]
        (assoc result
          :stats
          (assoc @local-stats
            :elapsed-time (- (System/currentTimeMillis) start-time)
            :run-date (.format (LocalDateTime/now)
                               (DateTimeFormatter/ofPattern
                                 "yyyy-MM-dd HH:mm"))))))))

(defn execute-asciidoc
  "Parse an Asciidoc document on `in` that contains bindings, assertions, and/or
   operations, executing them, and writing the modified markdown into the given
   `out-stream`."
  [initial-state in out-stream]
  (let [current-state (atom initial-state)
        local-stats (atom {"P" 0 "F" 0})
        local-status (fn [given-status content]
                       (swap! local-stats update given-status inc)
                       (ad/status given-status content))
        ad (Asciidoctor$Factory/create)
        start-time (System/currentTimeMillis)
        pass (partial local-status "P")
        fail (partial local-status "F")]
    (ad/register-macros ad current-state ops pass fail)
    (binding [pass (partial local-status "P")
              fail (partial local-status "F")]
      (.convert ad (io/reader in) (io/writer out-stream)
                {"header_footer" true
                 "safe" (.getLevel SafeMode/UNSAFE)})
      (assoc @current-state
        :stats
        (assoc @local-stats
          :elapsed-time (- (System/currentTimeMillis) start-time)
          :run-date (.format (LocalDateTime/now)
                             (DateTimeFormatter/ofPattern
                               "yyyy-MM-dd HH:mm")))))))

;; ## Utilities

(defn- handle-cell [s row k v]
  (let [expected (row k)
        actual (v s)]
    (assoc row k (if (= expected actual)
                   (pass expected)
                   (fail (format "%s (got: '%s')" expected actual))))))

(defn- handle-row [assign exec asserts [s rows] row]
  (let [s (reduce-kv (fn [s k v] (assoc s v (row k))) s assign)
        s (exec s)
        row (reduce-kv (partial handle-cell s) row asserts)]
    [s (conj rows row)]))

(defn column-table
  "Take a table of data and assert a bunch of stuff on it. The
  `assign` map should contain mappings from labels in the table to
  keys they should be mapped to in the `s` collection. The `exec` should be
  a single function to be executed for each row. It will receive the
  current state after having all the assignments made. It should return
  the state to be passed to the assertions, and eventually, the next row.
  Finally, the `assert` should contain yet another map of labels to keys in the
  state that should match."
  [s {:keys [labels sep rows] :as t} {:keys [assign exec asserts]}]
  (let [[s rows]
        (reduce (partial handle-row assign exec asserts) [s []] rows)]
    [s {:labels labels :sep sep :rows rows}]))