;
; Copyright (c) 2018.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.asciidoc
  (:require [hiccup.core :refer :all]
            [clojure.spec.alpha :as s])
  (:import (org.asciidoctor.extension InlineMacroProcessor)
           (java.util Map)))

(def colors {"P" "green", "F" "red"})

(defn status
  "Mark some content as passed or failed."
  [given-status content]
  {:pre [(s/valid? :itl.core/status given-status)]
   :post [(s/valid? string? %)]}
  (html [:code {:style (str "color:" (colors given-status))}
         content]))

(defn- completed-op [op] [:code [:strong op]])

(defn- missing-op [op] [:strong (str "Missing op: " op)])

(defmacro inline-macro [registry n b & body]
  `(.inlineMacro
     ~registry
     (proxy [InlineMacroProcessor] [~n]
       (process [~(first b) ^String ~(second b) ^Map ~(nth b 2)]
         (let [~(nth b 2) (into {} ~(nth b 2))]
           ~@body)))))

(defn- assertion [registry n op current-state pass fail]
  (inline-macro registry n [p t a]
    (let [expected (a "1")
          k (keyword t)
          actual (k @current-state)
          matches? (op expected (str actual))
          out-status (if matches? pass fail)
          to-show (str n " be '" expected "', and was '" actual "'")]
      (out-status (str k " " to-show)))))

(defn register-macros [ad current-state ops pass fail]
  (let [registry (.javaExtensionRegistry ad)]
    (inline-macro registry "bind" [p t a]
      (let [k (keyword t)
            v (a "1")]
        (swap! current-state assoc k v)
        (html (completed-op (format "%s = %s" k v)))))

    (inline-macro registry "op" [p t a]
      (let [op (a "1")
            opfn (@ops op)]
        (html (if opfn
                (do (reset! current-state (opfn @current-state))
                    (completed-op op))
                (missing-op op)))))

    (assertion registry "should" = current-state pass fail)
    (assertion registry "shouldnot" (complement =) current-state pass fail)
    (inline-macro registry "table" [p t a]
      (let [all-blocks (.blocks (.parent p))
            my-index (.indexOf all-blocks p)
            table (.get all-blocks my-index)
            labels (->> (.getHeader table) first (.getCells)
                        (map #(.getText %)))
            rows (map (fn [r] (zipmap labels (map (fn [c] (.getText c))
                                                  (.getCells r))))
                      (.getBody table))
            op (a "1")
            opfn (@ops op)]
        (html (if opfn
                (let [[s t] (opfn @current-state {:rows rows})]
                  (reset! current-state s)
                  ;;todo mutate table cells using output of `t`
                  ;;todo insert table caption to show completed op
                  nil)
                (missing-op op)))))))