/*
 * Copyright (c) 2018.
 *
 * This file is part of itl.
 *
 * itl is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * itl is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with itl.  If not, see <http://www.gnu.org/licenses/>.
 */

package itl;

import com.vladsch.flexmark.formatter.internal.Formatter;
import com.vladsch.flexmark.formatter.internal.NodeFormatter;
import com.vladsch.flexmark.formatter.internal.NodeFormattingHandler;
import com.vladsch.flexmark.html.HtmlRenderer;
import com.vladsch.flexmark.html.renderer.NodeRenderingHandler;
import com.vladsch.flexmark.parser.InlineParser;
import com.vladsch.flexmark.parser.InlineParserExtension;
import com.vladsch.flexmark.parser.InlineParserExtensionFactory;
import com.vladsch.flexmark.parser.Parser;
import com.vladsch.flexmark.util.options.MutableDataHolder;
import com.vladsch.flexmark.util.sequence.BasedSequence;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import static java.lang.String.format;

public class AssertionResultExtension implements Parser.ParserExtension,
    HtmlRenderer.HtmlRendererExtension, Formatter.FormatterExtension {
  private static final Pattern ASSERTION_PATTERN = Pattern.compile("~!([PF])/(.*?)/");

  @Override
  public void extend(Formatter.Builder builder) {
    builder.nodeFormatterFactory(dataHolder -> new NodeFormatter() {
      @Override
      public Set<NodeFormattingHandler<?>> getNodeFormattingHandlers() {
        return new HashSet<>(Collections.singletonList(
            new NodeFormattingHandler<>(AssertionResult.class,
                (assertionResult, nodeFormatterContext, markdownWriter) ->
                    markdownWriter.append(assertionResult.getChars()))));
      }

      @Override
      public Set<Class<?>> getNodeClasses() {
        return new HashSet<>(Collections.singletonList(AssertionResult.class));
      }
    });
  }

  @Override
  public void rendererOptions(MutableDataHolder options) {
  }

  @Override
  public void extend(HtmlRenderer.Builder rendererBuilder, String rendererType) {
    if ("HTML".equals(rendererType)) {
      rendererBuilder.nodeRendererFactory(options -> () ->
          new HashSet<>(Collections.singletonList(
              new NodeRenderingHandler<>(AssertionResult.class, (node, context, html) -> {
                html.attr("style", format("color: %s",
                    node.getStatus().choose("green", "red")));
                html.withAttr().tag("code");
                html.text(node.getContent());
                html.closeTag("code");
              }))));
    } else {
      System.err.println("[AssertionResultExtension] Renderer type not supported: " +
          rendererType);
    }
  }

  @Override
  public void parserOptions(MutableDataHolder options) {
  }

  @Override
  public void extend(Parser.Builder parserBuilder) {
    parserBuilder.customInlineParserExtensionFactory(new InlineParserExtensionFactory() {
      @Override
      public CharSequence getCharacters() {
        return "~";
      }

      @Override
      public InlineParserExtension create(InlineParser parser) {
        return new InlineParserExtension() {
          @Override
          public void finalizeDocument(InlineParser inlineParser) {

          }

          @Override
          public void finalizeBlock(InlineParser inlineParser) {

          }

          @Override
          public boolean parse(InlineParser inlineParser) {
            if (inlineParser.peek(1) == '!') {
              BasedSequence[] groups = inlineParser.matchWithGroups(ASSERTION_PATTERN);
              if (groups != null && groups.length == 3) {
                inlineParser.flushTextNode();
                inlineParser.getBlock().appendChild(new AssertionResult(groups));
                return true;
              }
            }
            return false;
          }
        };
      }

      @Override
      public Set<? extends Class> getAfterDependents() {
        return null;
      }

      @Override
      public Set<? extends Class> getBeforeDependents() {
        return null;
      }

      @Override
      public boolean affectsGlobalScope() {
        return false;
      }
    });
  }
}
