;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

;; # Welcome to defenv
;;
;; This library attempts to simplify the management of environment variables.
;; Put simply, it's `System/getenv` on steroids.
;;
;; Features include: documentation generation, default values, and custom
;; parsing. Also critical is the use of delayed binding so your application
;; doesn't throw exceptions when loading namespaces (a huge pet peeve!).
;;
;; Take a look at `defenv.usage` for examples of how to use the library.

(ns defenv.core
  (:require [clojure.pprint :refer :all]
            [clojure.repl :as repl]
            [clojure.set :refer :all]
            [slingshot.slingshot :refer [throw+]]
            [clojure.string :as str]))

(def ^:dynamic *getenv* (fn [n] (System/getenv n)))

(defn- env
  [env-name default-value]
  (let [val (*getenv* env-name)] (if-not val default-value val)))

;; ## Transformation convenience functions
;; (for use wherever you see `:tfn`)

(defn parse-bool [s] (Boolean/parseBoolean s))
(defn parse-int [s] (Integer/parseInt s))
(defn parse-long [s] (Long/parseLong s))
(defn parse-float [s] (Float/parseFloat s))
(defn parse-double [s] (Double/parseDouble s))
(defn parse-boolean
  "DEPRECATED in favor of parse-bool"
  {:deprecated "0.0.6"} [s] (parse-bool s))

(defn parse-map
  "Parses a string containing key-value pairs into a map. Each pair is
   separated by a comma, and the key and value by an equal sign. Whitespace
   surrounding keys and values is ignored. Example:
   `a=b,c=d` parses to {\"a\"=\"b\", \"c\"=\"d\"}.

   You can pass two configuration options into the `opts` map:

   `:default-key`
   specifies what happens when either you don't follow the syntax (you get a map
   with the value of `:default-key` as its key, and the variable value as its
   value) or when you have a key in the map that is blank.

   `:key-fn`
   specifies a function to execute on each key. This function is not executed
   on the `:default-key`

   See `defenv.tfn-spec` for examples."
  ([s] (parse-map {} s))
  ([{:keys [default-key key-fn] :or {default-key "default"
                                     key-fn identity}} s]
   (if (re-find #"=" s)
     (let [kvs (str/split s #",")]
       (reduce (fn [m s]
                 (let [[k v] (map str/trim (str/split s #"="))
                       k (if (str/blank? k) default-key (key-fn k))]
                   (when (m k)
                     (binding [*out* *err*]
                       (println "**WARNING** Duplicate key found in map:" k)))
                   (assoc m k (or v "")))) {} kvs))
     {default-key s})))

;; ## On-the-fly documentation

;; ### Generation
(def ^:private displays {::missing "*REQUIRED*"
                         ::masked "--MASKED--"
                         ::parse-error "*PARSE ERROR*"})

(defn- get-var-status [{:keys [env-name v default masked? optional?]
                        :as doc-map}]
  (let [e-val (env env-name default)]
    (-> doc-map
        (assoc :value
               (if e-val (if masked? ::masked v)
                         (if optional? "nil" ::missing)))
        (rename-keys {:env-name :name}))))

(defn- get-var-statuses [display-spec] (map get-var-status (vals display-spec)))

(defn- convert [v]
  (let [parsed? (vector? v)
        parse-difference? (and parsed? (not= (first v) (second v)))
        displayable (if parsed? (second v) v)
        displayable
        (if-let [replacement (displays displayable)] replacement displayable)]
    (if parse-difference? (format "\"%s\" -> %s" (first v) displayable)
                          displayable)))

(defn display-spec->docs
  "Convert a display spec into a collection of objects suitable for
  displaying documentation"
  [display-spec]
  (map #(update % :value convert) (get-var-statuses display-spec)))

(defn spec->docs
  "DEPRECATED in favor of parse-bool"
  {:deprecated "1.0.11"} [display-spec]
  (display-spec->docs display-spec))

(defn doc-table
  "Convert a connection of spec docs into a nice table"
  [env-docs]
  (with-out-str (print-table [:name :value :doc] env-docs)))

(defn- make-usage-string [display-spec prefix]
  (let [env-docs (display-spec->docs display-spec)]
    (if (empty? env-docs)
      "No environment variables defined!"
      (str prefix "\n" (doc-table env-docs)))))

;; ### Display

(def ^:dynamic ^:private print-usage? (atom false))

;; Enable user-friendly usage printing globally.
(defn set-error-print-enabled! [enabled?] (reset! print-usage? enabled?))

;; Enable user-friendly usage printing in the current thread.
(defmacro with-usage-print-enabled [& body]
  `(binding [print-usage? (delay true)] ~@body))

(defn- print-to-stderr [msg] (binding [*out* *err*] (println msg)))
(def ^:dynamic ^:private epfn (atom print-to-stderr))

(def ^:private missing "Environment variable(s) missing")

(defn print-usage [pfn message display-spec]
  (pfn (make-usage-string display-spec (str message ":"))))

;; Send user-friendly error messages somewhere else universally.
(defn set-err-print-fn! [f] (reset! epfn f))

;; Send user-friendly error messages somewhere else in the current thread.
(defmacro with-print-fn [f & body] `(binding [epfn (delay ~f)] ~@body))

(defn- throw-usage-if-missing [display-spec]
  (let [missing-vals
        (->> (get-var-statuses display-spec)
             (filter #(let [{:keys [value]} %
                            [_ parsed-v] (when (vector? value) value)]
                        (case (or parsed-v value)
                          ::missing true
                          ::parse-error true
                          false)))
             (map :name))]
    (when (seq missing-vals)
      (let [missing-msg
            (format "%s: %s" missing (str/join ", " missing-vals))]
        (when @print-usage? (print-usage @epfn missing display-spec))
        (throw+ {:type ::missing-env :missing missing-vals} missing-msg)))))

;; ## Core functionality

;; ### Retrieving environment variable values

(defn- parse-env [env-name {:keys [default tfn optional?] :as params}]
  (let [has-param? (partial contains? params)
        env-args [env-name]
        base-value (env env-name default)]
    {:tfn (if (and (not (nil? base-value))
                   (has-param? :tfn)) tfn
                                      identity)
     :env-args (if (or optional? (has-param? :default))
                 (conj env-args default) env-args)
     :params-to-display (assoc params :env-name env-name)
     :base-value base-value}))

(defn- pretty-demunge
  [fn-object]
  (let [demunged (repl/demunge (str fn-object))
        pretty (second (re-find #"(.*?\/.*?)@.*" demunged))]
    (or pretty demunged)))

(defn- try-tfn [env-name masked? tfn base-value]
  (try (tfn base-value)
       (catch Exception e
         (@epfn (format "Unable to parse %s='%s' using %s (%s)."
                        env-name
                        (if masked? (::masked displays) base-value)
                        (-> tfn str pretty-demunge)
                        (if masked? "--ERROR MASKED--" (.getMessage e))))
         ::parse-error)))

(defn- ->env-name [k]
  (-> k
      name
      str/upper-case
      (str/replace #"-" "_")
      (str/replace #"[^A-Z0-9_]" "")))

(defn- overlay-env [m k raw-params]
  (let [{:keys [optional? masked? env-name env] :as params}
        (merge {:masked? true} raw-params)

        env-name (or env-name env (->env-name k))
        {:keys [tfn params-to-display base-value]} (parse-env env-name params)
        v (try-tfn env-name masked? tfn base-value)]
    (-> m
        (update :env-map #(if (and (not optional?) (nil? v)) % (assoc % k v)))
        (update :display-spec assoc env-name
                (assoc params-to-display :v [base-value v])))))

(defn- get-env-info [env-spec]
  (reduce-kv overlay-env
             {:env-map (sorted-map), :display-spec (sorted-map)}
             (into (sorted-map) env-spec)))

(defn env->map
  "Operates much like `defenv` except you can define multiple bindings at once,
and receive a map of all values that have been found in the environment.
Unlike `defenv `, however, this function will throw an exception if any
required variable is missing, not just the one you ask for.

The map should look something like this:

    {:my-env-var {:tfn my-optional-parse-function
                  :default \"MY OPTIONAL DEFAULT VALUE\"
                  :masked? true
                  :doc \"Nice documentation\"
                  :optional? true}
     :path nil
     :other {:env-name \"OTHER_VAR\"}}

In this case, the `:my-env-var` key will be filled in with the value of
`MY_ENV_VAR`, unless it isn't present, in which case, it will receive the
value `MY OPTIONAL DEFAULT VALUE`.

Every key is optional. If :env-name is missing, we'll use the key in the map
to infer the name of the environment variable by stringifying it, uppercasing it,
and replacing dashes with underscores.

In the case of `:other`, if there is no value for `OTHER_VAR`, there will be
an exception thrown much like when attempting to deref a binding generated
by `defenv `that is required but missing."
  [env-spec]
  (let [{:keys [env-map display-spec]} (get-env-info env-spec)]
    (throw-usage-if-missing display-spec)
    env-map))

(defn one
  "Get a single environment variable value. You can use any of the params
  that you would use in `env->map` except `:env-name`, which is the `env-name`."
  [env-name & params]
  (:e (env->map {:e (into {:env-name env-name} (apply hash-map params))})))

;; ### Defining a global environment variable binding

(def ^:private global-defined-spec (ref (sorted-map)))
(def ^:private global-parsed-env (ref (sorted-map)))
(def ^:private global-display-spec (ref (sorted-map)))
(def ^:private on-load-handlers (atom []))

(defmacro ^:private with-new-parsed-env! [& body]
  `(dosync ~@body (alter global-parsed-env empty)))

(defn- initialize-global! []
  (when (empty? @global-parsed-env)
    (let [{:keys [env-map display-spec]} (get-env-info @global-defined-spec)]
      (dosync
       (ref-set global-parsed-env env-map)
       (ref-set global-display-spec display-spec))
      (doall (map #(% @global-display-spec) @on-load-handlers)))))

(defn on-load! [f] (swap! on-load-handlers conj f))

(defmacro ^:private guarantee-global! [& body]
  `(do (initialize-global!) ~@body))

(defn- add-doc [doc-present? doc-or-env params]
  (if doc-present? (assoc params :doc doc-or-env) params))

(defn get-global-env
  "Used primarily by `defenv` to retrieve the global environment state. "
  [env-name]
  (guarantee-global!
   (if ((set (keys @global-parsed-env)) env-name)
     (get @global-parsed-env env-name)
     (throw-usage-if-missing @global-display-spec))))

(defn add-to-global-defined-spec!
  "Used primarily by `defenv` to add to the global environment state. "
  [env-name params]
  (with-new-parsed-env! (alter global-defined-spec assoc env-name params)))

(defmacro defenv
  "Define a binding `b` to an environment variable. Creates a delayed
object that, when dereferenced, will load an environment variable of the
given key.

If `doc-or-env` and `env-or-fk` are both strings, we assume that
`doc-or-env` is a docstring and `env-or-fk` is the environment variable
to be pulled. Then, `remaining` become the `params`.

If `doc-or-env` is a keyword or nil, we assume that `b` should be
used to infer the name of the environment variable. We uppercase the symbol,
and replace dashes with underscores. So something like

`(defenv :fun-times)`

would be the same as

`(defenv :fun-times \"FUN_TIMES\")`

Similarly:

`(defenv :fun-times :tfn parse-int)`

would be identical do

`(defenv :fun-times \"FUN_TIMES\" :tfn parse-int)`

Unfortunately, this means we have trouble inferring documentation
strings. So, if you want to add documentation, you need to:

`(defenv :fun-times :doc \"Some fun variable\")`

Else, we assume `doc-or-env` is the environment variable and
we use `env-or-fk` as the first key of the `params`. This
convention also allows for your documentation generator
(like <https:// github.com / gdeer81/marginalia>)
to detect docstrings in their conventional position.

If `:default {value}` shows up in the params, will send back the
given value if the variable isn't defined.

If `:tfn {function}` shows up in the params, runs the given function against
the result of the environment variable load, which is by default a string. Best
not to use a lambda, as you won't get any helpful context on parse errors.

If you add `:masked?` and set it to `true`, the value won't be
displayed in usage documentation.

If you add `:optional?` and set it to `true`, then there need not
be a default value set, and the `tfn` will not get invoked if the value is
missing."
  [b & [doc-or-env env-or-fk & remaining]]
  (let [infer-name-from-symbol? (or (nil? doc-or-env) (keyword? doc-or-env))
        doc-present? (every? string? [doc-or-env env-or-fk])
        env-name (cond doc-present? env-or-fk
                       infer-name-from-symbol? (->env-name b)
                       :default doc-or-env)
        params (->> (cond doc-present? remaining

                          infer-name-from-symbol?
                          (when (and doc-or-env env-or-fk)
                            (concat [doc-or-env env-or-fk] remaining))

                          :default
                          (when env-or-fk (concat [env-or-fk] remaining)))
                    (apply hash-map)
                    (add-doc doc-present? doc-or-env)
                    (merge {:env-name env-name}))]
    `(do (add-to-global-defined-spec! ~env-name ~params)
         (def ~(vary-meta b assoc :dynamic true)
           (delay (get-global-env ~env-name))))))

;; ## Displaying environment information to your users

(defn display-spec
  "From an environment specification (or nil, if using the global),
   construct a specification for how to display the environment
   configuration."
  ([] (display-spec nil))
  ([env-spec]
   (if env-spec
     (:display-spec (get-env-info env-spec))
     (guarantee-global! @global-display-spec))))

(defn- display-env-internal [display-spec display-fn]
  (print-usage display-fn "Environment" display-spec))

;; A recipe for doing your own display of environment information
(comment
 (let [test-spec {:stuff {:env-name "STUFF" :default "bits" :doc "fun"}

                   :answer
                   {:env-name "ANSWER"
                    :default "42"
                    :tfn parse-int
                    :doc "ultimate"}

                   :path {:env-name "PATH"
                          :tfn (fn [v] (str/split v #"[:;]"))
                          :masked? true
                          :doc "System path!"}}]

   (println "\nInternal use:")
   (->> test-spec env->map clojure.pprint/pprint)
   (println "\nExternal view:")
   (-> test-spec
       display-spec
       display-spec->docs
       doc-table
       println))
 )

(defn display-env
  "Display the current environment to users in a friendly manner. If you call
  this function without an `env-spec`, we will print the result of documentation
  from all the `defenv` calls that have been executed in all the referred
  namespaces. Otherwise, it will be based on the spec given. This is the same
  format as you would send to `env->map`.

  Also, in case you want to send your own function (instead of println), you
  can call the 2-argument alternative. `out-fn` will be executed with a string
  representation of the given environment. If `env-spec` is nil, we will use
  the global environment (modified using `defenv`)."
  ([]
   (display-env nil println))
  ([env-spec]
   (display-env env-spec println))
  ([env-spec out-fn]
   (-> env-spec display-spec (display-env-internal out-fn))))

(defn extract-global-spec
  "Extract the global environment config as a list of maps"
  [f]
  (guarantee-global! (map (comp f second) @global-defined-spec)))

;; ## Test fixtures

(defn reset-defined-env!
  []
  (with-new-parsed-env!
   (alter global-display-spec empty)
   (alter global-defined-spec empty)))
