;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

;; # Usage example

(ns defenv.usage
  (:require [clojure.string :as str]
            [defenv.core :as env]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log])
  (:gen-class)
  (:import (clojure.lang ExceptionInfo)))

;; ## Environment variables

(def sensible-default "A Sensible Default™")

;; ### Global Bindings

(env/defenv testing
  "An environment variable with a default value."
  "DEFENV_TESTING" :default sensible-default)

(env/defenv missing
  "Shows you what happens when something is missing."
  "DEFENV_MISSING")

(env/defenv missing-no-docs "DEFENV_MISSING_NO_DOCS")

(env/defenv some-keyword
  "Shows how values can be converted to keywords."
  "DEFENV_KEYWORD" :default "test" :tfn keyword)

(env/defenv secret-thing
  "Shows what happens when you mask a var."
  "DEFENV_SECRET" :default "oops" :masked? true)

(env/defenv truly-optional
  "A truly optional value."
  "DEFENV_OPT" :tfn env/parse-int :optional? true)

(defn parse-broken [x]
  (throw (ExceptionInfo. (str "I refuse to parse: " x) {})))

(env/defenv secret-parse-error
  "Something secret that can't be parsed."
  "DEFENV_UNPARSEABLE_SECRET"
  :tfn parse-broken :default "secret" :masked? true)

(env/defenv parse-error "Something that can't be parsed." "DEFENV_UNPARSEABLE"
  :tfn parse-broken :default "broken")

;; ### Local Map

(def env-map-spec {:testing {:env-name "DEFENV_TESTING"
                             :default sensible-default}
                   :log-level {:env-name "LOG_LEVEL" :doc "Global log level."
                               :tfn keyword :default "info"}
                   :should-log? {:env-name "SHOULD_LOG"
                                 :doc "Should I log? A boolean."
                                 :tfn env/parse-bool :default "false"}
                   :optional {:env-name "DEFENV_OPT" :optional? true
                              :doc "A truly optional value."}})
(def env-map (env/env->map env-map-spec))

(defmacro handle-exception [& body]
  `(try+
     (printf "this should die: %s%n" ~@body)
     (catch [:type :defenv.core/missing-env] {missing# :missing}
       (println "Missing:" (str/join ", " missing#)))))

(defn -main
  "A very simple example of how to use the library. Just run `lein usage`."
  [& _]
  (println "* Display usage without trying to access a variable *")

  (env/on-load! (fn [_] (println "** Global Bindings On-Load **")))
  (env/on-load! (partial env/print-usage println "Environment"))

  (println "\n** Local Map **")
  (println env-map)
  (env/display-env env-map-spec)

  (println "\n** Single values **")
  (println (env/one "PATH"))
  (handle-exception (env/one "MISSING_SECRET" :masked? true :tfn env/parse-int))

  (println "\n* Error Handling *")
  (env/set-error-print-enabled! true)
  (env/set-err-print-fn! #(log/error %))

  (printf "Exception Printing: %s%n" @testing)
  (handle-exception @missing))