;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

;; # Usage example

(ns defenv.usage
  (:require [clojure.string :as str]
            [defenv.core :as env]
            [slingshot.slingshot :refer [try+]])
  (:gen-class))

;; ## Environment variables

(def sensible-default "A Sensible Default™")

(env/defenv testing
  "An environment variable with a default value."
  "DEFENV_TESTING" :default sensible-default)

(env/defenv missing
  "Shows you what happens when something is missing."
  "DEFENV_MISSING")

(env/defenv some-keyword
  "Shows how values can be converted to keywords."
  "DEFENV_KEYWORD" :default "test" :tfn keyword)

(env/defenv secret-thing
  "Shows what happens when you mask a var."
  "DEFENV_SECRET" :default "oops" :masked? true)

(env/defenv truly-optional
  "A truly optional value."
  "DEFENV_OPT" :tfn env/parse-int :optional? true)

(def env-map-spec {:testing {:env "DEFENV_TESTING" :default sensible-default}
                   :log-level {:env "LOG_LEVEL" :doc "Global log level."
                               :tfn keyword :default "info"}
                   :should-log? {:env "SHOULD_LOG"
                                 :doc "Should I log? A boolean."
                                 :tfn env/parse-bool :default "false"}
                   :optional {:env "DEFENV_OPT" :optional? true
                              :doc "A truly optional value."}})
(def bad-spec (assoc env-map-spec :missing {:env "DEFENV_MISSING"}))
(def env-map (env/env->map env-map-spec))

(defmacro handle-exception [& body]
  `(try+
     (printf "this should die: %s%n" ~@body)
     (catch [:type :defenv.core/missing-env] {missing# :missing}
       (println "Missing:" (str/join ", " missing#)))))

(defn -main
  "A very simple example of how to use the library."
  [& _]
  (env/set-usage-print-enabled! true)

  (println "** Global Bindings **")
  (printf "testing: %s%n" @testing)
  (handle-exception @missing)

  (println "\n** Environment Map **")
  (println env-map)
  (handle-exception (env/env->map bad-spec)))