;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

;; # Welcome to defenv
(ns defenv.core
  (:require [clojure.pprint :refer :all]
            [clojure.set :refer :all]))

(def defined-env (atom (sorted-map)))

;; ## Parsing convenience functions
(defn parse-bool [s] (Boolean/parseBoolean s))
(defn parse-int [s] (Integer/parseInt s))
(defn parse-long [s] (Long/parseLong s))
(defn parse-double [s] (Double/parseDouble s))
(defn parse-float [s] (Float/parseFloat s))

(defn parse-boolean
  "DEPRECATED in favor of parse-bool"
  {:deprecated "0.0.6"}
  [s] (parse-bool s))

;; ## Test fixtures
(defn reset-defined-env []
  (swap! defined-env empty))

;; ## On-the-fly documentation generation

(declare env)

(defn- render-doc [{:keys [e-name v default masked?] :as doc-map}]
  (let [e-val (env e-name default)]
    (-> doc-map
        (assoc :value (if e-val (if masked? "--MASKED--" @v) "*MISSING*"))
        (rename-keys {:e-name :name}))))

(defn make-usage-string
  "Display nice documentation telling folks which environment variables need
   to be set, and what the current values are. Please note that using this
   function within a main function that doesn't link to any namespaces that
   define environment variables will cause the library to tell you there are
   no environment variables defined. This is due to Clojure's lazy loading
   of namespaces."
  [prefix]
  (let [env-docs (map render-doc (vals @defined-env))
        doc-str (with-out-str (print-table [:name :value :doc] env-docs))]
    (if (empty? env-docs)
      "No environment variables defined!"
      (str prefix "\n" doc-str))))

(defn- throw-usage []
  (throw (ex-info (make-usage-string
                    "One or more environment variables is missing:") {})))

;; ## Core functionality

(defn- env-or [env-key f]
  (let [val (System/getenv env-key)]
    (if-not val (f) val)))

(defn env
  "### Retrieving an environment variable on-the-fly
   If no default value is set and the variable is not present, an exception will
   be thrown displaying which variables are missing and which are set.

   Using this function in your code is not recommended, as users will not have
   any way of knowing they missed an environment variable. Of course, you could
   use it if you want there to be secret variables to enable various things?"
  ([env-key] (env-or env-key throw-usage))
  ([env-key default-value] (env-or env-key #(identity default-value))))

(defmacro defenv
  "### Defining an environment variable binding

   Define a binding `b` to an environment variable. Creates a delayed
   object that, when dereferenced, will load an environment variable of the
   given key.

   If `doc-or-env` and `env-or-fk` are both strings, we assume that
   `doc-or-env` is a docstring and `env-or-fk` is the environment variable
   to be pulled. Then, `remaining` become the `params`.

   Else, we assume `doc-or-env` is the environment variable and
   we use `env-or-fk` as the first key of the `params`. This
   convention also allows for your documentation generator
   (like <https://github.com/gdeer81/marginalia>)
   to detect docstrings in their conventional position.

   If `:default {value}` shows up in the params, will send back the
   given value if the variable isn't defined.

   If `:tfn {function}` shows up in the params, runs the given function against
   the result of the environment variable load, which is by default a string.

   If you add a `:masked?` keyword and set it to `true`, the value won't be
   displayed in usage documentation."
  [b & [doc-or-env env-or-fk & remaining]]
  (let [doc-present? (every? string? [doc-or-env env-or-fk])
        env-name (if doc-present? env-or-fk doc-or-env)
        params (if doc-present? remaining (concat [env-or-fk] remaining))
        {:keys [default tfn] :as params} params
        params (if doc-present? (assoc params :doc doc-or-env) params)
        has-param? (partial contains? params)
        tfn (if (has-param? :tfn) tfn 'identity)
        env-args [env-name]
        env-args (if (has-param? :default) (conj env-args default) env-args)
        params-to-display (assoc params :e-name env-name)]
    `(let [v# (delay (~tfn (env ~@env-args)))]
       (swap! defined-env assoc ~env-name (assoc ~params-to-display :v v#))
       (def ~b v#))))