;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

(ns defenv.core
  (:require [clojure.pprint :refer :all]
            [clojure.set :refer :all]))

(def defined-env (atom (sorted-map)))

(defn- env-or [env-key f]
  (let [val (System/getenv env-key)]
    (if-not val (f) val)))

(declare env)

(defn- render-docs [{:keys [e-name default masked?] :as doc-map}]
  (let [e-val (env e-name default)]
    (-> doc-map
        (assoc :value (if e-val (if masked? "--MASKED--" e-val) "*MISSING*"))
        (dissoc :default)
        (rename-keys {:e-name :name}))))

(defn make-usage-string
  "Display nice documentation telling folks which environment variables need
   to be set, and what the current values are."
  []
  (let [env-docs (map render-docs (vals @defined-env))
        doc-str (with-out-str (print-table [:name :value :doc] env-docs))]
    (str "One or more environment variables is missing:\n"
         doc-str)))

(defn- throw-usage [] (throw (ex-info (make-usage-string) {})))

(defn env
  "Attempt to retrieve an environment variable. If no default value is set
   and the variable is not present, an exception will be thrown displaying
   which variables are missing and which are set."
  ([env-key] (env-or env-key throw-usage))
  ([env-key default-value] (env-or env-key #(identity default-value))))

(defmacro defenv
  "Define a binding to an environment variable. Creates a delayed object
   that, when dereferenced, will load an environment variable of the given
   key. If ':default {value}' shows up in the params, will send back the
   given value if the variable isn't defined. If ':tfn {function}' shows up
   in the params, runs the given function against the result of the environment
   variable load, which is by default a string. If you add a :doc keyword,
   it will be displayed whenever someone is missing the environment variable
   as helpful documentation to display to the user. If you add a :masked?
   keyword and set it to false, the value won't be displayed in usage
   documentation."
  [b env-name & {:keys [default tfn] :as params}]
  (let [has-param? (partial contains? params)
        tfn (if (has-param? :tfn) tfn 'identity)
        env-args [env-name]
        env-args (if (has-param? :default) (conj env-args default) env-args)
        params-to-display (-> params (dissoc :tfn) (assoc :e-name env-name))]
    `(do
       (swap! defined-env assoc ~env-name ~params-to-display)
       (def ~b (delay (~tfn (env ~@env-args)))))))