(ns conwip.modules
  (:require [cljs.core :as cc]
            [cljs.env]))

(def info {:preloads '[conwip.test.dev]
           :main 'conwip.test.runner
           :modules {:extra {:output-to "resources/prod_test/js/extra.js"
                             :entries #{"conwip.test.extra"}}
                     :dev {:output-to "resources/prod_test/js/test.js"
                           :entries #{"conwip.test.runner"}
                           :depends-on #{:extra}}}
           :asset-path "js"})

(defn module-ns
  "Gets a sequence of all the module entries turned into symbols

   (def modules
      :modules {:extra {:output-to \"path/to/extra.js\"
                        :entries #{\"my.app.extra\"}}
                :dev {:output-to \"path/to//test.js\"
                      :entries #{\"my.app.core\"}
                       :depends-on #{:extra}}})

   (modules-ns module)
   ; => (my.app.core my.app.extra)"
  [compiler-opts]
  (->> compiler-opts :modules vals
    (mapcat :entries)
    (remove nil?)
    (map symbol)))

(defn modules-preloads
  "Give all the modules namespaces needed to load into :preloads during development
  for dynamic module loading to work the same as it does in prdduction

  (def compiler-options
    {:preloads '[conwip.test.dev]
           :main 'my..runner
           :modules {:extra {:output-to \"path/to/extra.js\"
                             :entries #{\"my.app.extra\"}}
                     :dev {:output-to \"path/to//test.js\"
                           :entries #{\"my.app.core\"}
                            :depends-on #{:extra}}}})
  "
  [compiler-opts]
  (let [preloads (concat (:preloads compiler-opts) (module-ns compiler-opts))]
    (-> preloads
        set
        (disj (:main compiler-opts))
        vec)))

(defmacro env-module-uris
  "Gets the modules URI loaction from the compiler option
  :module-info {:moudle/uris <module-uris>} and turns it into
  a JavaScript object for ClojureScript consumption

  For example

  :module-info {:module/uris {\"dev\"   \"js/dev.js\"
                              \"extra\" \"js/extra.js\"}}

  is transformed into

  #js {\"dev\"   \"js/dev.js\"
       \"extra\" \"js/extra.js\"}"
  []
  (let [module-uris (get-in @cljs.env/*compiler* [:options :module-info :module/uris])]
    `(cc/js-obj ~@(mapcat (fn [[module uri]] [module uri]) module-uris))))

(defmacro env-module-deps []
  "Gets the modules dependency information from the compiler option
  :module-info {:moudle/deps <module-deps>} and turns it into a
  JavaScript object for ClojureScript consumption

  For example

  :module-info {:module/deps {\"dev\"    []
                              \"tools\"  []
                              \"hammer\" [\"tools\"]}

  is transformed into

  #js {\"dev\"    #js []
       \"tools\"  #js []
       \"hammer\" #js[\"tools\"]}"
  (let [module-deps (get-in @cljs.env/*compiler* [:options :module-info :module/deps])]
    `(cc/js-obj ~@(mapcat (fn [[module deps]] [module `(cc/array ~@deps)]) module-deps))))