(ns common-foxeriot.http-client
  (:require [aleph.http :as http]
            [cheshire.core :as json]
            [clojure.java.io :as io]
            [taoensso.timbre :as log]
            [manifold.deferred :as d])
  (:import (com.fasterxml.jackson.core JsonParseException)))

(defn connection-pool [] (http/connection-pool {}))

(defn report-exception [api-name url e]
  "Report an exception from Aleph HTTP request.

  Parameters
  api-name  -- human-readable name of the API called
  url       -- the request URL
  e         -- the exception."
  (let [{:keys [body status] :as data} (ex-data e)
        body (let [body-str (slurp body)]
               (try
                 (json/parse-string body-str true)
                 (catch JsonParseException json-e
                   (binding [log/*context* (assoc log/*context*
                                                  :request-url url
                                                  :body body-str)]
                     (log/warnf "Error parsing JSON from %s: %s %s" api-name (.getMessage json-e) status)
                     (throw e)))))]
    (log/warnf "%s call failure. %s %s %s" api-name url status (pr-str body))
    (throw (ex-info (.getMessage e) (assoc data :body body)))))

(def ^:private +default-timeout-ms+ 10000)

(def ^:private +default-request-options+
  "Default options for http/request."
  {:as                 :json
   :throw-exceptions   true
   ;; Each timeout is separate. If each timeout is 10s, the whole
   ;; request can take up to 30s.
   :pool-timeout       +default-timeout-ms+
   :connection-timeout +default-timeout-ms+
   :request-timeout    +default-timeout-ms+})

(defn request
  "Like aleph.http/request, but with our own error reporting."
  [api-name options]
  (let [url (:url options)]
    (d/chain (d/catch
              (http/request (merge +default-request-options+ options))
              clojure.lang.IExceptionInfo
               (partial report-exception api-name url))
             :body)))

(defn many-requests
  "Like request, but over sequence of many request options."
  [api-name many-options]
  (apply d/zip (map #(request api-name %) many-options)))
