(ns conceit.commons.restart)

(defn parse-line [l]
  (restartable
   [(use-value [value] value)]
   (if-let [[name pass host] (re-seq #"^(.+?):(.+?)@(.+?)$" l)]
     {:name name :pass pass :host host}
     (throw (Exception. (str "Can't parse the line: " l))))))

(defn parse-lines [lines]
  (map parse-line lines))

(defn do-parse []
  (with-restart-handlers
    [(Exception e (invoke-restart :use-value {:name "Unknown" :pass "" :host "Unknown"}))]
    (parse-lines ["foo:bar@hoge" "aaaaaa" "a:b@c"])))

(def ^{:dynamic true} *restarts* {})

(def ^{:dynamic true} *restart-handlers* [])

(defn invoke-restart [restart & args]
  (throw (RestartInvocation. restart args)))

(defmacro with-restart-handlers [[& handlers] & body]
  `(binding [*restart-handlers* (concat ~(map [[throwable-class var-name & body]]
                                              `[~throwable-class (fn [~var-name] ~@body)]
                                              handlers)
                                        *restart-handlers* )]
     ~@body))

(defn find-restart-handler [throwable]
  (some (fn [[throwable-class handler]]
          (if (instance? throwable-class throwable) handler nil))
        *restart-handlers*))

(defn invoke-restart-handler [throwable]
  (when-let [handler (find-restart-handler throwable)]
    (handler throwable))
  (throw throwable))

(defmacro restartable [[& restarts] & body]
  `(try
     (do ~@body)
     (catch Throwable t# (invoke-restart-handler t#))))
