;; gorilla-repl.fileformat = 1

;; **
;;; # Graphing tests
;;; 
;;; A showcase of gorilla-plot graph types. Here I load both the vega and core namespaces. This is useful for development, but if you're just using the library then you only need to load the 'gorilla-plot.core namespace.
;; **

;; @@
(use 'gorilla-plot.vega :reload)
(use 'gorilla-plot.core :reload)
(use 'clojure.pprint)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Some data to look at. And some gratuitous mathematics @@x = 5@@
;;; 
;;; $$a_i = \int_0^{l_i} \sin(x) dx$$
;; **

;; @@
(def d
  [1 3 5 7 9 5 4 6 9 8 3 5 6])
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;user/d</span>","value":"#'user/d"}
;; <=

;; **
;;; List plot plots data in the style of a scatter plot by default.
;; **

;; @@
(list-plot d)
;; @@
;; =>
;;; {"type":"vega","content":{"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"cb6175a5-91be-4816-a8c7-b5c27e18fc46","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"cb6175a5-91be-4816-a8c7-b5c27e18fc46","field":"data.y"}}],"marks":[{"type":"symbol","from":{"data":"cb6175a5-91be-4816-a8c7-b5c27e18fc46"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"fill":{"value":"steelblue"},"fillOpacity":{"value":1}},"update":{"shape":"circle","size":{"value":70},"stroke":{"value":"transparent"}},"hover":{"size":{"value":210},"stroke":{"value":"white"}}}}],"data":[{"name":"cb6175a5-91be-4816-a8c7-b5c27e18fc46","values":[{"x":0,"y":1},{"x":1,"y":3},{"x":2,"y":5},{"x":3,"y":7},{"x":4,"y":9},{"x":5,"y":5},{"x":6,"y":4},{"x":7,"y":6},{"x":8,"y":9},{"x":9,"y":8},{"x":10,"y":3},{"x":11,"y":5},{"x":12,"y":6}]}],"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50}},"value":"#gorilla_repl.vega.VegaView{:content {:axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"cb6175a5-91be-4816-a8c7-b5c27e18fc46\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"cb6175a5-91be-4816-a8c7-b5c27e18fc46\", :field \"data.y\"}}], :marks [{:type \"symbol\", :from {:data \"cb6175a5-91be-4816-a8c7-b5c27e18fc46\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 1}}, :update {:shape \"circle\", :size {:value 70}, :stroke {:value \"transparent\"}}, :hover {:size {:value 210}, :stroke {:value \"white\"}}}}], :data [{:name \"cb6175a5-91be-4816-a8c7-b5c27e18fc46\", :values ({:x 0, :y 1} {:x 1, :y 3} {:x 2, :y 5} {:x 3, :y 7} {:x 4, :y 9} {:x 5, :y 5} {:x 6, :y 4} {:x 7, :y 6} {:x 8, :y 9} {:x 9, :y 8} {:x 10, :y 3} {:x 11, :y 5} {:x 12, :y 6})}], :width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}}}"}
;; <=

;; **
;;; There are a number of options that can be given to the plot. At the moment your best bet is to peak at the source code to see what they are.
;; **

;; @@
(list-plot d :joined true :plot-size 600 :aspect-ratio 6 :opacity 0.5 :plot-range [:all :all])
;; @@
;; =>
;;; {"type":"vega","content":{"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"39d7833b-49eb-45df-998a-531c956f4d84","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"39d7833b-49eb-45df-998a-531c956f4d84","field":"data.y"}}],"marks":[{"type":"line","from":{"data":"39d7833b-49eb-45df-998a-531c956f4d84"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"#FF29D2"},"strokeWidth":{"value":2},"strokeOpacity":{"value":0.5}}}}],"data":[{"name":"39d7833b-49eb-45df-998a-531c956f4d84","values":[{"x":0,"y":1},{"x":1,"y":3},{"x":2,"y":5},{"x":3,"y":7},{"x":4,"y":9},{"x":5,"y":5},{"x":6,"y":4},{"x":7,"y":6},{"x":8,"y":9},{"x":9,"y":8},{"x":10,"y":3},{"x":11,"y":5},{"x":12,"y":6}]}],"width":600,"height":100.0,"padding":{"bottom":20,"top":10,"right":10,"left":50}},"value":"#gorilla_repl.vega.VegaView{:content {:axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"39d7833b-49eb-45df-998a-531c956f4d84\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"39d7833b-49eb-45df-998a-531c956f4d84\", :field \"data.y\"}}], :marks [{:type \"line\", :from {:data \"39d7833b-49eb-45df-998a-531c956f4d84\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"#FF29D2\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 0.5}}}}], :data [{:name \"39d7833b-49eb-45df-998a-531c956f4d84\", :values ({:x 0, :y 1} {:x 1, :y 3} {:x 2, :y 5} {:x 3, :y 7} {:x 4, :y 9} {:x 5, :y 5} {:x 6, :y 4} {:x 7, :y 6} {:x 8, :y 9} {:x 9, :y 8} {:x 10, :y 3} {:x 11, :y 5} {:x 12, :y 6})}], :width 600, :height 100.0, :padding {:bottom 20, :top 10, :right 10, :left 50}}}"}
;; <=

;; **
;;; You can also plot bar charts. The first argument is the list of category names, the second the category values.
;; **

;; @@
(bar-chart (range (count d)) d)
;; @@
;; =>
;;; {"type":"vega","content":{"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":{"data":"cb576eed-a95e-4cfa-a678-db97fbf8f6c1","field":"data.x"}},{"name":"y","range":"height","nice":true,"domain":{"data":"cb576eed-a95e-4cfa-a678-db97fbf8f6c1","field":"data.y"}}],"marks":[{"type":"rect","from":{"data":"cb576eed-a95e-4cfa-a678-db97fbf8f6c1"},"properties":{"enter":{"width":{"offset":-1,"band":true,"scale":"x"},"y":{"field":"data.y","scale":"y"},"x":{"field":"data.x","scale":"x"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"data":[{"name":"cb576eed-a95e-4cfa-a678-db97fbf8f6c1","values":[{"x":0,"y":1},{"x":1,"y":3},{"x":2,"y":5},{"x":3,"y":7},{"x":4,"y":9},{"x":5,"y":5},{"x":6,"y":4},{"x":7,"y":6},{"x":8,"y":9},{"x":9,"y":8},{"x":10,"y":3},{"x":11,"y":5},{"x":12,"y":6}]}],"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50}},"value":"#gorilla_repl.vega.VegaView{:content {:axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain {:data \"cb576eed-a95e-4cfa-a678-db97fbf8f6c1\", :field \"data.x\"}} {:name \"y\", :range \"height\", :nice true, :domain {:data \"cb576eed-a95e-4cfa-a678-db97fbf8f6c1\", :field \"data.y\"}}], :marks [{:type \"rect\", :from {:data \"cb576eed-a95e-4cfa-a678-db97fbf8f6c1\"}, :properties {:enter {:width {:offset -1, :band true, :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :x {:field \"data.x\", :scale \"x\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :data [{:name \"cb576eed-a95e-4cfa-a678-db97fbf8f6c1\", :values ({:x 0, :y 1} {:x 1, :y 3} {:x 2, :y 5} {:x 3, :y 7} {:x 4, :y 9} {:x 5, :y 5} {:x 6, :y 4} {:x 7, :y 6} {:x 8, :y 9} {:x 9, :y 8} {:x 10, :y 3} {:x 11, :y 5} {:x 12, :y 6})}], :width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}}}"}
;; <=

;; **
;;; It's interesting to look at the structure of a plot.
;; **

;; @@
(def p (list-plot d))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;user/p</span>","value":"#'user/p"}
;; <=

;; **
;;; It's nothing but a clojure value, which is interpreted by Gorilla as a Vega visualisation and rendered appropriately.
;; **

;; @@
(pprint p)
;; @@
;; ->
;;; {:content
;;;  {:axes [{:type &quot;x&quot;, :scale &quot;x&quot;} {:type &quot;y&quot;, :scale &quot;y&quot;}],
;;;   :scales
;;;   [{:name &quot;x&quot;,
;;;     :type &quot;linear&quot;,
;;;     :range &quot;width&quot;,
;;;     :zero false,
;;;     :domain
;;;     {:data &quot;90e8e8e5-08a3-4587-bb5f-ba3d6204af1e&quot;, :field &quot;data.x&quot;}}
;;;    {:name &quot;y&quot;,
;;;     :type &quot;linear&quot;,
;;;     :range &quot;height&quot;,
;;;     :nice true,
;;;     :zero false,
;;;     :domain
;;;     {:data &quot;90e8e8e5-08a3-4587-bb5f-ba3d6204af1e&quot;, :field &quot;data.y&quot;}}],
;;;   :marks
;;;   [{:type &quot;symbol&quot;,
;;;     :from {:data &quot;90e8e8e5-08a3-4587-bb5f-ba3d6204af1e&quot;},
;;;     :properties
;;;     {:enter
;;;      {:x {:field &quot;data.x&quot;, :scale &quot;x&quot;},
;;;       :y {:field &quot;data.y&quot;, :scale &quot;y&quot;},
;;;       :fill {:value &quot;steelblue&quot;},
;;;       :fillOpacity {:value 1}},
;;;      :update
;;;      {:shape &quot;circle&quot;,
;;;       :size {:value 70},
;;;       :stroke {:value &quot;transparent&quot;}},
;;;      :hover {:size {:value 210}, :stroke {:value &quot;white&quot;}}}}],
;;;   :data
;;;   [{:name &quot;90e8e8e5-08a3-4587-bb5f-ba3d6204af1e&quot;,
;;;     :values
;;;     ({:x 0, :y 1}
;;;      {:x 1, :y 3}
;;;      {:x 2, :y 5}
;;;      {:x 3, :y 7}
;;;      {:x 4, :y 9}
;;;      {:x 5, :y 5}
;;;      {:x 6, :y 4}
;;;      {:x 7, :y 6}
;;;      {:x 8, :y 9}
;;;      {:x 9, :y 8}
;;;      {:x 10, :y 3}
;;;      {:x 11, :y 5}
;;;      {:x 12, :y 6})}],
;;;   :width 400,
;;;   :height 247.2188,
;;;   :padding {:bottom 20, :top 10, :right 10, :left 50}}}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; This makes it easy to tweak plot properties 'after the fact'.
;; **

;; @@
(assoc-in p [:content "marks" 0 "properties" "enter" "fill" "value"] "red")
;; @@
;; =>
;;; {"type":"vega","content":{"marks":{"0":{"properties":{"enter":{"fill":{"value":"red"}}}}},"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e","field":"data.y"}}],"marks":[{"type":"symbol","from":{"data":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"fill":{"value":"steelblue"},"fillOpacity":{"value":1}},"update":{"shape":"circle","size":{"value":70},"stroke":{"value":"transparent"}},"hover":{"size":{"value":210},"stroke":{"value":"white"}}}}],"data":[{"name":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e","values":[{"x":0,"y":1},{"x":1,"y":3},{"x":2,"y":5},{"x":3,"y":7},{"x":4,"y":9},{"x":5,"y":5},{"x":6,"y":4},{"x":7,"y":6},{"x":8,"y":9},{"x":9,"y":8},{"x":10,"y":3},{"x":11,"y":5},{"x":12,"y":6}]}],"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50}},"value":"#gorilla_repl.vega.VegaView{:content {\"marks\" {0 {\"properties\" {\"enter\" {\"fill\" {\"value\" \"red\"}}}}}, :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\", :field \"data.y\"}}], :marks [{:type \"symbol\", :from {:data \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 1}}, :update {:shape \"circle\", :size {:value 70}, :stroke {:value \"transparent\"}}, :hover {:size {:value 210}, :stroke {:value \"white\"}}}}], :data [{:name \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\", :values ({:x 0, :y 1} {:x 1, :y 3} {:x 2, :y 5} {:x 3, :y 7} {:x 4, :y 9} {:x 5, :y 5} {:x 6, :y 4} {:x 7, :y 6} {:x 8, :y 9} {:x 9, :y 8} {:x 10, :y 3} {:x 11, :y 5} {:x 12, :y 6})}], :width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}}}"}
;; <=

;; **
;;; Another key feature of gorilla-plot, is that because plots are values, it's straighforward to compose them. Here's another plot
;; **

;; @@
(def p2 (list-plot d :joined true :plot-range [:all [1 5]]))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;user/p2</span>","value":"#'user/p2"}
;; <=

;; **
;;; And here we compose them to form a single plot. The axes and plot range are taken from the first plot given to compose.
;; **

;; @@
(compose p p2)
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50},"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"data":[{"name":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e","values":[{"x":0,"y":1},{"x":1,"y":3},{"x":2,"y":5},{"x":3,"y":7},{"x":4,"y":9},{"x":5,"y":5},{"x":6,"y":4},{"x":7,"y":6},{"x":8,"y":9},{"x":9,"y":8},{"x":10,"y":3},{"x":11,"y":5},{"x":12,"y":6}]},{"name":"70224453-07a1-4e75-940c-928423bfe52f","values":[{"x":0,"y":1},{"x":1,"y":3},{"x":2,"y":5},{"x":3,"y":7},{"x":4,"y":9},{"x":5,"y":5},{"x":6,"y":4},{"x":7,"y":6},{"x":8,"y":9},{"x":9,"y":8},{"x":10,"y":3},{"x":11,"y":5},{"x":12,"y":6}]}],"marks":[{"type":"symbol","from":{"data":"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"fill":{"value":"steelblue"},"fillOpacity":{"value":1}},"update":{"shape":"circle","size":{"value":70},"stroke":{"value":"transparent"}},"hover":{"size":{"value":210},"stroke":{"value":"white"}}}},{"type":"line","from":{"data":"70224453-07a1-4e75-940c-928423bfe52f"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"#FF29D2"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}, :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :data ({:name \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\", :values ({:x 0, :y 1} {:x 1, :y 3} {:x 2, :y 5} {:x 3, :y 7} {:x 4, :y 9} {:x 5, :y 5} {:x 6, :y 4} {:x 7, :y 6} {:x 8, :y 9} {:x 9, :y 8} {:x 10, :y 3} {:x 11, :y 5} {:x 12, :y 6})} {:name \"70224453-07a1-4e75-940c-928423bfe52f\", :values ({:x 0, :y 1} {:x 1, :y 3} {:x 2, :y 5} {:x 3, :y 7} {:x 4, :y 9} {:x 5, :y 5} {:x 6, :y 4} {:x 7, :y 6} {:x 8, :y 9} {:x 9, :y 8} {:x 10, :y 3} {:x 11, :y 5} {:x 12, :y 6})}), :marks ({:type \"symbol\", :from {:data \"90e8e8e5-08a3-4587-bb5f-ba3d6204af1e\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 1}}, :update {:shape \"circle\", :size {:value 70}, :stroke {:value \"transparent\"}}, :hover {:size {:value 210}, :stroke {:value \"white\"}}}} {:type \"line\", :from {:data \"70224453-07a1-4e75-940c-928423bfe52f\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"#FF29D2\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}})}}"}
;; <=

;; **
;;; You can also plot functions.
;; **

;; @@
(plot (fn [x] (Math/sin x)) [0 10] :color "orange" :plot-points 50)
;; @@
;; =>
;;; {"type":"vega","content":{"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"03174726-813a-438d-a8f6-f5befd42fc57","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"03174726-813a-438d-a8f6-f5befd42fc57","field":"data.y"}}],"marks":[{"type":"line","from":{"data":"03174726-813a-438d-a8f6-f5befd42fc57"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"orange"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}],"data":[{"name":"03174726-813a-438d-a8f6-f5befd42fc57","values":[{"x":0,"y":0.0},{"x":0.20000000298023224,"y":0.19866933371588721},{"x":0.4000000059604645,"y":0.3894183477986018},{"x":0.6000000089406967,"y":0.5646424807741107},{"x":0.800000011920929,"y":0.7173560992049138},{"x":1.0000000149011612,"y":0.8414709928590282},{"x":1.2000000178813934,"y":0.9320390924466878},{"x":1.4000000208616257,"y":0.9854497335342509},{"x":1.600000023841858,"y":0.999573602345334},{"x":1.8000000268220901,"y":0.9738476247841598},{"x":2.0000000298023224,"y":0.9092974144235391},{"x":2.2000000327825546,"y":0.8084963845270198},{"x":2.400000035762787,"y":0.6754631541798962},{"x":2.600000038743019,"y":0.5155013386230065},{"x":2.8000000417232513,"y":0.3349881108433251},{"x":3.0000000447034836,"y":0.14111996380375377},{"x":3.200000047683716,"y":-0.058374191029984236},{"x":3.400000050663948,"y":-0.2555411510086444},{"x":3.6000000536441803,"y":-0.4425204914007219},{"x":3.8000000566244125,"y":-0.6118579357308002},{"x":4.000000059604645,"y":-0.7568025342681227},{"x":4.200000062584877,"y":-0.8715758030964995},{"x":4.400000065565109,"y":-0.9516020940398271},{"x":4.6000000685453415,"y":-0.9936910113209954},{"x":4.800000071525574,"y":-0.9961646025774231},{"x":5.000000074505806,"y":-0.9589242535286561},{"x":5.200000077486038,"y":-0.8834546194166499},{"x":5.4000000804662704,"y":-0.7727644364846162},{"x":5.600000083446503,"y":-0.631266573154059},{"x":5.800000086426735,"y":-0.4646021028811949},{"x":6.000000089406967,"y":-0.2794154123530115},{"x":6.200000092387199,"y":-0.0830893107497628},{"x":6.400000095367432,"y":0.11654929956798761},{"x":6.600000098347664,"y":0.31154145696653224},{"x":6.800000101327896,"y":0.49411343923282436},{"x":7.000000104308128,"y":0.6569866773569186},{"x":7.200000107288361,"y":0.7936679291181636},{"x":7.400000110268593,"y":0.898708144169618},{"x":7.600000113248825,"y":0.9679197004863622},{"x":7.800000116229057,"y":0.9985433516457859},{"x":8.00000011920929,"y":0.9893582292784191},{"x":8.200000122189522,"y":0.9407305152385956},{"x":8.400000125169754,"y":0.8545988430890409},{"x":8.600000128149986,"y":0.7343970108961424},{"x":8.800000131130219,"y":0.584917086532953},{"x":9.00000013411045,"y":0.41211836304966276},{"x":9.200000137090683,"y":0.222889780458267},{"x":9.400000140070915,"y":0.02477528542543854},{"x":9.600000143051147,"y":-0.17432692208370587},{"x":9.80000014603138,"y":-0.36647926512335605}]}],"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50}},"value":"#gorilla_repl.vega.VegaView{:content {:axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"03174726-813a-438d-a8f6-f5befd42fc57\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"03174726-813a-438d-a8f6-f5befd42fc57\", :field \"data.y\"}}], :marks [{:type \"line\", :from {:data \"03174726-813a-438d-a8f6-f5befd42fc57\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"orange\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}}], :data [{:name \"03174726-813a-438d-a8f6-f5befd42fc57\", :values ({:x 0, :y 0.0} {:x 0.20000000298023224, :y 0.19866933371588721} {:x 0.4000000059604645, :y 0.3894183477986018} {:x 0.6000000089406967, :y 0.5646424807741107} {:x 0.800000011920929, :y 0.7173560992049138} {:x 1.0000000149011612, :y 0.8414709928590282} {:x 1.2000000178813934, :y 0.9320390924466878} {:x 1.4000000208616257, :y 0.9854497335342509} {:x 1.600000023841858, :y 0.999573602345334} {:x 1.8000000268220901, :y 0.9738476247841598} {:x 2.0000000298023224, :y 0.9092974144235391} {:x 2.2000000327825546, :y 0.8084963845270198} {:x 2.400000035762787, :y 0.6754631541798962} {:x 2.600000038743019, :y 0.5155013386230065} {:x 2.8000000417232513, :y 0.3349881108433251} {:x 3.0000000447034836, :y 0.14111996380375377} {:x 3.200000047683716, :y -0.058374191029984236} {:x 3.400000050663948, :y -0.2555411510086444} {:x 3.6000000536441803, :y -0.4425204914007219} {:x 3.8000000566244125, :y -0.6118579357308002} {:x 4.000000059604645, :y -0.7568025342681227} {:x 4.200000062584877, :y -0.8715758030964995} {:x 4.400000065565109, :y -0.9516020940398271} {:x 4.6000000685453415, :y -0.9936910113209954} {:x 4.800000071525574, :y -0.9961646025774231} {:x 5.000000074505806, :y -0.9589242535286561} {:x 5.200000077486038, :y -0.8834546194166499} {:x 5.4000000804662704, :y -0.7727644364846162} {:x 5.600000083446503, :y -0.631266573154059} {:x 5.800000086426735, :y -0.4646021028811949} {:x 6.000000089406967, :y -0.2794154123530115} {:x 6.200000092387199, :y -0.0830893107497628} {:x 6.400000095367432, :y 0.11654929956798761} {:x 6.600000098347664, :y 0.31154145696653224} {:x 6.800000101327896, :y 0.49411343923282436} {:x 7.000000104308128, :y 0.6569866773569186} {:x 7.200000107288361, :y 0.7936679291181636} {:x 7.400000110268593, :y 0.898708144169618} {:x 7.600000113248825, :y 0.9679197004863622} {:x 7.800000116229057, :y 0.9985433516457859} {:x 8.00000011920929, :y 0.9893582292784191} {:x 8.200000122189522, :y 0.9407305152385956} {:x 8.400000125169754, :y 0.8545988430890409} {:x 8.600000128149986, :y 0.7343970108961424} {:x 8.800000131130219, :y 0.584917086532953} {:x 9.00000013411045, :y 0.41211836304966276} {:x 9.200000137090683, :y 0.222889780458267} {:x 9.400000140070915, :y 0.02477528542543854} {:x 9.600000143051147, :y -0.17432692208370587} {:x 9.80000014603138, :y -0.36647926512335605})}], :width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}}}"}
;; <=

;; **
;;; And, of course, these plots compose too.
;; **

;; @@
(compose
  (list-plot (map #(vector % (rand %)) (range 0 10 0.01)) :opacity 0.3 :symbol-size 50)
  (plot (fn [x] (* x (Math/pow (Math/sin x) 2))) [0 10]))
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50},"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"data":[{"name":"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6","values":[{"x":0,"y":0.0},{"x":0.01,"y":5.398421659901543E-4},{"x":0.02,"y":0.0022078270776596143},{"x":0.03,"y":0.014699722478001756},{"x":0.04,"y":0.03880856675762969},{"x":0.05,"y":0.048497360775456806},{"x":0.060000000000000005,"y":0.04133135892065133},{"x":0.07,"y":0.03860803263755669},{"x":0.08,"y":0.022559157093072244},{"x":0.09,"y":0.05937787616411635},{"x":0.09999999999999999,"y":0.02034394553636346},{"x":0.10999999999999999,"y":0.02073904234513112},{"x":0.11999999999999998,"y":0.11775254558807377},{"x":0.12999999999999998,"y":0.03580653538221089},{"x":0.13999999999999999,"y":0.06989600617536991},{"x":0.15,"y":0.01558731613371262},{"x":0.16,"y":0.0040657512580307125},{"x":0.17,"y":0.03971488248370004},{"x":0.18000000000000002,"y":0.08480104744112552},{"x":0.19000000000000003,"y":0.10671004969858121},{"x":0.20000000000000004,"y":0.039686630288586974},{"x":0.21000000000000005,"y":0.15941675191181415},{"x":0.22000000000000006,"y":0.20880863068054883},{"x":0.23000000000000007,"y":0.05251564303355356},{"x":0.24000000000000007,"y":0.09175238395112068},{"x":0.25000000000000006,"y":0.12426339002326894},{"x":0.26000000000000006,"y":0.007519097735057327},{"x":0.2700000000000001,"y":0.23331696813248287},{"x":0.2800000000000001,"y":0.20107071335214616},{"x":0.2900000000000001,"y":0.2239948346672196},{"x":0.3000000000000001,"y":0.0700250006951655},{"x":0.3100000000000001,"y":0.06200552657621937},{"x":0.3200000000000001,"y":0.2930464222793822},{"x":0.3300000000000001,"y":0.19828331424359558},{"x":0.34000000000000014,"y":0.155284233498906},{"x":0.35000000000000014,"y":0.0014449503211177279},{"x":0.36000000000000015,"y":0.10795721464714836},{"x":0.37000000000000016,"y":0.22569516309589507},{"x":0.38000000000000017,"y":0.3743761401050732},{"x":0.3900000000000002,"y":0.014025961318826257},{"x":0.4000000000000002,"y":0.31591928627169424},{"x":0.4100000000000002,"y":0.2528794364180144},{"x":0.4200000000000002,"y":0.007165129592618084},{"x":0.4300000000000002,"y":0.13527029040556288},{"x":0.4400000000000002,"y":0.39755765476163313},{"x":0.45000000000000023,"y":0.3267253841288474},{"x":0.46000000000000024,"y":0.00399578150781339},{"x":0.47000000000000025,"y":0.41641793554987233},{"x":0.48000000000000026,"y":0.02304603122719935},{"x":0.49000000000000027,"y":0.22701574673147695},{"x":0.5000000000000002,"y":0.4323697227427657},{"x":0.5100000000000002,"y":0.03308042988666946},{"x":0.5200000000000002,"y":0.4223084562810144},{"x":0.5300000000000002,"y":0.33723982455883983},{"x":0.5400000000000003,"y":0.2476071348696873},{"x":0.5500000000000003,"y":0.5207930969106707},{"x":0.5600000000000003,"y":0.40027810665791375},{"x":0.5700000000000003,"y":0.18921838215219203},{"x":0.5800000000000003,"y":0.12777526982385282},{"x":0.5900000000000003,"y":0.4961504513193089},{"x":0.6000000000000003,"y":0.5256889101533392},{"x":0.6100000000000003,"y":0.4255323503258928},{"x":0.6200000000000003,"y":0.34917193269507474},{"x":0.6300000000000003,"y":0.09666229878658686},{"x":0.6400000000000003,"y":0.24243226829997416},{"x":0.6500000000000004,"y":0.6049387862131121},{"x":0.6600000000000004,"y":0.48188462723335723},{"x":0.6700000000000004,"y":0.11602277526246256},{"x":0.6800000000000004,"y":0.4515442595325231},{"x":0.6900000000000004,"y":0.14250026795884216},{"x":0.7000000000000004,"y":0.1679047106735959},{"x":0.7100000000000004,"y":0.10664055147998841},{"x":0.7200000000000004,"y":0.12189856710413462},{"x":0.7300000000000004,"y":0.08054881611368087},{"x":0.7400000000000004,"y":0.3594617231597144},{"x":0.7500000000000004,"y":0.03053663303760461},{"x":0.7600000000000005,"y":0.42116675815983806},{"x":0.7700000000000005,"y":0.3913510551251715},{"x":0.7800000000000005,"y":0.15515513148666304},{"x":0.7900000000000005,"y":0.23557521763440045},{"x":0.8000000000000005,"y":0.22807024631913433},{"x":0.8100000000000005,"y":0.5599394665984069},{"x":0.8200000000000005,"y":0.3485990764880174},{"x":0.8300000000000005,"y":0.10895414325371701},{"x":0.8400000000000005,"y":0.044887785984757275},{"x":0.8500000000000005,"y":0.04852345474114983},{"x":0.8600000000000005,"y":0.3311615142143673},{"x":0.8700000000000006,"y":0.626444139685543},{"x":0.8800000000000006,"y":0.4280986667523344},{"x":0.8900000000000006,"y":0.49469482813826193},{"x":0.9000000000000006,"y":0.0794703957589415},{"x":0.9100000000000006,"y":0.8965295864070829},{"x":0.9200000000000006,"y":0.8074337268700722},{"x":0.9300000000000006,"y":0.5767947715995505},{"x":0.9400000000000006,"y":0.8021782255539507},{"x":0.9500000000000006,"y":0.6058701534647117},{"x":0.9600000000000006,"y":0.2019760711301762},{"x":0.9700000000000006,"y":0.3138130864091943},{"x":0.9800000000000006,"y":0.9315211374811561},{"x":0.9900000000000007,"y":0.39886845010829597},{"x":1.0000000000000007,"y":0.5787987415228129},{"x":1.0100000000000007,"y":0.06654596870086399},{"x":1.0200000000000007,"y":0.4359800672021171},{"x":1.0300000000000007,"y":0.5220990232252575},{"x":1.0400000000000007,"y":0.22744898287205026},{"x":1.0500000000000007,"y":0.1839460610864435},{"x":1.0600000000000007,"y":0.8533498744499903},{"x":1.0700000000000007,"y":0.7964232454502889},{"x":1.0800000000000007,"y":0.8148872702105039},{"x":1.0900000000000007,"y":0.46615618394327674},{"x":1.1000000000000008,"y":0.8112464889874446},{"x":1.1100000000000008,"y":0.7782871234715933},{"x":1.1200000000000008,"y":1.0527654913490003},{"x":1.1300000000000008,"y":0.7261292804917082},{"x":1.1400000000000008,"y":1.0470336429766902},{"x":1.1500000000000008,"y":1.1494723566786214},{"x":1.1600000000000008,"y":0.14467921459845276},{"x":1.1700000000000008,"y":0.21749845881747512},{"x":1.1800000000000008,"y":0.5738285611629873},{"x":1.1900000000000008,"y":0.8939457463992905},{"x":1.2000000000000008,"y":1.097759144509554},{"x":1.2100000000000009,"y":0.5199737485772478},{"x":1.2200000000000009,"y":1.1361498025098085},{"x":1.2300000000000009,"y":0.9339290018658525},{"x":1.2400000000000009,"y":0.013170624699442524},{"x":1.2500000000000009,"y":1.1975083340342207},{"x":1.260000000000001,"y":0.05040176991511057},{"x":1.270000000000001,"y":0.49738874403059735},{"x":1.280000000000001,"y":0.006272286170290049},{"x":1.290000000000001,"y":0.7512917046186459},{"x":1.300000000000001,"y":0.9442477734719107},{"x":1.310000000000001,"y":0.608326610113623},{"x":1.320000000000001,"y":0.0018431595289033456},{"x":1.330000000000001,"y":0.05179694205315745},{"x":1.340000000000001,"y":0.9275072588839394},{"x":1.350000000000001,"y":1.242030075250892},{"x":1.360000000000001,"y":0.3754220768922943},{"x":1.370000000000001,"y":0.7841274640020042},{"x":1.380000000000001,"y":1.0180233780495533},{"x":1.390000000000001,"y":0.7823277674900635},{"x":1.400000000000001,"y":1.180914738941596},{"x":1.410000000000001,"y":0.6812803302679422},{"x":1.420000000000001,"y":0.2814848396591268},{"x":1.430000000000001,"y":0.8893778297397118},{"x":1.440000000000001,"y":1.029711715142625},{"x":1.450000000000001,"y":0.16316567743217653},{"x":1.460000000000001,"y":0.6169484752586413},{"x":1.470000000000001,"y":0.6671026626109712},{"x":1.480000000000001,"y":1.2401247569433116},{"x":1.490000000000001,"y":0.9977965197954077},{"x":1.500000000000001,"y":0.6423003817528923},{"x":1.5100000000000011,"y":0.4015719674679796},{"x":1.5200000000000011,"y":0.2904903343669004},{"x":1.5300000000000011,"y":1.1774878348363937},{"x":1.5400000000000011,"y":1.3386093847426133},{"x":1.5500000000000012,"y":0.5379247562769631},{"x":1.5600000000000012,"y":0.4373065514877508},{"x":1.5700000000000012,"y":0.8496824793175859},{"x":1.5800000000000012,"y":0.37177065668615594},{"x":1.5900000000000012,"y":1.466311431866839},{"x":1.6000000000000012,"y":0.5137638531359638},{"x":1.6100000000000012,"y":1.1116496265143185},{"x":1.6200000000000012,"y":1.236506524714334},{"x":1.6300000000000012,"y":0.6793292736301372},{"x":1.6400000000000012,"y":1.1965002147262713},{"x":1.6500000000000012,"y":1.5832819273482148},{"x":1.6600000000000013,"y":1.4741886262968147},{"x":1.6700000000000013,"y":1.0170185832938818},{"x":1.6800000000000013,"y":0.251860148566999},{"x":1.6900000000000013,"y":0.015050067530115584},{"x":1.7000000000000013,"y":0.31456832748306046},{"x":1.7100000000000013,"y":1.5031820227052617},{"x":1.7200000000000013,"y":0.2398583681636518},{"x":1.7300000000000013,"y":1.501356283933557},{"x":1.7400000000000013,"y":1.154851801008506},{"x":1.7500000000000013,"y":0.9110863838085098},{"x":1.7600000000000013,"y":1.4169516790267258},{"x":1.7700000000000014,"y":1.3255645419220343},{"x":1.7800000000000014,"y":1.7101326104114933},{"x":1.7900000000000014,"y":1.560004676217984},{"x":1.8000000000000014,"y":0.25009834707218215},{"x":1.8100000000000014,"y":0.3643703467707408},{"x":1.8200000000000014,"y":0.8194971543030658},{"x":1.8300000000000014,"y":1.5390560828888344},{"x":1.8400000000000014,"y":0.9869376717137699},{"x":1.8500000000000014,"y":0.5384133224156971},{"x":1.8600000000000014,"y":1.0046479319101662},{"x":1.8700000000000014,"y":0.49390361828273727},{"x":1.8800000000000014,"y":1.8727532234864914},{"x":1.8900000000000015,"y":1.3956238372008352},{"x":1.9000000000000015,"y":0.8651448092785847},{"x":1.9100000000000015,"y":0.7575713077666987},{"x":1.9200000000000015,"y":1.5366895686385642},{"x":1.9300000000000015,"y":1.3862370640323443},{"x":1.9400000000000015,"y":1.208848087887971},{"x":1.9500000000000015,"y":1.5214487073305465},{"x":1.9600000000000015,"y":0.09997483236211901},{"x":1.9700000000000015,"y":0.6562013326642258},{"x":1.9800000000000015,"y":0.7303541813491781},{"x":1.9900000000000015,"y":1.2989556343317417},{"x":2.0000000000000013,"y":0.600162613784154},{"x":2.010000000000001,"y":1.2213511047378105},{"x":2.020000000000001,"y":0.016652889139142},{"x":2.0300000000000007,"y":0.11484875815111481},{"x":2.0400000000000005,"y":1.3301417396164783},{"x":2.0500000000000003,"y":0.45071174552966803},{"x":2.06,"y":1.190394742825987},{"x":2.07,"y":1.6702602050847142},{"x":2.0799999999999996,"y":0.2913759613544047},{"x":2.0899999999999994,"y":0.17615060289890672},{"x":2.099999999999999,"y":0.8095475226645792},{"x":2.109999999999999,"y":1.099770418131657},{"x":2.1199999999999988,"y":1.8700250970665462},{"x":2.1299999999999986,"y":0.7097887858914748},{"x":2.1399999999999983,"y":0.600335061770539},{"x":2.149999999999998,"y":2.1146281789222985},{"x":2.159999999999998,"y":1.0947647297737115},{"x":2.1699999999999977,"y":0.44263723142724},{"x":2.1799999999999975,"y":0.10069409032864259},{"x":2.1899999999999973,"y":0.4781472821910741},{"x":2.199999999999997,"y":1.3387650104301185},{"x":2.209999999999997,"y":1.840325011721693},{"x":2.2199999999999966,"y":1.3265525388735941},{"x":2.2299999999999964,"y":1.7478224065291768},{"x":2.239999999999996,"y":1.197906778386139},{"x":2.249999999999996,"y":2.2070402584204603},{"x":2.259999999999996,"y":1.6555280107346095},{"x":2.2699999999999956,"y":2.176325572496733},{"x":2.2799999999999954,"y":1.4654109141402145},{"x":2.289999999999995,"y":0.6294100180977333},{"x":2.299999999999995,"y":0.06466695644617297},{"x":2.3099999999999947,"y":1.6027633332694997},{"x":2.3199999999999945,"y":0.6221271422384828},{"x":2.3299999999999943,"y":1.4605028337207737},{"x":2.339999999999994,"y":0.2330919209909163},{"x":2.349999999999994,"y":0.9988933766231244},{"x":2.3599999999999937,"y":0.10427843705688346},{"x":2.3699999999999934,"y":0.29294529557853677},{"x":2.3799999999999932,"y":2.019400744508853},{"x":2.389999999999993,"y":0.254923852818212},{"x":2.399999999999993,"y":2.1089829997212712},{"x":2.4099999999999926,"y":1.971805610846062},{"x":2.4199999999999924,"y":1.2398605446996118},{"x":2.429999999999992,"y":0.2685398232877039},{"x":2.439999999999992,"y":2.084875725697424},{"x":2.4499999999999917,"y":1.3249070706011035},{"x":2.4599999999999915,"y":0.7393690165988231},{"x":2.4699999999999913,"y":2.3714272935624487},{"x":2.479999999999991,"y":1.363132338004649},{"x":2.489999999999991,"y":0.1430992809905421},{"x":2.4999999999999907,"y":0.3753089323927991},{"x":2.5099999999999905,"y":0.5522915436728815},{"x":2.5199999999999902,"y":2.005652258400631},{"x":2.52999999999999,"y":1.6382287090645309},{"x":2.53999999999999,"y":1.6973020155615255},{"x":2.5499999999999896,"y":0.6266955402613141},{"x":2.5599999999999894,"y":1.7545887536194773},{"x":2.569999999999989,"y":1.144617301419211},{"x":2.579999999999989,"y":1.6559775680920183},{"x":2.5899999999999888,"y":2.364471757815446},{"x":2.5999999999999885,"y":1.8733233802749822},{"x":2.6099999999999883,"y":1.8232748758868798},{"x":2.619999999999988,"y":1.5437032033681404},{"x":2.629999999999988,"y":2.3474945226318678},{"x":2.6399999999999877,"y":0.9458658816167869},{"x":2.6499999999999875,"y":1.1393417476805132},{"x":2.6599999999999873,"y":1.8722868116208555},{"x":2.669999999999987,"y":1.376730636284824},{"x":2.679999999999987,"y":2.2620226330841944},{"x":2.6899999999999866,"y":2.3148783449075694},{"x":2.6999999999999864,"y":0.5394458008520994},{"x":2.709999999999986,"y":2.1102224667066625},{"x":2.719999999999986,"y":0.4120140576233859},{"x":2.7299999999999858,"y":0.9535917270486317},{"x":2.7399999999999856,"y":1.379853313752464},{"x":2.7499999999999853,"y":1.6054913234041364},{"x":2.759999999999985,"y":0.13903596442723373},{"x":2.769999999999985,"y":1.483911891084266},{"x":2.7799999999999847,"y":2.670310515420634},{"x":2.7899999999999845,"y":2.4432887685654334},{"x":2.7999999999999843,"y":2.2640519868731666},{"x":2.809999999999984,"y":2.344508881514406},{"x":2.819999999999984,"y":2.034562715233205},{"x":2.8299999999999836,"y":2.6671605282375443},{"x":2.8399999999999834,"y":2.27636945681668},{"x":2.849999999999983,"y":1.2043664602113162},{"x":2.859999999999983,"y":0.7439680138768742},{"x":2.869999999999983,"y":2.237429831019891},{"x":2.8799999999999826,"y":1.0087972894171333},{"x":2.8899999999999824,"y":1.1204089926063538},{"x":2.899999999999982,"y":1.548149742002961},{"x":2.909999999999982,"y":0.24144852378519424},{"x":2.9199999999999817,"y":1.2799048658293524},{"x":2.9299999999999815,"y":2.211907526568156},{"x":2.9399999999999813,"y":0.42740788621425235},{"x":2.949999999999981,"y":0.38680780865122066},{"x":2.959999999999981,"y":1.68059697726399},{"x":2.9699999999999807,"y":2.094973226476324},{"x":2.9799999999999804,"y":2.9485399268142847},{"x":2.9899999999999802,"y":1.6331464263194044},{"x":2.99999999999998,"y":2.720394781158697},{"x":3.00999999999998,"y":1.2348811250424423},{"x":3.0199999999999796,"y":2.3403187750316805},{"x":3.0299999999999794,"y":2.063740402882318},{"x":3.039999999999979,"y":0.16296315966437303},{"x":3.049999999999979,"y":1.701249706571135},{"x":3.0599999999999787,"y":1.536705284239315},{"x":3.0699999999999785,"y":0.5227071454717113},{"x":3.0799999999999783,"y":0.6030508948569143},{"x":3.089999999999978,"y":2.5637417022333957},{"x":3.099999999999978,"y":1.3774205550696368},{"x":3.1099999999999777,"y":0.009906951832155076},{"x":3.1199999999999775,"y":0.5606148577232218},{"x":3.1299999999999772,"y":0.9264374134816045},{"x":3.139999999999977,"y":1.359287442611299},{"x":3.149999999999977,"y":1.0459760475883617},{"x":3.1599999999999766,"y":0.06497811245655262},{"x":3.1699999999999764,"y":2.979872579729387},{"x":3.179999999999976,"y":0.8902533077741367},{"x":3.189999999999976,"y":0.34687845133543205},{"x":3.1999999999999758,"y":2.1849300706820034},{"x":3.2099999999999755,"y":0.9634244064201346},{"x":3.2199999999999753,"y":3.163739953338034},{"x":3.229999999999975,"y":1.813419179105547},{"x":3.239999999999975,"y":1.7422077895722903},{"x":3.2499999999999747,"y":1.6897718705452938},{"x":3.2599999999999745,"y":3.0964249027630584},{"x":3.2699999999999743,"y":1.5288173254210238},{"x":3.279999999999974,"y":2.518310520678372},{"x":3.289999999999974,"y":3.080051258315208},{"x":3.2999999999999736,"y":2.870244500868692},{"x":3.3099999999999734,"y":0.5191944470928433},{"x":3.319999999999973,"y":0.420582224462231},{"x":3.329999999999973,"y":2.335574221998874},{"x":3.3399999999999728,"y":3.3018206030356785},{"x":3.3499999999999726,"y":0.6301949413032795},{"x":3.3599999999999723,"y":2.9854624542676005},{"x":3.369999999999972,"y":2.8220722020369458},{"x":3.379999999999972,"y":1.7421521471667307},{"x":3.3899999999999717,"y":2.3104138252959525},{"x":3.3999999999999715,"y":1.7772444631896507},{"x":3.4099999999999713,"y":1.8678816483109897},{"x":3.419999999999971,"y":1.998777210549902},{"x":3.429999999999971,"y":1.2413721298758178},{"x":3.4399999999999706,"y":2.122095341674623},{"x":3.4499999999999704,"y":1.9907849731722855},{"x":3.45999999999997,"y":2.4442037861295978},{"x":3.46999999999997,"y":1.6269417453033377},{"x":3.47999999999997,"y":2.9119853312768993},{"x":3.4899999999999696,"y":3.141873149397687},{"x":3.4999999999999694,"y":3.2407254514599617},{"x":3.509999999999969,"y":2.894889838620878},{"x":3.519999999999969,"y":0.9140433148332595},{"x":3.5299999999999687,"y":0.17130189037557036},{"x":3.5399999999999685,"y":3.4052824080076425},{"x":3.5499999999999683,"y":1.8320774504455828},{"x":3.559999999999968,"y":0.8802010568545677},{"x":3.569999999999968,"y":3.3326669550691603},{"x":3.5799999999999677,"y":1.631302435687567},{"x":3.5899999999999674,"y":3.0153176404893847},{"x":3.5999999999999672,"y":0.48231001044844274},{"x":3.609999999999967,"y":0.4324327428216076},{"x":3.619999999999967,"y":0.2699493899834946},{"x":3.6299999999999666,"y":1.4344680816758295},{"x":3.6399999999999664,"y":1.451714738290243},{"x":3.649999999999966,"y":0.4109052082266942},{"x":3.659999999999966,"y":2.303438228002478},{"x":3.6699999999999657,"y":1.6755049088321243},{"x":3.6799999999999655,"y":2.432041667557286},{"x":3.6899999999999653,"y":1.3941522200980498},{"x":3.699999999999965,"y":2.0305191196688055},{"x":3.709999999999965,"y":0.22159790166940735},{"x":3.7199999999999647,"y":2.952213648382213},{"x":3.7299999999999645,"y":1.8345340572366085},{"x":3.7399999999999642,"y":0.10141873088611186},{"x":3.749999999999964,"y":2.0520829310452933},{"x":3.759999999999964,"y":2.585930124011471},{"x":3.7699999999999636,"y":0.113622643159977},{"x":3.7799999999999634,"y":3.4389236550057847},{"x":3.789999999999963,"y":2.5652021222917663},{"x":3.799999999999963,"y":2.6965564188319493},{"x":3.8099999999999627,"y":2.0877499448638663},{"x":3.8199999999999625,"y":3.4964280359186652},{"x":3.8299999999999623,"y":2.9830639222745123},{"x":3.839999999999962,"y":2.685049418214062},{"x":3.849999999999962,"y":1.075744779963071},{"x":3.8599999999999617,"y":0.40512075033509715},{"x":3.8699999999999615,"y":3.3389501606771472},{"x":3.8799999999999613,"y":1.6757022273913362},{"x":3.889999999999961,"y":3.843084587106392},{"x":3.899999999999961,"y":2.3635730524525798},{"x":3.9099999999999606,"y":0.08038347764319287},{"x":3.9199999999999604,"y":3.2426285746127648},{"x":3.92999999999996,"y":2.8992071041535654},{"x":3.93999999999996,"y":1.8562986132243475},{"x":3.9499999999999598,"y":3.7587467176899008},{"x":3.9599999999999596,"y":3.2915154530103794},{"x":3.9699999999999593,"y":0.8875997560956681},{"x":3.979999999999959,"y":0.8458936915333946},{"x":3.989999999999959,"y":3.760233888388318},{"x":3.9999999999999587,"y":3.278763638011273},{"x":4.009999999999959,"y":2.794006148406674},{"x":4.019999999999959,"y":3.978629765090821},{"x":4.0299999999999585,"y":0.7059960856035993},{"x":4.039999999999958,"y":2.326198048711626},{"x":4.049999999999958,"y":1.8391632569736032},{"x":4.059999999999958,"y":2.628361230770177},{"x":4.069999999999958,"y":3.2940339473762354},{"x":4.079999999999957,"y":1.5465714524244691},{"x":4.089999999999957,"y":1.3114093890395742},{"x":4.099999999999957,"y":3.29082684621221},{"x":4.109999999999957,"y":3.6341772744050846},{"x":4.119999999999957,"y":2.105603379819089},{"x":4.129999999999956,"y":2.2637791238363034},{"x":4.139999999999956,"y":3.9823676945841555},{"x":4.149999999999956,"y":3.0028829340249708},{"x":4.159999999999956,"y":2.2027889279146824},{"x":4.1699999999999555,"y":0.3223322550616315},{"x":4.179999999999955,"y":0.8862540216706211},{"x":4.189999999999955,"y":0.7890753422502149},{"x":4.199999999999955,"y":4.168183988045357},{"x":4.209999999999955,"y":0.699941843194316},{"x":4.2199999999999545,"y":1.5787095901217727},{"x":4.229999999999954,"y":3.468344762032132},{"x":4.239999999999954,"y":3.918320108223203},{"x":4.249999999999954,"y":4.14510085623999},{"x":4.259999999999954,"y":2.7078460637241744},{"x":4.269999999999953,"y":2.753018005868473},{"x":4.279999999999953,"y":0.8838260602657648},{"x":4.289999999999953,"y":3.500988088697334},{"x":4.299999999999953,"y":2.5045683956888003},{"x":4.3099999999999525,"y":0.7224095229168292},{"x":4.319999999999952,"y":1.0867340376504329},{"x":4.329999999999952,"y":1.8285722224228904},{"x":4.339999999999952,"y":1.0475889396321436},{"x":4.349999999999952,"y":3.803460971842525},{"x":4.3599999999999515,"y":3.3393240910350412},{"x":4.369999999999951,"y":3.4114160449870727},{"x":4.379999999999951,"y":0.9374914357686269},{"x":4.389999999999951,"y":1.67947793059512},{"x":4.399999999999951,"y":0.671216455924884},{"x":4.40999999999995,"y":3.4015841446087944},{"x":4.41999999999995,"y":1.0298137177943432},{"x":4.42999999999995,"y":1.036271408031536},{"x":4.43999999999995,"y":0.49974807399466026},{"x":4.4499999999999496,"y":0.4981454789836499},{"x":4.459999999999949,"y":2.2468752404885675},{"x":4.469999999999949,"y":0.9132180082150437},{"x":4.479999999999949,"y":1.3729787833139357},{"x":4.489999999999949,"y":3.914705044909433},{"x":4.4999999999999485,"y":0.04505766317827979},{"x":4.509999999999948,"y":4.164055254185302},{"x":4.519999999999948,"y":2.3274564248991942},{"x":4.529999999999948,"y":0.5926889912663521},{"x":4.539999999999948,"y":2.5109547976217885},{"x":4.549999999999947,"y":2.461136518246732},{"x":4.559999999999947,"y":2.726625640823365},{"x":4.569999999999947,"y":0.8752571298239804},{"x":4.579999999999947,"y":3.0887452589901},{"x":4.589999999999947,"y":0.7198743390169127},{"x":4.599999999999946,"y":3.1930078312820833},{"x":4.609999999999946,"y":0.575206768558524},{"x":4.619999999999946,"y":3.534801922336055},{"x":4.629999999999946,"y":3.4097163474477106},{"x":4.6399999999999455,"y":2.4304863466871356},{"x":4.649999999999945,"y":1.5115839786152543},{"x":4.659999999999945,"y":2.826441736835714},{"x":4.669999999999945,"y":1.296701276398053},{"x":4.679999999999945,"y":1.7047246501204583},{"x":4.689999999999944,"y":0.8664167854243952},{"x":4.699999999999944,"y":1.432334766298035},{"x":4.709999999999944,"y":0.30296318261119637},{"x":4.719999999999944,"y":1.1824106583414598},{"x":4.729999999999944,"y":4.527532945431609},{"x":4.739999999999943,"y":0.3949322304451271},{"x":4.749999999999943,"y":4.482957913336911},{"x":4.759999999999943,"y":1.056160020114835},{"x":4.769999999999943,"y":1.7783312471800183},{"x":4.7799999999999425,"y":3.2528425126221214},{"x":4.789999999999942,"y":3.3572095494891085},{"x":4.799999999999942,"y":4.71403431321551},{"x":4.809999999999942,"y":2.863126033190511},{"x":4.819999999999942,"y":0.7758796429633972},{"x":4.8299999999999415,"y":3.694484138920159},{"x":4.839999999999941,"y":0.8232235842344509},{"x":4.849999999999941,"y":3.2097917655717154},{"x":4.859999999999941,"y":3.8700273074043863},{"x":4.869999999999941,"y":3.3250231333361073},{"x":4.87999999999994,"y":2.679014712113577},{"x":4.88999999999994,"y":3.12239035487871},{"x":4.89999999999994,"y":1.4463140656457991},{"x":4.90999999999994,"y":0.09328448895304228},{"x":4.9199999999999395,"y":3.995217196773686},{"x":4.929999999999939,"y":1.6677744968708872},{"x":4.939999999999939,"y":0.34906618172911297},{"x":4.949999999999939,"y":2.9810650181681275},{"x":4.959999999999939,"y":1.8732419946108254},{"x":4.9699999999999385,"y":2.9843180431723666},{"x":4.979999999999938,"y":2.0977065160199446},{"x":4.989999999999938,"y":1.6333977208233188},{"x":4.999999999999938,"y":2.496533127949724},{"x":5.009999999999938,"y":4.060437173952934},{"x":5.019999999999937,"y":2.364697813549784},{"x":5.029999999999937,"y":0.22274553282737145},{"x":5.039999999999937,"y":1.2675264374187871},{"x":5.049999999999937,"y":2.7779306676234135},{"x":5.0599999999999365,"y":2.362879987356525},{"x":5.069999999999936,"y":2.1142378250791696},{"x":5.079999999999936,"y":2.3882698248220433},{"x":5.089999999999936,"y":0.06287971396200587},{"x":5.099999999999936,"y":2.860833929848697},{"x":5.1099999999999355,"y":0.09852417794585762},{"x":5.119999999999935,"y":2.6757668232101546},{"x":5.129999999999935,"y":0.4574168084039595},{"x":5.139999999999935,"y":2.6775110599366863},{"x":5.149999999999935,"y":1.8017515147314147},{"x":5.159999999999934,"y":4.451515742201196},{"x":5.169999999999934,"y":4.37158956184408},{"x":5.179999999999934,"y":4.699701239098009},{"x":5.189999999999934,"y":4.64669916748889},{"x":5.199999999999934,"y":3.564101420824132},{"x":5.209999999999933,"y":2.248315319512665},{"x":5.219999999999933,"y":0.6704742957908782},{"x":5.229999999999933,"y":4.427740465353794},{"x":5.239999999999933,"y":3.346555908713266},{"x":5.2499999999999325,"y":1.8385440573816902},{"x":5.259999999999932,"y":2.3460146286554737},{"x":5.269999999999932,"y":3.07753568530085},{"x":5.279999999999932,"y":2.4890455067772357},{"x":5.289999999999932,"y":2.120717901462258},{"x":5.299999999999931,"y":2.3247836212949653},{"x":5.309999999999931,"y":3.1076930854768805},{"x":5.319999999999931,"y":3.7347531278669512},{"x":5.329999999999931,"y":2.9109422129731803},{"x":5.339999999999931,"y":3.4891863937937595},{"x":5.34999999999993,"y":5.261576623651832},{"x":5.35999999999993,"y":4.695117349812701},{"x":5.36999999999993,"y":2.2204587002963083},{"x":5.37999999999993,"y":0.9383817950808354},{"x":5.3899999999999295,"y":3.247474472711777},{"x":5.399999999999929,"y":4.0423573716132415},{"x":5.409999999999929,"y":0.7375346833981192},{"x":5.419999999999929,"y":3.3640728835066596},{"x":5.429999999999929,"y":5.066985048899524},{"x":5.4399999999999284,"y":4.224956176024419},{"x":5.449999999999928,"y":4.1542672721186005},{"x":5.459999999999928,"y":3.5415322674959544},{"x":5.469999999999928,"y":0.9692198934468507},{"x":5.479999999999928,"y":3.457995880544312},{"x":5.489999999999927,"y":2.8175274923357767},{"x":5.499999999999927,"y":2.0882622332890577},{"x":5.509999999999927,"y":1.0087633831589842},{"x":5.519999999999927,"y":0.29743572853804856},{"x":5.5299999999999265,"y":4.249852538192594},{"x":5.539999999999926,"y":1.6550666118796435},{"x":5.549999999999926,"y":4.576779208780081},{"x":5.559999999999926,"y":1.5280974481276017},{"x":5.569999999999926,"y":4.233028690119004},{"x":5.5799999999999255,"y":0.4314061387533993},{"x":5.589999999999925,"y":0.7426962669185353},{"x":5.599999999999925,"y":2.0917742142356484},{"x":5.609999999999925,"y":3.4668918165886806},{"x":5.619999999999925,"y":2.6012388267611946},{"x":5.629999999999924,"y":1.5810578518573493},{"x":5.639999999999924,"y":2.4270026036004513},{"x":5.649999999999924,"y":3.321009521132676},{"x":5.659999999999924,"y":4.789598393386025},{"x":5.6699999999999235,"y":1.9468651423598986},{"x":5.679999999999923,"y":0.23814636099039405},{"x":5.689999999999923,"y":5.389896111152566},{"x":5.699999999999923,"y":2.106441373600592},{"x":5.709999999999923,"y":3.455602579917994},{"x":5.7199999999999225,"y":4.551014611341899},{"x":5.729999999999922,"y":2.295753831517114},{"x":5.739999999999922,"y":2.6953035090117154},{"x":5.749999999999922,"y":0.4335788338516777},{"x":5.759999999999922,"y":3.417798597315438},{"x":5.769999999999921,"y":2.868922537638807},{"x":5.779999999999921,"y":3.2625339311118013},{"x":5.789999999999921,"y":4.522777925550773},{"x":5.799999999999921,"y":1.7181419640024531},{"x":5.809999999999921,"y":5.397764771777134},{"x":5.81999999999992,"y":1.0908374159118608},{"x":5.82999999999992,"y":2.09456909595203},{"x":5.83999999999992,"y":0.11816963818550123},{"x":5.84999999999992,"y":2.339106877229818},{"x":5.8599999999999195,"y":1.441039834128449},{"x":5.869999999999919,"y":2.600879375967315},{"x":5.879999999999919,"y":3.267787100422037},{"x":5.889999999999919,"y":0.7301028328145656},{"x":5.899999999999919,"y":1.4309040897844734},{"x":5.909999999999918,"y":2.062296864959538},{"x":5.919999999999918,"y":4.771023305230153},{"x":5.929999999999918,"y":5.424986680564862},{"x":5.939999999999918,"y":1.3819616148798817},{"x":5.949999999999918,"y":3.622530590155992},{"x":5.959999999999917,"y":5.027133556638011},{"x":5.969999999999917,"y":1.5081250986375792},{"x":5.979999999999917,"y":0.0771913684196489},{"x":5.989999999999917,"y":5.780214948857677},{"x":5.9999999999999165,"y":1.0240902259371192},{"x":6.009999999999916,"y":4.100587373971292},{"x":6.019999999999916,"y":3.226469066034051},{"x":6.029999999999916,"y":3.294712599759812},{"x":6.039999999999916,"y":0.2034840557474503},{"x":6.0499999999999154,"y":2.343598031990255},{"x":6.059999999999915,"y":5.025527198989543},{"x":6.069999999999915,"y":0.48483552577090355},{"x":6.079999999999915,"y":1.1592283324271944},{"x":6.089999999999915,"y":4.9212430229002555},{"x":6.099999999999914,"y":4.630212779610673},{"x":6.109999999999914,"y":0.9407682708381743},{"x":6.119999999999914,"y":2.2143032062073904},{"x":6.129999999999914,"y":4.469156846376697},{"x":6.1399999999999135,"y":5.063600597397378},{"x":6.149999999999913,"y":4.612448740547254},{"x":6.159999999999913,"y":4.3125990149184465},{"x":6.169999999999913,"y":2.1481227820941378},{"x":6.179999999999913,"y":4.921516049372504},{"x":6.1899999999999125,"y":2.984732797914748},{"x":6.199999999999912,"y":0.0792177516141292},{"x":6.209999999999912,"y":2.8340797808039144},{"x":6.219999999999912,"y":2.5747507563435246},{"x":6.229999999999912,"y":1.5433374570587732},{"x":6.239999999999911,"y":0.8548659286970697},{"x":6.249999999999911,"y":0.05143559928886956},{"x":6.259999999999911,"y":3.4761986027283087},{"x":6.269999999999911,"y":1.73833399459127},{"x":6.2799999999999105,"y":5.190725933102908},{"x":6.28999999999991,"y":0.26735322497152425},{"x":6.29999999999991,"y":1.5808256521228565},{"x":6.30999999999991,"y":5.6663371204586035},{"x":6.31999999999991,"y":4.691917516084117},{"x":6.3299999999999095,"y":0.7932843489038839},{"x":6.339999999999909,"y":4.268283722881138},{"x":6.349999999999909,"y":4.281009731566307},{"x":6.359999999999909,"y":4.172871671650441},{"x":6.369999999999909,"y":1.468995197926315},{"x":6.379999999999908,"y":4.854613922907892},{"x":6.389999999999908,"y":6.216586141187558},{"x":6.399999999999908,"y":4.949116319115817},{"x":6.409999999999908,"y":1.586140636618281},{"x":6.419999999999908,"y":2.903342594643824},{"x":6.429999999999907,"y":2.6932905485192618},{"x":6.439999999999907,"y":1.049632969337272},{"x":6.449999999999907,"y":0.705613013469506},{"x":6.459999999999907,"y":3.8498527521779886},{"x":6.4699999999999065,"y":0.9593138354018698},{"x":6.479999999999906,"y":4.144578973931108},{"x":6.489999999999906,"y":5.697913527761578},{"x":6.499999999999906,"y":3.1487302169251183},{"x":6.509999999999906,"y":1.6138917466888532},{"x":6.519999999999905,"y":0.5572803874281668},{"x":6.529999999999905,"y":6.401161787010368},{"x":6.539999999999905,"y":3.994315859588411},{"x":6.549999999999905,"y":1.8125119447096654},{"x":6.559999999999905,"y":0.8194885536052532},{"x":6.569999999999904,"y":1.3195818184274812},{"x":6.579999999999904,"y":4.542462387341144},{"x":6.589999999999904,"y":2.0108643535526287},{"x":6.599999999999904,"y":1.7474575583369507},{"x":6.6099999999999035,"y":6.328881574274118},{"x":6.619999999999903,"y":1.3689819948738091},{"x":6.629999999999903,"y":3.1813372984351806},{"x":6.639999999999903,"y":0.6759428006195939},{"x":6.649999999999903,"y":5.676362751315365},{"x":6.659999999999902,"y":4.949989791374742},{"x":6.669999999999902,"y":4.82714838890809},{"x":6.679999999999902,"y":6.440235111607413},{"x":6.689999999999902,"y":3.2953791122951746},{"x":6.699999999999902,"y":0.9537893930104108},{"x":6.709999999999901,"y":5.4670924966219125},{"x":6.719999999999901,"y":2.695341678884974},{"x":6.729999999999901,"y":0.42838922678124014},{"x":6.739999999999901,"y":0.6856203459544725},{"x":6.7499999999999005,"y":5.080879509082309},{"x":6.7599999999999,"y":1.7988945078602876},{"x":6.7699999999999,"y":1.7928292694049148},{"x":6.7799999999999,"y":1.0579777859983481},{"x":6.7899999999999,"y":0.12363001558151863},{"x":6.7999999999998995,"y":5.072815721282966},{"x":6.809999999999899,"y":4.871865690148699},{"x":6.819999999999899,"y":0.8428446558882287},{"x":6.829999999999899,"y":0.39079028419372097},{"x":6.839999999999899,"y":3.9906903931695594},{"x":6.849999999999898,"y":0.8802671896318731},{"x":6.859999999999898,"y":0.010521195432309088},{"x":6.869999999999898,"y":0.5640268813775505},{"x":6.879999999999898,"y":6.265700121442904},{"x":6.8899999999998975,"y":1.1266763498732513},{"x":6.899999999999897,"y":6.785314512544095},{"x":6.909999999999897,"y":0.487057877798723},{"x":6.919999999999897,"y":3.9303479401781978},{"x":6.929999999999897,"y":0.409941656859702},{"x":6.9399999999998965,"y":5.143654268801617},{"x":6.949999999999896,"y":4.620378374643769},{"x":6.959999999999896,"y":3.5765554872984433},{"x":6.969999999999896,"y":5.991414923984163},{"x":6.979999999999896,"y":2.5577298224277376},{"x":6.989999999999895,"y":3.687941460622465},{"x":6.999999999999895,"y":6.675547005650224},{"x":7.009999999999895,"y":2.029703046410533},{"x":7.019999999999895,"y":2.779996083969424},{"x":7.0299999999998946,"y":4.15955418474919},{"x":7.039999999999894,"y":0.7537396783105762},{"x":7.049999999999894,"y":0.7618104701724906},{"x":7.059999999999894,"y":1.1433868829538734},{"x":7.069999999999894,"y":3.9525578709075173},{"x":7.0799999999998935,"y":0.8361436887594367},{"x":7.089999999999893,"y":0.5473965664250178},{"x":7.099999999999893,"y":2.131271867565085},{"x":7.109999999999893,"y":6.747766400431408},{"x":7.119999999999893,"y":3.3350325589069625},{"x":7.129999999999892,"y":3.6935189471135974},{"x":7.139999999999892,"y":4.482638920852901},{"x":7.149999999999892,"y":6.96014071915204},{"x":7.159999999999892,"y":2.8513868356742766},{"x":7.169999999999892,"y":5.698232064880834},{"x":7.179999999999891,"y":5.434138968336816},{"x":7.189999999999891,"y":3.7767949362940354},{"x":7.199999999999891,"y":1.8823178922117751},{"x":7.209999999999891,"y":6.975251814684659},{"x":7.2199999999998905,"y":5.414023845635103},{"x":7.22999999999989,"y":4.396130717529783},{"x":7.23999999999989,"y":3.221878428235525},{"x":7.24999999999989,"y":2.0611037230542704},{"x":7.25999999999989,"y":6.126117819610812},{"x":7.269999999999889,"y":4.384717326725126},{"x":7.279999999999889,"y":3.4178233811648733},{"x":7.289999999999889,"y":1.0632163466789837},{"x":7.299999999999889,"y":5.607311558028261},{"x":7.309999999999889,"y":6.405444055243212},{"x":7.319999999999888,"y":1.1155326679971005},{"x":7.329999999999888,"y":4.894829418764613},{"x":7.339999999999888,"y":3.422129977545889},{"x":7.349999999999888,"y":2.623988883600407},{"x":7.3599999999998875,"y":1.954924542686524},{"x":7.369999999999887,"y":7.3128351585141695},{"x":7.379999999999887,"y":0.6852227938516495},{"x":7.389999999999887,"y":6.583264467300067},{"x":7.399999999999887,"y":6.9992854612420485},{"x":7.4099999999998865,"y":0.40028193699066383},{"x":7.419999999999886,"y":0.02964948089884218},{"x":7.429999999999886,"y":2.669720842232049},{"x":7.439999999999886,"y":1.6990116130592254},{"x":7.449999999999886,"y":4.428720207368497},{"x":7.459999999999885,"y":0.6180742073845705},{"x":7.469999999999885,"y":5.913706026580454},{"x":7.479999999999885,"y":3.480809607761758},{"x":7.489999999999885,"y":6.24938584464422},{"x":7.4999999999998845,"y":1.1379669480304875},{"x":7.509999999999884,"y":7.131003337865179},{"x":7.519999999999884,"y":0.4764270072722739},{"x":7.529999999999884,"y":2.7588709250746866},{"x":7.539999999999884,"y":3.0831008762141696},{"x":7.5499999999998835,"y":1.1866948760915388},{"x":7.559999999999883,"y":7.016985712567578},{"x":7.569999999999883,"y":3.846947178246566},{"x":7.579999999999883,"y":6.0545713022771785},{"x":7.589999999999883,"y":3.5681179456138943},{"x":7.599999999999882,"y":1.831971435638476},{"x":7.609999999999882,"y":4.334068259104327},{"x":7.619999999999882,"y":7.427710121816816},{"x":7.629999999999882,"y":4.793413275725009},{"x":7.6399999999998816,"y":5.398383506790628},{"x":7.649999999999881,"y":1.7562929864997467},{"x":7.659999999999881,"y":0.35915216619986284},{"x":7.669999999999881,"y":4.778489177960532},{"x":7.679999999999881,"y":1.6179860002309654},{"x":7.6899999999998805,"y":3.8084382708654285},{"x":7.69999999999988,"y":4.763419966397642},{"x":7.70999999999988,"y":0.9506565486559073},{"x":7.71999999999988,"y":6.991505802681903},{"x":7.72999999999988,"y":3.9946881281789457},{"x":7.739999999999879,"y":5.523223693502714},{"x":7.749999999999879,"y":3.4942200942664874},{"x":7.759999999999879,"y":0.7805090999128286},{"x":7.769999999999879,"y":6.792153926074055},{"x":7.779999999999879,"y":0.5247338677089654},{"x":7.789999999999878,"y":2.134446100828869},{"x":7.799999999999878,"y":7.61304913514713},{"x":7.809999999999878,"y":6.36950336669301},{"x":7.819999999999878,"y":7.1808762806988815},{"x":7.8299999999998775,"y":5.454057655065537},{"x":7.839999999999877,"y":0.5105155297627533},{"x":7.849999999999877,"y":2.5822487450522598},{"x":7.859999999999877,"y":6.814705050211519},{"x":7.869999999999877,"y":7.553248100890202},{"x":7.879999999999876,"y":5.102931860153244},{"x":7.889999999999876,"y":6.771105273545683},{"x":7.899999999999876,"y":4.733655863891144},{"x":7.909999999999876,"y":4.312637996087194},{"x":7.919999999999876,"y":5.450735653050234},{"x":7.929999999999875,"y":7.432513818568855},{"x":7.939999999999875,"y":2.066445236252012},{"x":7.949999999999875,"y":0.5611422342359056},{"x":7.959999999999875,"y":5.715345548566473},{"x":7.9699999999998745,"y":0.07344028376157195},{"x":7.979999999999874,"y":0.47007549506234736},{"x":7.989999999999874,"y":0.019853544541840393},{"x":7.999999999999874,"y":0.8310629793517628},{"x":8.009999999999874,"y":1.7684501853157404},{"x":8.019999999999873,"y":1.8973269077208752},{"x":8.029999999999873,"y":7.041288852826024},{"x":8.039999999999873,"y":3.8795789566137056},{"x":8.049999999999873,"y":0.3182846125188164},{"x":8.059999999999873,"y":7.137761189009596},{"x":8.069999999999872,"y":3.333193637365638},{"x":8.079999999999872,"y":6.00804800378902},{"x":8.089999999999872,"y":1.0282454866635278},{"x":8.099999999999872,"y":1.8201369991482945},{"x":8.109999999999872,"y":5.15003950441022},{"x":8.119999999999871,"y":6.814872175599068},{"x":8.129999999999871,"y":0.16413128901373933},{"x":8.139999999999871,"y":6.903736653582432},{"x":8.14999999999987,"y":7.115900418076526},{"x":8.15999999999987,"y":7.106484651454529},{"x":8.16999999999987,"y":6.705751901431591},{"x":8.17999999999987,"y":1.2429291451423925},{"x":8.18999999999987,"y":2.5220398668902653},{"x":8.19999999999987,"y":2.9160484656267647},{"x":8.20999999999987,"y":2.2756858041584356},{"x":8.21999999999987,"y":5.035846659407277},{"x":8.229999999999869,"y":4.453791476880923},{"x":8.239999999999869,"y":2.583591807994696},{"x":8.249999999999869,"y":2.886431144391351},{"x":8.259999999999868,"y":6.956832982375714},{"x":8.269999999999868,"y":4.376669403893347},{"x":8.279999999999868,"y":6.879181605215516},{"x":8.289999999999868,"y":4.964942998149749},{"x":8.299999999999867,"y":3.1358258795744023},{"x":8.309999999999867,"y":0.8335919518402937},{"x":8.319999999999867,"y":0.4387904237372596},{"x":8.329999999999867,"y":4.20224192295152},{"x":8.339999999999867,"y":8.138569792303375},{"x":8.349999999999866,"y":1.5861657556449815},{"x":8.359999999999866,"y":1.1690487921465882},{"x":8.369999999999866,"y":1.14981592455259},{"x":8.379999999999866,"y":6.5120130048675025},{"x":8.389999999999866,"y":7.355624398331649},{"x":8.399999999999865,"y":4.874582777382538},{"x":8.409999999999865,"y":3.6464419438576896},{"x":8.419999999999865,"y":0.7019289042077081},{"x":8.429999999999865,"y":2.3357803853904984},{"x":8.439999999999864,"y":3.3226227024284762},{"x":8.449999999999864,"y":5.871289020698836},{"x":8.459999999999864,"y":5.457214639162601},{"x":8.469999999999864,"y":3.1137046459701927},{"x":8.479999999999864,"y":1.5565337286820549},{"x":8.489999999999863,"y":7.104271938573498},{"x":8.499999999999863,"y":2.5199568960294445},{"x":8.509999999999863,"y":7.166311639070812},{"x":8.519999999999863,"y":5.975355893795453},{"x":8.529999999999863,"y":2.8750660905265035},{"x":8.539999999999862,"y":4.234721473440319},{"x":8.549999999999862,"y":1.5650867865264249},{"x":8.559999999999862,"y":4.862858935084451},{"x":8.569999999999862,"y":6.830499227519712},{"x":8.579999999999862,"y":7.107114022640628},{"x":8.589999999999861,"y":1.9693623919165852},{"x":8.599999999999861,"y":0.5895105357431474},{"x":8.60999999999986,"y":4.518687654102903},{"x":8.61999999999986,"y":6.9929793630337365},{"x":8.62999999999986,"y":5.653307526050116},{"x":8.63999999999986,"y":4.136621722706541},{"x":8.64999999999986,"y":4.939787096617512},{"x":8.65999999999986,"y":3.518148705822111},{"x":8.66999999999986,"y":8.165669655748026},{"x":8.67999999999986,"y":1.8962858369005757},{"x":8.68999999999986,"y":1.7800846990308237},{"x":8.699999999999859,"y":4.226919368719593},{"x":8.709999999999859,"y":5.386773691294555},{"x":8.719999999999859,"y":1.996041797946463},{"x":8.729999999999858,"y":7.89086563368231},{"x":8.739999999999858,"y":1.127945611953833},{"x":8.749999999999858,"y":6.786850213905449},{"x":8.759999999999858,"y":3.2359970974484873},{"x":8.769999999999857,"y":4.8385454817435045},{"x":8.779999999999857,"y":0.2414483231496114},{"x":8.789999999999857,"y":1.025942518365114},{"x":8.799999999999857,"y":2.6209198266730285},{"x":8.809999999999857,"y":6.02047291863414},{"x":8.819999999999856,"y":4.164097289129989},{"x":8.829999999999856,"y":0.8709121095803296},{"x":8.839999999999856,"y":2.2530579191150646},{"x":8.849999999999856,"y":2.674063203797171},{"x":8.859999999999856,"y":3.190103129916653},{"x":8.869999999999855,"y":0.33193773256997716},{"x":8.879999999999855,"y":3.6156785812752212},{"x":8.889999999999855,"y":0.018041510839812234},{"x":8.899999999999855,"y":3.671454584454948},{"x":8.909999999999854,"y":3.611877967185461},{"x":8.919999999999854,"y":8.215019294003536},{"x":8.929999999999854,"y":7.796038652042019},{"x":8.939999999999854,"y":2.91937285798715},{"x":8.949999999999854,"y":8.329452191209638},{"x":8.959999999999853,"y":6.592026564892616},{"x":8.969999999999853,"y":8.447096495562045},{"x":8.979999999999853,"y":8.055408189891452},{"x":8.989999999999853,"y":4.723863691846962},{"x":8.999999999999853,"y":0.7774193874198837},{"x":9.009999999999852,"y":6.347308480011832},{"x":9.019999999999852,"y":2.166690256512381},{"x":9.029999999999852,"y":4.646780257873708},{"x":9.039999999999852,"y":1.656743270022525},{"x":9.049999999999851,"y":5.8643440703022325},{"x":9.059999999999851,"y":1.1381533836636275},{"x":9.069999999999851,"y":1.0399640090871476},{"x":9.07999999999985,"y":0.9086956263192252},{"x":9.08999999999985,"y":4.724470579224962},{"x":9.09999999999985,"y":0.7375957441669981},{"x":9.10999999999985,"y":5.66582868978124},{"x":9.11999999999985,"y":7.69259314199651},{"x":9.12999999999985,"y":4.662415252156544},{"x":9.13999999999985,"y":3.409692185605036},{"x":9.14999999999985,"y":2.816288645040154},{"x":9.15999999999985,"y":7.161911051634419},{"x":9.169999999999849,"y":4.060478554221811},{"x":9.179999999999849,"y":1.705077793029636},{"x":9.189999999999849,"y":5.434951858547066},{"x":9.199999999999848,"y":0.8828682491964102},{"x":9.209999999999848,"y":4.757241020859892},{"x":9.219999999999848,"y":2.962824799368668},{"x":9.229999999999848,"y":6.111495520971406},{"x":9.239999999999847,"y":6.292551955598708},{"x":9.249999999999847,"y":1.5304777880479654},{"x":9.259999999999847,"y":4.891591800251573},{"x":9.269999999999847,"y":3.317458339790448},{"x":9.279999999999847,"y":9.072563355758858},{"x":9.289999999999846,"y":0.8460293534206856},{"x":9.299999999999846,"y":1.1688836973342664},{"x":9.309999999999846,"y":7.172169849100504},{"x":9.319999999999846,"y":5.014144427649479},{"x":9.329999999999846,"y":5.646853312094293},{"x":9.339999999999845,"y":8.771760767533808},{"x":9.349999999999845,"y":6.968635828031535},{"x":9.359999999999845,"y":6.96168606840819},{"x":9.369999999999845,"y":9.326814969319742},{"x":9.379999999999844,"y":6.438463770178991},{"x":9.389999999999844,"y":1.6143930300556601},{"x":9.399999999999844,"y":9.255946616421479},{"x":9.409999999999844,"y":7.01446158851624},{"x":9.419999999999844,"y":8.111259346333918},{"x":9.429999999999843,"y":6.754139419563133},{"x":9.439999999999843,"y":7.348707279118209},{"x":9.449999999999843,"y":0.8955534256314677},{"x":9.459999999999843,"y":8.038900481265493},{"x":9.469999999999843,"y":4.181851160447709},{"x":9.479999999999842,"y":7.279468881409791},{"x":9.489999999999842,"y":5.682019229351771},{"x":9.499999999999842,"y":4.634886994955726},{"x":9.509999999999842,"y":5.532592479948148},{"x":9.519999999999841,"y":2.5246503666329465},{"x":9.529999999999841,"y":0.6217342189788361},{"x":9.539999999999841,"y":4.867749638469798},{"x":9.54999999999984,"y":5.580826665995294},{"x":9.55999999999984,"y":0.3030724200526778},{"x":9.56999999999984,"y":4.650618096084057},{"x":9.57999999999984,"y":3.6943310847486748},{"x":9.58999999999984,"y":4.085244522668825},{"x":9.59999999999984,"y":2.660826502933344},{"x":9.60999999999984,"y":3.4084796271788464},{"x":9.61999999999984,"y":8.049021763035412},{"x":9.62999999999984,"y":1.1124218455738624},{"x":9.639999999999839,"y":4.276402722474884},{"x":9.649999999999839,"y":0.6853404193365019},{"x":9.659999999999838,"y":9.103621545859701},{"x":9.669999999999838,"y":2.4055347286217685},{"x":9.679999999999838,"y":1.9157168278948655},{"x":9.689999999999838,"y":9.3481048175681},{"x":9.699999999999838,"y":0.9264227728734049},{"x":9.709999999999837,"y":4.412360544541769},{"x":9.719999999999837,"y":1.938908972880634},{"x":9.729999999999837,"y":8.210944124521943},{"x":9.739999999999837,"y":2.3877854138821513},{"x":9.749999999999837,"y":5.661405837265113},{"x":9.759999999999836,"y":6.169003304604087},{"x":9.769999999999836,"y":8.499110536820224},{"x":9.779999999999836,"y":7.755356005149888},{"x":9.789999999999836,"y":5.008714179272278},{"x":9.799999999999836,"y":1.8892014497304808},{"x":9.809999999999835,"y":9.737591846524689},{"x":9.819999999999835,"y":2.267696723125725},{"x":9.829999999999835,"y":5.643215974574373},{"x":9.839999999999835,"y":2.13325516448791},{"x":9.849999999999834,"y":1.3855769162824358},{"x":9.859999999999834,"y":1.0829376951594958},{"x":9.869999999999834,"y":5.897899517671505},{"x":9.879999999999834,"y":8.724502490573142},{"x":9.889999999999834,"y":8.467592262957647},{"x":9.899999999999833,"y":2.0816308894956026},{"x":9.909999999999833,"y":8.170096376501508},{"x":9.919999999999833,"y":9.856269431242795},{"x":9.929999999999833,"y":0.8549354864604157},{"x":9.939999999999833,"y":8.452776971645347},{"x":9.949999999999832,"y":0.3192934817032741},{"x":9.959999999999832,"y":6.461941011123203},{"x":9.969999999999832,"y":3.979129787901795},{"x":9.979999999999832,"y":1.1015236616972586},{"x":9.989999999999831,"y":4.247598769573557},{"x":9.999999999999831,"y":2.5095700122491915}]},{"name":"af9febcd-43c7-4018-acdf-ef02a96c8669","values":[{"x":0,"y":0.0},{"x":0.10000000149011612,"y":9.966711523935131E-4},{"x":0.20000000298023224,"y":0.0078939009494512},{"x":0.30000000447034836,"y":0.026199658911198156},{"x":0.4000000059604645,"y":0.06065866074476158},{"x":0.5000000074505806,"y":0.11492442838019619},{"x":0.6000000089406967,"y":0.19129268150732823},{"x":0.7000000104308128,"y":0.2905115115092036},{"x":0.800000011920929,"y":0.41167982458769947},{"x":0.9000000134110451,"y":0.5522409625952036},{"x":1.0000000149011612,"y":0.7080734423742749},{"x":1.1000000163912773,"y":0.8736756420867389},{"x":1.2000000178813934,"y":1.042436259352125},{"x":1.3000000193715096,"y":1.2069777206570362},{"x":1.4000000208616257,"y":1.3595556685107744},{"x":1.5000000223517418,"y":1.4924943994216495},{"x":1.600000023841858,"y":1.5986358422305347},{"x":1.700000025331974,"y":1.6717784775992683},{"x":1.8000000268220901,"y":1.7070825787734616},{"x":1.9000000283122063,"y":1.701419318758029},{"x":2.0000000298023224,"y":1.653643600395876},{"x":2.1000000312924385,"y":1.5647738284498447},{"x":2.2000000327825546,"y":1.4380661097740322},{"x":2.3000000342726707,"y":1.2789753467037075},{"x":2.400000035762787,"y":1.0950011506879587},{"x":2.500000037252903,"y":0.8954221922070159},{"x":2.600000038743019,"y":0.6909282486131232},{"x":2.7000000402331352,"y":0.49316454088117767},{"x":2.8000000417232513,"y":0.3142077010199231},{"x":2.9000000432133675,"y":0.16599664468526332},{"x":3.0000000447034836,"y":0.05974453344218124},{"x":3.1000000461935997,"y":0.005359737795470119},{"x":3.200000047683716,"y":0.010904147933380758},{"x":3.300000049173832,"y":0.08211627504701455},{"x":3.400000050663948,"y":0.22202435482841815},{"x":3.500000052154064,"y":0.43067118124254894},{"x":3.6000000536441803,"y":0.7049677976191696},{"x":3.7000000551342964,"y":1.0386876427986966},{"x":3.8000000566244125,"y":1.4226065285621616},{"x":3.9000000581145287,"y":1.8447871837088208},{"x":4.000000059604645,"y":2.2910003376371773},{"x":4.100000061094761,"y":2.7452677415667375},{"x":4.200000062584877,"y":3.190506445824144},{"x":4.300000064074993,"y":3.6092483578629087},{"x":4.400000065565109,"y":3.9844048590485874},{"x":4.500000067055225,"y":4.300043277672282},{"x":4.6000000685453415,"y":4.542140467191822},{"x":4.700000070035458,"y":4.699278726962665},{"x":4.800000071525574,"y":4.7632508650334975},{"x":4.90000007301569,"y":4.729544306014563},{"x":5.000000074505806,"y":4.597678688538202},{"x":5.100000075995922,"y":4.3713772065510605},{"x":5.200000077486038,"y":4.05855879623405},{"x":5.300000078976154,"y":3.671145874000224},{"x":5.4000000804662704,"y":3.224690369246717},{"x":5.500000081956387,"y":2.7378289205739783},{"x":5.600000083446503,"y":2.2315859569905676},{"x":5.700000084936619,"y":1.7285506115833968},{"x":5.800000086426735,"y":1.2519596798650976},{"x":5.900000087916851,"y":0.8247238395284076},{"x":6.000000089406967,"y":0.46843784294268836},{"x":6.100000090897083,"y":0.20241718996047256},{"x":6.200000092387199,"y":0.04280376871522386},{"x":6.3000000938773155,"y":0.0017810755840114996},{"x":6.400000095367432,"y":0.08693593236609283},{"x":6.500000096857548,"y":0.30079822929542793},{"x":6.600000098347664,"y":0.6405833336437111},{"x":6.70000009983778,"y":1.0981536817068416},{"x":6.800000101327896,"y":1.6602070423863446},{"x":6.900000102818012,"y":2.308689374977461},{"x":7.000000104308128,"y":3.0214205045940603},{"x":7.1000001057982445,"y":3.77291142228647},{"x":7.200000107288361,"y":4.5353432958990245},{"x":7.300000108778477,"y":5.2796706428663045},{"x":7.400000110268593,"y":5.976804919197644},{"x":7.500000111758709,"y":6.598830316615712},{"x":7.600000113248825,"y":7.120201060180292},{"x":7.700000114738941,"y":7.518869109857416},{"x":7.800000116229057,"y":7.777292951795491},{"x":7.900000117719173,"y":7.883282087047769},{"x":8.00000011920929,"y":7.830637763413025},{"x":8.100000120699406,"y":7.619558237636681},{"x":8.200000122189522,"y":7.256786107003342},{"x":8.300000123679638,"y":6.755485647926689},{"x":8.400000125169754,"y":6.134849225333044},{"x":8.50000012665987,"y":5.419443232426153},{"x":8.600000128149986,"y":4.638315207789705},{"x":8.700000129640102,"y":3.823894277701979},{"x":8.800000131130219,"y":3.0107264283034625},{"x":8.900000132620335,"y":2.2340939073053003},{"x":9.00000013411045,"y":1.528573929242129},{"x":9.100000135600567,"y":0.9265955276366005},{"x":9.200000137090683,"y":0.45705466575180215},{"x":9.300000138580799,"y":0.1440464866800439},{"x":9.400000140070915,"y":0.0057698589043499015},{"x":9.500000141561031,"y":0.05365326597234279},{"x":9.600000143051147,"y":0.2917428116738198},{"x":9.700000144541264,"y":0.7163829732921025},{"x":9.80000014603138,"y":1.316209126913524},{"x":9.900000147521496,"y":2.0724582507409313}]}],"marks":[{"type":"symbol","from":{"data":"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"fill":{"value":"steelblue"},"fillOpacity":{"value":0.3}},"update":{"shape":"circle","size":{"value":50},"stroke":{"value":"transparent"}},"hover":{"size":{"value":150},"stroke":{"value":"white"}}}},{"type":"line","from":{"data":"af9febcd-43c7-4018-acdf-ef02a96c8669"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"#FF29D2"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}, :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :data ({:name \"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6\", :values ({:x 0, :y 0.0} {:x 0.01, :y 5.398421659901543E-4} {:x 0.02, :y 0.0022078270776596143} {:x 0.03, :y 0.014699722478001756} {:x 0.04, :y 0.03880856675762969} {:x 0.05, :y 0.048497360775456806} {:x 0.060000000000000005, :y 0.04133135892065133} {:x 0.07, :y 0.03860803263755669} {:x 0.08, :y 0.022559157093072244} {:x 0.09, :y 0.05937787616411635} {:x 0.09999999999999999, :y 0.02034394553636346} {:x 0.10999999999999999, :y 0.02073904234513112} {:x 0.11999999999999998, :y 0.11775254558807377} {:x 0.12999999999999998, :y 0.03580653538221089} {:x 0.13999999999999999, :y 0.06989600617536991} {:x 0.15, :y 0.01558731613371262} {:x 0.16, :y 0.0040657512580307125} {:x 0.17, :y 0.03971488248370004} {:x 0.18000000000000002, :y 0.08480104744112552} {:x 0.19000000000000003, :y 0.10671004969858121} {:x 0.20000000000000004, :y 0.039686630288586974} {:x 0.21000000000000005, :y 0.15941675191181415} {:x 0.22000000000000006, :y 0.20880863068054883} {:x 0.23000000000000007, :y 0.05251564303355356} {:x 0.24000000000000007, :y 0.09175238395112068} {:x 0.25000000000000006, :y 0.12426339002326894} {:x 0.26000000000000006, :y 0.007519097735057327} {:x 0.2700000000000001, :y 0.23331696813248287} {:x 0.2800000000000001, :y 0.20107071335214616} {:x 0.2900000000000001, :y 0.2239948346672196} {:x 0.3000000000000001, :y 0.0700250006951655} {:x 0.3100000000000001, :y 0.06200552657621937} {:x 0.3200000000000001, :y 0.2930464222793822} {:x 0.3300000000000001, :y 0.19828331424359558} {:x 0.34000000000000014, :y 0.155284233498906} {:x 0.35000000000000014, :y 0.0014449503211177279} {:x 0.36000000000000015, :y 0.10795721464714836} {:x 0.37000000000000016, :y 0.22569516309589507} {:x 0.38000000000000017, :y 0.3743761401050732} {:x 0.3900000000000002, :y 0.014025961318826257} {:x 0.4000000000000002, :y 0.31591928627169424} {:x 0.4100000000000002, :y 0.2528794364180144} {:x 0.4200000000000002, :y 0.007165129592618084} {:x 0.4300000000000002, :y 0.13527029040556288} {:x 0.4400000000000002, :y 0.39755765476163313} {:x 0.45000000000000023, :y 0.3267253841288474} {:x 0.46000000000000024, :y 0.00399578150781339} {:x 0.47000000000000025, :y 0.41641793554987233} {:x 0.48000000000000026, :y 0.02304603122719935} {:x 0.49000000000000027, :y 0.22701574673147695} {:x 0.5000000000000002, :y 0.4323697227427657} {:x 0.5100000000000002, :y 0.03308042988666946} {:x 0.5200000000000002, :y 0.4223084562810144} {:x 0.5300000000000002, :y 0.33723982455883983} {:x 0.5400000000000003, :y 0.2476071348696873} {:x 0.5500000000000003, :y 0.5207930969106707} {:x 0.5600000000000003, :y 0.40027810665791375} {:x 0.5700000000000003, :y 0.18921838215219203} {:x 0.5800000000000003, :y 0.12777526982385282} {:x 0.5900000000000003, :y 0.4961504513193089} {:x 0.6000000000000003, :y 0.5256889101533392} {:x 0.6100000000000003, :y 0.4255323503258928} {:x 0.6200000000000003, :y 0.34917193269507474} {:x 0.6300000000000003, :y 0.09666229878658686} {:x 0.6400000000000003, :y 0.24243226829997416} {:x 0.6500000000000004, :y 0.6049387862131121} {:x 0.6600000000000004, :y 0.48188462723335723} {:x 0.6700000000000004, :y 0.11602277526246256} {:x 0.6800000000000004, :y 0.4515442595325231} {:x 0.6900000000000004, :y 0.14250026795884216} {:x 0.7000000000000004, :y 0.1679047106735959} {:x 0.7100000000000004, :y 0.10664055147998841} {:x 0.7200000000000004, :y 0.12189856710413462} {:x 0.7300000000000004, :y 0.08054881611368087} {:x 0.7400000000000004, :y 0.3594617231597144} {:x 0.7500000000000004, :y 0.03053663303760461} {:x 0.7600000000000005, :y 0.42116675815983806} {:x 0.7700000000000005, :y 0.3913510551251715} {:x 0.7800000000000005, :y 0.15515513148666304} {:x 0.7900000000000005, :y 0.23557521763440045} {:x 0.8000000000000005, :y 0.22807024631913433} {:x 0.8100000000000005, :y 0.5599394665984069} {:x 0.8200000000000005, :y 0.3485990764880174} {:x 0.8300000000000005, :y 0.10895414325371701} {:x 0.8400000000000005, :y 0.044887785984757275} {:x 0.8500000000000005, :y 0.04852345474114983} {:x 0.8600000000000005, :y 0.3311615142143673} {:x 0.8700000000000006, :y 0.626444139685543} {:x 0.8800000000000006, :y 0.4280986667523344} {:x 0.8900000000000006, :y 0.49469482813826193} {:x 0.9000000000000006, :y 0.0794703957589415} {:x 0.9100000000000006, :y 0.8965295864070829} {:x 0.9200000000000006, :y 0.8074337268700722} {:x 0.9300000000000006, :y 0.5767947715995505} {:x 0.9400000000000006, :y 0.8021782255539507} {:x 0.9500000000000006, :y 0.6058701534647117} {:x 0.9600000000000006, :y 0.2019760711301762} {:x 0.9700000000000006, :y 0.3138130864091943} {:x 0.9800000000000006, :y 0.9315211374811561} {:x 0.9900000000000007, :y 0.39886845010829597} {:x 1.0000000000000007, :y 0.5787987415228129} {:x 1.0100000000000007, :y 0.06654596870086399} {:x 1.0200000000000007, :y 0.4359800672021171} {:x 1.0300000000000007, :y 0.5220990232252575} {:x 1.0400000000000007, :y 0.22744898287205026} {:x 1.0500000000000007, :y 0.1839460610864435} {:x 1.0600000000000007, :y 0.8533498744499903} {:x 1.0700000000000007, :y 0.7964232454502889} {:x 1.0800000000000007, :y 0.8148872702105039} {:x 1.0900000000000007, :y 0.46615618394327674} {:x 1.1000000000000008, :y 0.8112464889874446} {:x 1.1100000000000008, :y 0.7782871234715933} {:x 1.1200000000000008, :y 1.0527654913490003} {:x 1.1300000000000008, :y 0.7261292804917082} {:x 1.1400000000000008, :y 1.0470336429766902} {:x 1.1500000000000008, :y 1.1494723566786214} {:x 1.1600000000000008, :y 0.14467921459845276} {:x 1.1700000000000008, :y 0.21749845881747512} {:x 1.1800000000000008, :y 0.5738285611629873} {:x 1.1900000000000008, :y 0.8939457463992905} {:x 1.2000000000000008, :y 1.097759144509554} {:x 1.2100000000000009, :y 0.5199737485772478} {:x 1.2200000000000009, :y 1.1361498025098085} {:x 1.2300000000000009, :y 0.9339290018658525} {:x 1.2400000000000009, :y 0.013170624699442524} {:x 1.2500000000000009, :y 1.1975083340342207} {:x 1.260000000000001, :y 0.05040176991511057} {:x 1.270000000000001, :y 0.49738874403059735} {:x 1.280000000000001, :y 0.006272286170290049} {:x 1.290000000000001, :y 0.7512917046186459} {:x 1.300000000000001, :y 0.9442477734719107} {:x 1.310000000000001, :y 0.608326610113623} {:x 1.320000000000001, :y 0.0018431595289033456} {:x 1.330000000000001, :y 0.05179694205315745} {:x 1.340000000000001, :y 0.9275072588839394} {:x 1.350000000000001, :y 1.242030075250892} {:x 1.360000000000001, :y 0.3754220768922943} {:x 1.370000000000001, :y 0.7841274640020042} {:x 1.380000000000001, :y 1.0180233780495533} {:x 1.390000000000001, :y 0.7823277674900635} {:x 1.400000000000001, :y 1.180914738941596} {:x 1.410000000000001, :y 0.6812803302679422} {:x 1.420000000000001, :y 0.2814848396591268} {:x 1.430000000000001, :y 0.8893778297397118} {:x 1.440000000000001, :y 1.029711715142625} {:x 1.450000000000001, :y 0.16316567743217653} {:x 1.460000000000001, :y 0.6169484752586413} {:x 1.470000000000001, :y 0.6671026626109712} {:x 1.480000000000001, :y 1.2401247569433116} {:x 1.490000000000001, :y 0.9977965197954077} {:x 1.500000000000001, :y 0.6423003817528923} {:x 1.5100000000000011, :y 0.4015719674679796} {:x 1.5200000000000011, :y 0.2904903343669004} {:x 1.5300000000000011, :y 1.1774878348363937} {:x 1.5400000000000011, :y 1.3386093847426133} {:x 1.5500000000000012, :y 0.5379247562769631} {:x 1.5600000000000012, :y 0.4373065514877508} {:x 1.5700000000000012, :y 0.8496824793175859} {:x 1.5800000000000012, :y 0.37177065668615594} {:x 1.5900000000000012, :y 1.466311431866839} {:x 1.6000000000000012, :y 0.5137638531359638} {:x 1.6100000000000012, :y 1.1116496265143185} {:x 1.6200000000000012, :y 1.236506524714334} {:x 1.6300000000000012, :y 0.6793292736301372} {:x 1.6400000000000012, :y 1.1965002147262713} {:x 1.6500000000000012, :y 1.5832819273482148} {:x 1.6600000000000013, :y 1.4741886262968147} {:x 1.6700000000000013, :y 1.0170185832938818} {:x 1.6800000000000013, :y 0.251860148566999} {:x 1.6900000000000013, :y 0.015050067530115584} {:x 1.7000000000000013, :y 0.31456832748306046} {:x 1.7100000000000013, :y 1.5031820227052617} {:x 1.7200000000000013, :y 0.2398583681636518} {:x 1.7300000000000013, :y 1.501356283933557} {:x 1.7400000000000013, :y 1.154851801008506} {:x 1.7500000000000013, :y 0.9110863838085098} {:x 1.7600000000000013, :y 1.4169516790267258} {:x 1.7700000000000014, :y 1.3255645419220343} {:x 1.7800000000000014, :y 1.7101326104114933} {:x 1.7900000000000014, :y 1.560004676217984} {:x 1.8000000000000014, :y 0.25009834707218215} {:x 1.8100000000000014, :y 0.3643703467707408} {:x 1.8200000000000014, :y 0.8194971543030658} {:x 1.8300000000000014, :y 1.5390560828888344} {:x 1.8400000000000014, :y 0.9869376717137699} {:x 1.8500000000000014, :y 0.5384133224156971} {:x 1.8600000000000014, :y 1.0046479319101662} {:x 1.8700000000000014, :y 0.49390361828273727} {:x 1.8800000000000014, :y 1.8727532234864914} {:x 1.8900000000000015, :y 1.3956238372008352} {:x 1.9000000000000015, :y 0.8651448092785847} {:x 1.9100000000000015, :y 0.7575713077666987} {:x 1.9200000000000015, :y 1.5366895686385642} {:x 1.9300000000000015, :y 1.3862370640323443} {:x 1.9400000000000015, :y 1.208848087887971} {:x 1.9500000000000015, :y 1.5214487073305465} {:x 1.9600000000000015, :y 0.09997483236211901} {:x 1.9700000000000015, :y 0.6562013326642258} {:x 1.9800000000000015, :y 0.7303541813491781} {:x 1.9900000000000015, :y 1.2989556343317417} {:x 2.0000000000000013, :y 0.600162613784154} {:x 2.010000000000001, :y 1.2213511047378105} {:x 2.020000000000001, :y 0.016652889139142} {:x 2.0300000000000007, :y 0.11484875815111481} {:x 2.0400000000000005, :y 1.3301417396164783} {:x 2.0500000000000003, :y 0.45071174552966803} {:x 2.06, :y 1.190394742825987} {:x 2.07, :y 1.6702602050847142} {:x 2.0799999999999996, :y 0.2913759613544047} {:x 2.0899999999999994, :y 0.17615060289890672} {:x 2.099999999999999, :y 0.8095475226645792} {:x 2.109999999999999, :y 1.099770418131657} {:x 2.1199999999999988, :y 1.8700250970665462} {:x 2.1299999999999986, :y 0.7097887858914748} {:x 2.1399999999999983, :y 0.600335061770539} {:x 2.149999999999998, :y 2.1146281789222985} {:x 2.159999999999998, :y 1.0947647297737115} {:x 2.1699999999999977, :y 0.44263723142724} {:x 2.1799999999999975, :y 0.10069409032864259} {:x 2.1899999999999973, :y 0.4781472821910741} {:x 2.199999999999997, :y 1.3387650104301185} {:x 2.209999999999997, :y 1.840325011721693} {:x 2.2199999999999966, :y 1.3265525388735941} {:x 2.2299999999999964, :y 1.7478224065291768} {:x 2.239999999999996, :y 1.197906778386139} {:x 2.249999999999996, :y 2.2070402584204603} {:x 2.259999999999996, :y 1.6555280107346095} {:x 2.2699999999999956, :y 2.176325572496733} {:x 2.2799999999999954, :y 1.4654109141402145} {:x 2.289999999999995, :y 0.6294100180977333} {:x 2.299999999999995, :y 0.06466695644617297} {:x 2.3099999999999947, :y 1.6027633332694997} {:x 2.3199999999999945, :y 0.6221271422384828} {:x 2.3299999999999943, :y 1.4605028337207737} {:x 2.339999999999994, :y 0.2330919209909163} {:x 2.349999999999994, :y 0.9988933766231244} {:x 2.3599999999999937, :y 0.10427843705688346} {:x 2.3699999999999934, :y 0.29294529557853677} {:x 2.3799999999999932, :y 2.019400744508853} {:x 2.389999999999993, :y 0.254923852818212} {:x 2.399999999999993, :y 2.1089829997212712} {:x 2.4099999999999926, :y 1.971805610846062} {:x 2.4199999999999924, :y 1.2398605446996118} {:x 2.429999999999992, :y 0.2685398232877039} {:x 2.439999999999992, :y 2.084875725697424} {:x 2.4499999999999917, :y 1.3249070706011035} {:x 2.4599999999999915, :y 0.7393690165988231} {:x 2.4699999999999913, :y 2.3714272935624487} {:x 2.479999999999991, :y 1.363132338004649} {:x 2.489999999999991, :y 0.1430992809905421} {:x 2.4999999999999907, :y 0.3753089323927991} {:x 2.5099999999999905, :y 0.5522915436728815} {:x 2.5199999999999902, :y 2.005652258400631} {:x 2.52999999999999, :y 1.6382287090645309} {:x 2.53999999999999, :y 1.6973020155615255} {:x 2.5499999999999896, :y 0.6266955402613141} {:x 2.5599999999999894, :y 1.7545887536194773} {:x 2.569999999999989, :y 1.144617301419211} {:x 2.579999999999989, :y 1.6559775680920183} {:x 2.5899999999999888, :y 2.364471757815446} {:x 2.5999999999999885, :y 1.8733233802749822} {:x 2.6099999999999883, :y 1.8232748758868798} {:x 2.619999999999988, :y 1.5437032033681404} {:x 2.629999999999988, :y 2.3474945226318678} {:x 2.6399999999999877, :y 0.9458658816167869} {:x 2.6499999999999875, :y 1.1393417476805132} {:x 2.6599999999999873, :y 1.8722868116208555} {:x 2.669999999999987, :y 1.376730636284824} {:x 2.679999999999987, :y 2.2620226330841944} {:x 2.6899999999999866, :y 2.3148783449075694} {:x 2.6999999999999864, :y 0.5394458008520994} {:x 2.709999999999986, :y 2.1102224667066625} {:x 2.719999999999986, :y 0.4120140576233859} {:x 2.7299999999999858, :y 0.9535917270486317} {:x 2.7399999999999856, :y 1.379853313752464} {:x 2.7499999999999853, :y 1.6054913234041364} {:x 2.759999999999985, :y 0.13903596442723373} {:x 2.769999999999985, :y 1.483911891084266} {:x 2.7799999999999847, :y 2.670310515420634} {:x 2.7899999999999845, :y 2.4432887685654334} {:x 2.7999999999999843, :y 2.2640519868731666} {:x 2.809999999999984, :y 2.344508881514406} {:x 2.819999999999984, :y 2.034562715233205} {:x 2.8299999999999836, :y 2.6671605282375443} {:x 2.8399999999999834, :y 2.27636945681668} {:x 2.849999999999983, :y 1.2043664602113162} {:x 2.859999999999983, :y 0.7439680138768742} {:x 2.869999999999983, :y 2.237429831019891} {:x 2.8799999999999826, :y 1.0087972894171333} {:x 2.8899999999999824, :y 1.1204089926063538} {:x 2.899999999999982, :y 1.548149742002961} {:x 2.909999999999982, :y 0.24144852378519424} {:x 2.9199999999999817, :y 1.2799048658293524} {:x 2.9299999999999815, :y 2.211907526568156} {:x 2.9399999999999813, :y 0.42740788621425235} {:x 2.949999999999981, :y 0.38680780865122066} {:x 2.959999999999981, :y 1.68059697726399} {:x 2.9699999999999807, :y 2.094973226476324} {:x 2.9799999999999804, :y 2.9485399268142847} {:x 2.9899999999999802, :y 1.6331464263194044} {:x 2.99999999999998, :y 2.720394781158697} {:x 3.00999999999998, :y 1.2348811250424423} {:x 3.0199999999999796, :y 2.3403187750316805} {:x 3.0299999999999794, :y 2.063740402882318} {:x 3.039999999999979, :y 0.16296315966437303} {:x 3.049999999999979, :y 1.701249706571135} {:x 3.0599999999999787, :y 1.536705284239315} {:x 3.0699999999999785, :y 0.5227071454717113} {:x 3.0799999999999783, :y 0.6030508948569143} {:x 3.089999999999978, :y 2.5637417022333957} {:x 3.099999999999978, :y 1.3774205550696368} {:x 3.1099999999999777, :y 0.009906951832155076} {:x 3.1199999999999775, :y 0.5606148577232218} {:x 3.1299999999999772, :y 0.9264374134816045} {:x 3.139999999999977, :y 1.359287442611299} {:x 3.149999999999977, :y 1.0459760475883617} {:x 3.1599999999999766, :y 0.06497811245655262} {:x 3.1699999999999764, :y 2.979872579729387} {:x 3.179999999999976, :y 0.8902533077741367} {:x 3.189999999999976, :y 0.34687845133543205} {:x 3.1999999999999758, :y 2.1849300706820034} {:x 3.2099999999999755, :y 0.9634244064201346} {:x 3.2199999999999753, :y 3.163739953338034} {:x 3.229999999999975, :y 1.813419179105547} {:x 3.239999999999975, :y 1.7422077895722903} {:x 3.2499999999999747, :y 1.6897718705452938} {:x 3.2599999999999745, :y 3.0964249027630584} {:x 3.2699999999999743, :y 1.5288173254210238} {:x 3.279999999999974, :y 2.518310520678372} {:x 3.289999999999974, :y 3.080051258315208} {:x 3.2999999999999736, :y 2.870244500868692} {:x 3.3099999999999734, :y 0.5191944470928433} {:x 3.319999999999973, :y 0.420582224462231} {:x 3.329999999999973, :y 2.335574221998874} {:x 3.3399999999999728, :y 3.3018206030356785} {:x 3.3499999999999726, :y 0.6301949413032795} {:x 3.3599999999999723, :y 2.9854624542676005} {:x 3.369999999999972, :y 2.8220722020369458} {:x 3.379999999999972, :y 1.7421521471667307} {:x 3.3899999999999717, :y 2.3104138252959525} {:x 3.3999999999999715, :y 1.7772444631896507} {:x 3.4099999999999713, :y 1.8678816483109897} {:x 3.419999999999971, :y 1.998777210549902} {:x 3.429999999999971, :y 1.2413721298758178} {:x 3.4399999999999706, :y 2.122095341674623} {:x 3.4499999999999704, :y 1.9907849731722855} {:x 3.45999999999997, :y 2.4442037861295978} {:x 3.46999999999997, :y 1.6269417453033377} {:x 3.47999999999997, :y 2.9119853312768993} {:x 3.4899999999999696, :y 3.141873149397687} {:x 3.4999999999999694, :y 3.2407254514599617} {:x 3.509999999999969, :y 2.894889838620878} {:x 3.519999999999969, :y 0.9140433148332595} {:x 3.5299999999999687, :y 0.17130189037557036} {:x 3.5399999999999685, :y 3.4052824080076425} {:x 3.5499999999999683, :y 1.8320774504455828} {:x 3.559999999999968, :y 0.8802010568545677} {:x 3.569999999999968, :y 3.3326669550691603} {:x 3.5799999999999677, :y 1.631302435687567} {:x 3.5899999999999674, :y 3.0153176404893847} {:x 3.5999999999999672, :y 0.48231001044844274} {:x 3.609999999999967, :y 0.4324327428216076} {:x 3.619999999999967, :y 0.2699493899834946} {:x 3.6299999999999666, :y 1.4344680816758295} {:x 3.6399999999999664, :y 1.451714738290243} {:x 3.649999999999966, :y 0.4109052082266942} {:x 3.659999999999966, :y 2.303438228002478} {:x 3.6699999999999657, :y 1.6755049088321243} {:x 3.6799999999999655, :y 2.432041667557286} {:x 3.6899999999999653, :y 1.3941522200980498} {:x 3.699999999999965, :y 2.0305191196688055} {:x 3.709999999999965, :y 0.22159790166940735} {:x 3.7199999999999647, :y 2.952213648382213} {:x 3.7299999999999645, :y 1.8345340572366085} {:x 3.7399999999999642, :y 0.10141873088611186} {:x 3.749999999999964, :y 2.0520829310452933} {:x 3.759999999999964, :y 2.585930124011471} {:x 3.7699999999999636, :y 0.113622643159977} {:x 3.7799999999999634, :y 3.4389236550057847} {:x 3.789999999999963, :y 2.5652021222917663} {:x 3.799999999999963, :y 2.6965564188319493} {:x 3.8099999999999627, :y 2.0877499448638663} {:x 3.8199999999999625, :y 3.4964280359186652} {:x 3.8299999999999623, :y 2.9830639222745123} {:x 3.839999999999962, :y 2.685049418214062} {:x 3.849999999999962, :y 1.075744779963071} {:x 3.8599999999999617, :y 0.40512075033509715} {:x 3.8699999999999615, :y 3.3389501606771472} {:x 3.8799999999999613, :y 1.6757022273913362} {:x 3.889999999999961, :y 3.843084587106392} {:x 3.899999999999961, :y 2.3635730524525798} {:x 3.9099999999999606, :y 0.08038347764319287} {:x 3.9199999999999604, :y 3.2426285746127648} {:x 3.92999999999996, :y 2.8992071041535654} {:x 3.93999999999996, :y 1.8562986132243475} {:x 3.9499999999999598, :y 3.7587467176899008} {:x 3.9599999999999596, :y 3.2915154530103794} {:x 3.9699999999999593, :y 0.8875997560956681} {:x 3.979999999999959, :y 0.8458936915333946} {:x 3.989999999999959, :y 3.760233888388318} {:x 3.9999999999999587, :y 3.278763638011273} {:x 4.009999999999959, :y 2.794006148406674} {:x 4.019999999999959, :y 3.978629765090821} {:x 4.0299999999999585, :y 0.7059960856035993} {:x 4.039999999999958, :y 2.326198048711626} {:x 4.049999999999958, :y 1.8391632569736032} {:x 4.059999999999958, :y 2.628361230770177} {:x 4.069999999999958, :y 3.2940339473762354} {:x 4.079999999999957, :y 1.5465714524244691} {:x 4.089999999999957, :y 1.3114093890395742} {:x 4.099999999999957, :y 3.29082684621221} {:x 4.109999999999957, :y 3.6341772744050846} {:x 4.119999999999957, :y 2.105603379819089} {:x 4.129999999999956, :y 2.2637791238363034} {:x 4.139999999999956, :y 3.9823676945841555} {:x 4.149999999999956, :y 3.0028829340249708} {:x 4.159999999999956, :y 2.2027889279146824} {:x 4.1699999999999555, :y 0.3223322550616315} {:x 4.179999999999955, :y 0.8862540216706211} {:x 4.189999999999955, :y 0.7890753422502149} {:x 4.199999999999955, :y 4.168183988045357} {:x 4.209999999999955, :y 0.699941843194316} {:x 4.2199999999999545, :y 1.5787095901217727} {:x 4.229999999999954, :y 3.468344762032132} {:x 4.239999999999954, :y 3.918320108223203} {:x 4.249999999999954, :y 4.14510085623999} {:x 4.259999999999954, :y 2.7078460637241744} {:x 4.269999999999953, :y 2.753018005868473} {:x 4.279999999999953, :y 0.8838260602657648} {:x 4.289999999999953, :y 3.500988088697334} {:x 4.299999999999953, :y 2.5045683956888003} {:x 4.3099999999999525, :y 0.7224095229168292} {:x 4.319999999999952, :y 1.0867340376504329} {:x 4.329999999999952, :y 1.8285722224228904} {:x 4.339999999999952, :y 1.0475889396321436} {:x 4.349999999999952, :y 3.803460971842525} {:x 4.3599999999999515, :y 3.3393240910350412} {:x 4.369999999999951, :y 3.4114160449870727} {:x 4.379999999999951, :y 0.9374914357686269} {:x 4.389999999999951, :y 1.67947793059512} {:x 4.399999999999951, :y 0.671216455924884} {:x 4.40999999999995, :y 3.4015841446087944} {:x 4.41999999999995, :y 1.0298137177943432} {:x 4.42999999999995, :y 1.036271408031536} {:x 4.43999999999995, :y 0.49974807399466026} {:x 4.4499999999999496, :y 0.4981454789836499} {:x 4.459999999999949, :y 2.2468752404885675} {:x 4.469999999999949, :y 0.9132180082150437} {:x 4.479999999999949, :y 1.3729787833139357} {:x 4.489999999999949, :y 3.914705044909433} {:x 4.4999999999999485, :y 0.04505766317827979} {:x 4.509999999999948, :y 4.164055254185302} {:x 4.519999999999948, :y 2.3274564248991942} {:x 4.529999999999948, :y 0.5926889912663521} {:x 4.539999999999948, :y 2.5109547976217885} {:x 4.549999999999947, :y 2.461136518246732} {:x 4.559999999999947, :y 2.726625640823365} {:x 4.569999999999947, :y 0.8752571298239804} {:x 4.579999999999947, :y 3.0887452589901} {:x 4.589999999999947, :y 0.7198743390169127} {:x 4.599999999999946, :y 3.1930078312820833} {:x 4.609999999999946, :y 0.575206768558524} {:x 4.619999999999946, :y 3.534801922336055} {:x 4.629999999999946, :y 3.4097163474477106} {:x 4.6399999999999455, :y 2.4304863466871356} {:x 4.649999999999945, :y 1.5115839786152543} {:x 4.659999999999945, :y 2.826441736835714} {:x 4.669999999999945, :y 1.296701276398053} {:x 4.679999999999945, :y 1.7047246501204583} {:x 4.689999999999944, :y 0.8664167854243952} {:x 4.699999999999944, :y 1.432334766298035} {:x 4.709999999999944, :y 0.30296318261119637} {:x 4.719999999999944, :y 1.1824106583414598} {:x 4.729999999999944, :y 4.527532945431609} {:x 4.739999999999943, :y 0.3949322304451271} {:x 4.749999999999943, :y 4.482957913336911} {:x 4.759999999999943, :y 1.056160020114835} {:x 4.769999999999943, :y 1.7783312471800183} {:x 4.7799999999999425, :y 3.2528425126221214} {:x 4.789999999999942, :y 3.3572095494891085} {:x 4.799999999999942, :y 4.71403431321551} {:x 4.809999999999942, :y 2.863126033190511} {:x 4.819999999999942, :y 0.7758796429633972} {:x 4.8299999999999415, :y 3.694484138920159} {:x 4.839999999999941, :y 0.8232235842344509} {:x 4.849999999999941, :y 3.2097917655717154} {:x 4.859999999999941, :y 3.8700273074043863} {:x 4.869999999999941, :y 3.3250231333361073} {:x 4.87999999999994, :y 2.679014712113577} {:x 4.88999999999994, :y 3.12239035487871} {:x 4.89999999999994, :y 1.4463140656457991} {:x 4.90999999999994, :y 0.09328448895304228} {:x 4.9199999999999395, :y 3.995217196773686} {:x 4.929999999999939, :y 1.6677744968708872} {:x 4.939999999999939, :y 0.34906618172911297} {:x 4.949999999999939, :y 2.9810650181681275} {:x 4.959999999999939, :y 1.8732419946108254} {:x 4.9699999999999385, :y 2.9843180431723666} {:x 4.979999999999938, :y 2.0977065160199446} {:x 4.989999999999938, :y 1.6333977208233188} {:x 4.999999999999938, :y 2.496533127949724} {:x 5.009999999999938, :y 4.060437173952934} {:x 5.019999999999937, :y 2.364697813549784} {:x 5.029999999999937, :y 0.22274553282737145} {:x 5.039999999999937, :y 1.2675264374187871} {:x 5.049999999999937, :y 2.7779306676234135} {:x 5.0599999999999365, :y 2.362879987356525} {:x 5.069999999999936, :y 2.1142378250791696} {:x 5.079999999999936, :y 2.3882698248220433} {:x 5.089999999999936, :y 0.06287971396200587} {:x 5.099999999999936, :y 2.860833929848697} {:x 5.1099999999999355, :y 0.09852417794585762} {:x 5.119999999999935, :y 2.6757668232101546} {:x 5.129999999999935, :y 0.4574168084039595} {:x 5.139999999999935, :y 2.6775110599366863} {:x 5.149999999999935, :y 1.8017515147314147} {:x 5.159999999999934, :y 4.451515742201196} {:x 5.169999999999934, :y 4.37158956184408} {:x 5.179999999999934, :y 4.699701239098009} {:x 5.189999999999934, :y 4.64669916748889} {:x 5.199999999999934, :y 3.564101420824132} {:x 5.209999999999933, :y 2.248315319512665} {:x 5.219999999999933, :y 0.6704742957908782} {:x 5.229999999999933, :y 4.427740465353794} {:x 5.239999999999933, :y 3.346555908713266} {:x 5.2499999999999325, :y 1.8385440573816902} {:x 5.259999999999932, :y 2.3460146286554737} {:x 5.269999999999932, :y 3.07753568530085} {:x 5.279999999999932, :y 2.4890455067772357} {:x 5.289999999999932, :y 2.120717901462258} {:x 5.299999999999931, :y 2.3247836212949653} {:x 5.309999999999931, :y 3.1076930854768805} {:x 5.319999999999931, :y 3.7347531278669512} {:x 5.329999999999931, :y 2.9109422129731803} {:x 5.339999999999931, :y 3.4891863937937595} {:x 5.34999999999993, :y 5.261576623651832} {:x 5.35999999999993, :y 4.695117349812701} {:x 5.36999999999993, :y 2.2204587002963083} {:x 5.37999999999993, :y 0.9383817950808354} {:x 5.3899999999999295, :y 3.247474472711777} {:x 5.399999999999929, :y 4.0423573716132415} {:x 5.409999999999929, :y 0.7375346833981192} {:x 5.419999999999929, :y 3.3640728835066596} {:x 5.429999999999929, :y 5.066985048899524} {:x 5.4399999999999284, :y 4.224956176024419} {:x 5.449999999999928, :y 4.1542672721186005} {:x 5.459999999999928, :y 3.5415322674959544} {:x 5.469999999999928, :y 0.9692198934468507} {:x 5.479999999999928, :y 3.457995880544312} {:x 5.489999999999927, :y 2.8175274923357767} {:x 5.499999999999927, :y 2.0882622332890577} {:x 5.509999999999927, :y 1.0087633831589842} {:x 5.519999999999927, :y 0.29743572853804856} {:x 5.5299999999999265, :y 4.249852538192594} {:x 5.539999999999926, :y 1.6550666118796435} {:x 5.549999999999926, :y 4.576779208780081} {:x 5.559999999999926, :y 1.5280974481276017} {:x 5.569999999999926, :y 4.233028690119004} {:x 5.5799999999999255, :y 0.4314061387533993} {:x 5.589999999999925, :y 0.7426962669185353} {:x 5.599999999999925, :y 2.0917742142356484} {:x 5.609999999999925, :y 3.4668918165886806} {:x 5.619999999999925, :y 2.6012388267611946} {:x 5.629999999999924, :y 1.5810578518573493} {:x 5.639999999999924, :y 2.4270026036004513} {:x 5.649999999999924, :y 3.321009521132676} {:x 5.659999999999924, :y 4.789598393386025} {:x 5.6699999999999235, :y 1.9468651423598986} {:x 5.679999999999923, :y 0.23814636099039405} {:x 5.689999999999923, :y 5.389896111152566} {:x 5.699999999999923, :y 2.106441373600592} {:x 5.709999999999923, :y 3.455602579917994} {:x 5.7199999999999225, :y 4.551014611341899} {:x 5.729999999999922, :y 2.295753831517114} {:x 5.739999999999922, :y 2.6953035090117154} {:x 5.749999999999922, :y 0.4335788338516777} {:x 5.759999999999922, :y 3.417798597315438} {:x 5.769999999999921, :y 2.868922537638807} {:x 5.779999999999921, :y 3.2625339311118013} {:x 5.789999999999921, :y 4.522777925550773} {:x 5.799999999999921, :y 1.7181419640024531} {:x 5.809999999999921, :y 5.397764771777134} {:x 5.81999999999992, :y 1.0908374159118608} {:x 5.82999999999992, :y 2.09456909595203} {:x 5.83999999999992, :y 0.11816963818550123} {:x 5.84999999999992, :y 2.339106877229818} {:x 5.8599999999999195, :y 1.441039834128449} {:x 5.869999999999919, :y 2.600879375967315} {:x 5.879999999999919, :y 3.267787100422037} {:x 5.889999999999919, :y 0.7301028328145656} {:x 5.899999999999919, :y 1.4309040897844734} {:x 5.909999999999918, :y 2.062296864959538} {:x 5.919999999999918, :y 4.771023305230153} {:x 5.929999999999918, :y 5.424986680564862} {:x 5.939999999999918, :y 1.3819616148798817} {:x 5.949999999999918, :y 3.622530590155992} {:x 5.959999999999917, :y 5.027133556638011} {:x 5.969999999999917, :y 1.5081250986375792} {:x 5.979999999999917, :y 0.0771913684196489} {:x 5.989999999999917, :y 5.780214948857677} {:x 5.9999999999999165, :y 1.0240902259371192} {:x 6.009999999999916, :y 4.100587373971292} {:x 6.019999999999916, :y 3.226469066034051} {:x 6.029999999999916, :y 3.294712599759812} {:x 6.039999999999916, :y 0.2034840557474503} {:x 6.0499999999999154, :y 2.343598031990255} {:x 6.059999999999915, :y 5.025527198989543} {:x 6.069999999999915, :y 0.48483552577090355} {:x 6.079999999999915, :y 1.1592283324271944} {:x 6.089999999999915, :y 4.9212430229002555} {:x 6.099999999999914, :y 4.630212779610673} {:x 6.109999999999914, :y 0.9407682708381743} {:x 6.119999999999914, :y 2.2143032062073904} {:x 6.129999999999914, :y 4.469156846376697} {:x 6.1399999999999135, :y 5.063600597397378} {:x 6.149999999999913, :y 4.612448740547254} {:x 6.159999999999913, :y 4.3125990149184465} {:x 6.169999999999913, :y 2.1481227820941378} {:x 6.179999999999913, :y 4.921516049372504} {:x 6.1899999999999125, :y 2.984732797914748} {:x 6.199999999999912, :y 0.0792177516141292} {:x 6.209999999999912, :y 2.8340797808039144} {:x 6.219999999999912, :y 2.5747507563435246} {:x 6.229999999999912, :y 1.5433374570587732} {:x 6.239999999999911, :y 0.8548659286970697} {:x 6.249999999999911, :y 0.05143559928886956} {:x 6.259999999999911, :y 3.4761986027283087} {:x 6.269999999999911, :y 1.73833399459127} {:x 6.2799999999999105, :y 5.190725933102908} {:x 6.28999999999991, :y 0.26735322497152425} {:x 6.29999999999991, :y 1.5808256521228565} {:x 6.30999999999991, :y 5.6663371204586035} {:x 6.31999999999991, :y 4.691917516084117} {:x 6.3299999999999095, :y 0.7932843489038839} {:x 6.339999999999909, :y 4.268283722881138} {:x 6.349999999999909, :y 4.281009731566307} {:x 6.359999999999909, :y 4.172871671650441} {:x 6.369999999999909, :y 1.468995197926315} {:x 6.379999999999908, :y 4.854613922907892} {:x 6.389999999999908, :y 6.216586141187558} {:x 6.399999999999908, :y 4.949116319115817} {:x 6.409999999999908, :y 1.586140636618281} {:x 6.419999999999908, :y 2.903342594643824} {:x 6.429999999999907, :y 2.6932905485192618} {:x 6.439999999999907, :y 1.049632969337272} {:x 6.449999999999907, :y 0.705613013469506} {:x 6.459999999999907, :y 3.8498527521779886} {:x 6.4699999999999065, :y 0.9593138354018698} {:x 6.479999999999906, :y 4.144578973931108} {:x 6.489999999999906, :y 5.697913527761578} {:x 6.499999999999906, :y 3.1487302169251183} {:x 6.509999999999906, :y 1.6138917466888532} {:x 6.519999999999905, :y 0.5572803874281668} {:x 6.529999999999905, :y 6.401161787010368} {:x 6.539999999999905, :y 3.994315859588411} {:x 6.549999999999905, :y 1.8125119447096654} {:x 6.559999999999905, :y 0.8194885536052532} {:x 6.569999999999904, :y 1.3195818184274812} {:x 6.579999999999904, :y 4.542462387341144} {:x 6.589999999999904, :y 2.0108643535526287} {:x 6.599999999999904, :y 1.7474575583369507} {:x 6.6099999999999035, :y 6.328881574274118} {:x 6.619999999999903, :y 1.3689819948738091} {:x 6.629999999999903, :y 3.1813372984351806} {:x 6.639999999999903, :y 0.6759428006195939} {:x 6.649999999999903, :y 5.676362751315365} {:x 6.659999999999902, :y 4.949989791374742} {:x 6.669999999999902, :y 4.82714838890809} {:x 6.679999999999902, :y 6.440235111607413} {:x 6.689999999999902, :y 3.2953791122951746} {:x 6.699999999999902, :y 0.9537893930104108} {:x 6.709999999999901, :y 5.4670924966219125} {:x 6.719999999999901, :y 2.695341678884974} {:x 6.729999999999901, :y 0.42838922678124014} {:x 6.739999999999901, :y 0.6856203459544725} {:x 6.7499999999999005, :y 5.080879509082309} {:x 6.7599999999999, :y 1.7988945078602876} {:x 6.7699999999999, :y 1.7928292694049148} {:x 6.7799999999999, :y 1.0579777859983481} {:x 6.7899999999999, :y 0.12363001558151863} {:x 6.7999999999998995, :y 5.072815721282966} {:x 6.809999999999899, :y 4.871865690148699} {:x 6.819999999999899, :y 0.8428446558882287} {:x 6.829999999999899, :y 0.39079028419372097} {:x 6.839999999999899, :y 3.9906903931695594} {:x 6.849999999999898, :y 0.8802671896318731} {:x 6.859999999999898, :y 0.010521195432309088} {:x 6.869999999999898, :y 0.5640268813775505} {:x 6.879999999999898, :y 6.265700121442904} {:x 6.8899999999998975, :y 1.1266763498732513} {:x 6.899999999999897, :y 6.785314512544095} {:x 6.909999999999897, :y 0.487057877798723} {:x 6.919999999999897, :y 3.9303479401781978} {:x 6.929999999999897, :y 0.409941656859702} {:x 6.9399999999998965, :y 5.143654268801617} {:x 6.949999999999896, :y 4.620378374643769} {:x 6.959999999999896, :y 3.5765554872984433} {:x 6.969999999999896, :y 5.991414923984163} {:x 6.979999999999896, :y 2.5577298224277376} {:x 6.989999999999895, :y 3.687941460622465} {:x 6.999999999999895, :y 6.675547005650224} {:x 7.009999999999895, :y 2.029703046410533} {:x 7.019999999999895, :y 2.779996083969424} {:x 7.0299999999998946, :y 4.15955418474919} {:x 7.039999999999894, :y 0.7537396783105762} {:x 7.049999999999894, :y 0.7618104701724906} {:x 7.059999999999894, :y 1.1433868829538734} {:x 7.069999999999894, :y 3.9525578709075173} {:x 7.0799999999998935, :y 0.8361436887594367} {:x 7.089999999999893, :y 0.5473965664250178} {:x 7.099999999999893, :y 2.131271867565085} {:x 7.109999999999893, :y 6.747766400431408} {:x 7.119999999999893, :y 3.3350325589069625} {:x 7.129999999999892, :y 3.6935189471135974} {:x 7.139999999999892, :y 4.482638920852901} {:x 7.149999999999892, :y 6.96014071915204} {:x 7.159999999999892, :y 2.8513868356742766} {:x 7.169999999999892, :y 5.698232064880834} {:x 7.179999999999891, :y 5.434138968336816} {:x 7.189999999999891, :y 3.7767949362940354} {:x 7.199999999999891, :y 1.8823178922117751} {:x 7.209999999999891, :y 6.975251814684659} {:x 7.2199999999998905, :y 5.414023845635103} {:x 7.22999999999989, :y 4.396130717529783} {:x 7.23999999999989, :y 3.221878428235525} {:x 7.24999999999989, :y 2.0611037230542704} {:x 7.25999999999989, :y 6.126117819610812} {:x 7.269999999999889, :y 4.384717326725126} {:x 7.279999999999889, :y 3.4178233811648733} {:x 7.289999999999889, :y 1.0632163466789837} {:x 7.299999999999889, :y 5.607311558028261} {:x 7.309999999999889, :y 6.405444055243212} {:x 7.319999999999888, :y 1.1155326679971005} {:x 7.329999999999888, :y 4.894829418764613} {:x 7.339999999999888, :y 3.422129977545889} {:x 7.349999999999888, :y 2.623988883600407} {:x 7.3599999999998875, :y 1.954924542686524} {:x 7.369999999999887, :y 7.3128351585141695} {:x 7.379999999999887, :y 0.6852227938516495} {:x 7.389999999999887, :y 6.583264467300067} {:x 7.399999999999887, :y 6.9992854612420485} {:x 7.4099999999998865, :y 0.40028193699066383} {:x 7.419999999999886, :y 0.02964948089884218} {:x 7.429999999999886, :y 2.669720842232049} {:x 7.439999999999886, :y 1.6990116130592254} {:x 7.449999999999886, :y 4.428720207368497} {:x 7.459999999999885, :y 0.6180742073845705} {:x 7.469999999999885, :y 5.913706026580454} {:x 7.479999999999885, :y 3.480809607761758} {:x 7.489999999999885, :y 6.24938584464422} {:x 7.4999999999998845, :y 1.1379669480304875} {:x 7.509999999999884, :y 7.131003337865179} {:x 7.519999999999884, :y 0.4764270072722739} {:x 7.529999999999884, :y 2.7588709250746866} {:x 7.539999999999884, :y 3.0831008762141696} {:x 7.5499999999998835, :y 1.1866948760915388} {:x 7.559999999999883, :y 7.016985712567578} {:x 7.569999999999883, :y 3.846947178246566} {:x 7.579999999999883, :y 6.0545713022771785} {:x 7.589999999999883, :y 3.5681179456138943} {:x 7.599999999999882, :y 1.831971435638476} {:x 7.609999999999882, :y 4.334068259104327} {:x 7.619999999999882, :y 7.427710121816816} {:x 7.629999999999882, :y 4.793413275725009} {:x 7.6399999999998816, :y 5.398383506790628} {:x 7.649999999999881, :y 1.7562929864997467} {:x 7.659999999999881, :y 0.35915216619986284} {:x 7.669999999999881, :y 4.778489177960532} {:x 7.679999999999881, :y 1.6179860002309654} {:x 7.6899999999998805, :y 3.8084382708654285} {:x 7.69999999999988, :y 4.763419966397642} {:x 7.70999999999988, :y 0.9506565486559073} {:x 7.71999999999988, :y 6.991505802681903} {:x 7.72999999999988, :y 3.9946881281789457} {:x 7.739999999999879, :y 5.523223693502714} {:x 7.749999999999879, :y 3.4942200942664874} {:x 7.759999999999879, :y 0.7805090999128286} {:x 7.769999999999879, :y 6.792153926074055} {:x 7.779999999999879, :y 0.5247338677089654} {:x 7.789999999999878, :y 2.134446100828869} {:x 7.799999999999878, :y 7.61304913514713} {:x 7.809999999999878, :y 6.36950336669301} {:x 7.819999999999878, :y 7.1808762806988815} {:x 7.8299999999998775, :y 5.454057655065537} {:x 7.839999999999877, :y 0.5105155297627533} {:x 7.849999999999877, :y 2.5822487450522598} {:x 7.859999999999877, :y 6.814705050211519} {:x 7.869999999999877, :y 7.553248100890202} {:x 7.879999999999876, :y 5.102931860153244} {:x 7.889999999999876, :y 6.771105273545683} {:x 7.899999999999876, :y 4.733655863891144} {:x 7.909999999999876, :y 4.312637996087194} {:x 7.919999999999876, :y 5.450735653050234} {:x 7.929999999999875, :y 7.432513818568855} {:x 7.939999999999875, :y 2.066445236252012} {:x 7.949999999999875, :y 0.5611422342359056} {:x 7.959999999999875, :y 5.715345548566473} {:x 7.9699999999998745, :y 0.07344028376157195} {:x 7.979999999999874, :y 0.47007549506234736} {:x 7.989999999999874, :y 0.019853544541840393} {:x 7.999999999999874, :y 0.8310629793517628} {:x 8.009999999999874, :y 1.7684501853157404} {:x 8.019999999999873, :y 1.8973269077208752} {:x 8.029999999999873, :y 7.041288852826024} {:x 8.039999999999873, :y 3.8795789566137056} {:x 8.049999999999873, :y 0.3182846125188164} {:x 8.059999999999873, :y 7.137761189009596} {:x 8.069999999999872, :y 3.333193637365638} {:x 8.079999999999872, :y 6.00804800378902} {:x 8.089999999999872, :y 1.0282454866635278} {:x 8.099999999999872, :y 1.8201369991482945} {:x 8.109999999999872, :y 5.15003950441022} {:x 8.119999999999871, :y 6.814872175599068} {:x 8.129999999999871, :y 0.16413128901373933} {:x 8.139999999999871, :y 6.903736653582432} {:x 8.14999999999987, :y 7.115900418076526} {:x 8.15999999999987, :y 7.106484651454529} {:x 8.16999999999987, :y 6.705751901431591} {:x 8.17999999999987, :y 1.2429291451423925} {:x 8.18999999999987, :y 2.5220398668902653} {:x 8.19999999999987, :y 2.9160484656267647} {:x 8.20999999999987, :y 2.2756858041584356} {:x 8.21999999999987, :y 5.035846659407277} {:x 8.229999999999869, :y 4.453791476880923} {:x 8.239999999999869, :y 2.583591807994696} {:x 8.249999999999869, :y 2.886431144391351} {:x 8.259999999999868, :y 6.956832982375714} {:x 8.269999999999868, :y 4.376669403893347} {:x 8.279999999999868, :y 6.879181605215516} {:x 8.289999999999868, :y 4.964942998149749} {:x 8.299999999999867, :y 3.1358258795744023} {:x 8.309999999999867, :y 0.8335919518402937} {:x 8.319999999999867, :y 0.4387904237372596} {:x 8.329999999999867, :y 4.20224192295152} {:x 8.339999999999867, :y 8.138569792303375} {:x 8.349999999999866, :y 1.5861657556449815} {:x 8.359999999999866, :y 1.1690487921465882} {:x 8.369999999999866, :y 1.14981592455259} {:x 8.379999999999866, :y 6.5120130048675025} {:x 8.389999999999866, :y 7.355624398331649} {:x 8.399999999999865, :y 4.874582777382538} {:x 8.409999999999865, :y 3.6464419438576896} {:x 8.419999999999865, :y 0.7019289042077081} {:x 8.429999999999865, :y 2.3357803853904984} {:x 8.439999999999864, :y 3.3226227024284762} {:x 8.449999999999864, :y 5.871289020698836} {:x 8.459999999999864, :y 5.457214639162601} {:x 8.469999999999864, :y 3.1137046459701927} {:x 8.479999999999864, :y 1.5565337286820549} {:x 8.489999999999863, :y 7.104271938573498} {:x 8.499999999999863, :y 2.5199568960294445} {:x 8.509999999999863, :y 7.166311639070812} {:x 8.519999999999863, :y 5.975355893795453} {:x 8.529999999999863, :y 2.8750660905265035} {:x 8.539999999999862, :y 4.234721473440319} {:x 8.549999999999862, :y 1.5650867865264249} {:x 8.559999999999862, :y 4.862858935084451} {:x 8.569999999999862, :y 6.830499227519712} {:x 8.579999999999862, :y 7.107114022640628} {:x 8.589999999999861, :y 1.9693623919165852} {:x 8.599999999999861, :y 0.5895105357431474} {:x 8.60999999999986, :y 4.518687654102903} {:x 8.61999999999986, :y 6.9929793630337365} {:x 8.62999999999986, :y 5.653307526050116} {:x 8.63999999999986, :y 4.136621722706541} {:x 8.64999999999986, :y 4.939787096617512} {:x 8.65999999999986, :y 3.518148705822111} {:x 8.66999999999986, :y 8.165669655748026} {:x 8.67999999999986, :y 1.8962858369005757} {:x 8.68999999999986, :y 1.7800846990308237} {:x 8.699999999999859, :y 4.226919368719593} {:x 8.709999999999859, :y 5.386773691294555} {:x 8.719999999999859, :y 1.996041797946463} {:x 8.729999999999858, :y 7.89086563368231} {:x 8.739999999999858, :y 1.127945611953833} {:x 8.749999999999858, :y 6.786850213905449} {:x 8.759999999999858, :y 3.2359970974484873} {:x 8.769999999999857, :y 4.8385454817435045} {:x 8.779999999999857, :y 0.2414483231496114} {:x 8.789999999999857, :y 1.025942518365114} {:x 8.799999999999857, :y 2.6209198266730285} {:x 8.809999999999857, :y 6.02047291863414} {:x 8.819999999999856, :y 4.164097289129989} {:x 8.829999999999856, :y 0.8709121095803296} {:x 8.839999999999856, :y 2.2530579191150646} {:x 8.849999999999856, :y 2.674063203797171} {:x 8.859999999999856, :y 3.190103129916653} {:x 8.869999999999855, :y 0.33193773256997716} {:x 8.879999999999855, :y 3.6156785812752212} {:x 8.889999999999855, :y 0.018041510839812234} {:x 8.899999999999855, :y 3.671454584454948} {:x 8.909999999999854, :y 3.611877967185461} {:x 8.919999999999854, :y 8.215019294003536} {:x 8.929999999999854, :y 7.796038652042019} {:x 8.939999999999854, :y 2.91937285798715} {:x 8.949999999999854, :y 8.329452191209638} {:x 8.959999999999853, :y 6.592026564892616} {:x 8.969999999999853, :y 8.447096495562045} {:x 8.979999999999853, :y 8.055408189891452} {:x 8.989999999999853, :y 4.723863691846962} {:x 8.999999999999853, :y 0.7774193874198837} {:x 9.009999999999852, :y 6.347308480011832} {:x 9.019999999999852, :y 2.166690256512381} {:x 9.029999999999852, :y 4.646780257873708} {:x 9.039999999999852, :y 1.656743270022525} {:x 9.049999999999851, :y 5.8643440703022325} {:x 9.059999999999851, :y 1.1381533836636275} {:x 9.069999999999851, :y 1.0399640090871476} {:x 9.07999999999985, :y 0.9086956263192252} {:x 9.08999999999985, :y 4.724470579224962} {:x 9.09999999999985, :y 0.7375957441669981} {:x 9.10999999999985, :y 5.66582868978124} {:x 9.11999999999985, :y 7.69259314199651} {:x 9.12999999999985, :y 4.662415252156544} {:x 9.13999999999985, :y 3.409692185605036} {:x 9.14999999999985, :y 2.816288645040154} {:x 9.15999999999985, :y 7.161911051634419} {:x 9.169999999999849, :y 4.060478554221811} {:x 9.179999999999849, :y 1.705077793029636} {:x 9.189999999999849, :y 5.434951858547066} {:x 9.199999999999848, :y 0.8828682491964102} {:x 9.209999999999848, :y 4.757241020859892} {:x 9.219999999999848, :y 2.962824799368668} {:x 9.229999999999848, :y 6.111495520971406} {:x 9.239999999999847, :y 6.292551955598708} {:x 9.249999999999847, :y 1.5304777880479654} {:x 9.259999999999847, :y 4.891591800251573} {:x 9.269999999999847, :y 3.317458339790448} {:x 9.279999999999847, :y 9.072563355758858} {:x 9.289999999999846, :y 0.8460293534206856} {:x 9.299999999999846, :y 1.1688836973342664} {:x 9.309999999999846, :y 7.172169849100504} {:x 9.319999999999846, :y 5.014144427649479} {:x 9.329999999999846, :y 5.646853312094293} {:x 9.339999999999845, :y 8.771760767533808} {:x 9.349999999999845, :y 6.968635828031535} {:x 9.359999999999845, :y 6.96168606840819} {:x 9.369999999999845, :y 9.326814969319742} {:x 9.379999999999844, :y 6.438463770178991} {:x 9.389999999999844, :y 1.6143930300556601} {:x 9.399999999999844, :y 9.255946616421479} {:x 9.409999999999844, :y 7.01446158851624} {:x 9.419999999999844, :y 8.111259346333918} {:x 9.429999999999843, :y 6.754139419563133} {:x 9.439999999999843, :y 7.348707279118209} {:x 9.449999999999843, :y 0.8955534256314677} {:x 9.459999999999843, :y 8.038900481265493} {:x 9.469999999999843, :y 4.181851160447709} {:x 9.479999999999842, :y 7.279468881409791} {:x 9.489999999999842, :y 5.682019229351771} {:x 9.499999999999842, :y 4.634886994955726} {:x 9.509999999999842, :y 5.532592479948148} {:x 9.519999999999841, :y 2.5246503666329465} {:x 9.529999999999841, :y 0.6217342189788361} {:x 9.539999999999841, :y 4.867749638469798} {:x 9.54999999999984, :y 5.580826665995294} {:x 9.55999999999984, :y 0.3030724200526778} {:x 9.56999999999984, :y 4.650618096084057} {:x 9.57999999999984, :y 3.6943310847486748} {:x 9.58999999999984, :y 4.085244522668825} {:x 9.59999999999984, :y 2.660826502933344} {:x 9.60999999999984, :y 3.4084796271788464} {:x 9.61999999999984, :y 8.049021763035412} {:x 9.62999999999984, :y 1.1124218455738624} {:x 9.639999999999839, :y 4.276402722474884} {:x 9.649999999999839, :y 0.6853404193365019} {:x 9.659999999999838, :y 9.103621545859701} {:x 9.669999999999838, :y 2.4055347286217685} {:x 9.679999999999838, :y 1.9157168278948655} {:x 9.689999999999838, :y 9.3481048175681} {:x 9.699999999999838, :y 0.9264227728734049} {:x 9.709999999999837, :y 4.412360544541769} {:x 9.719999999999837, :y 1.938908972880634} {:x 9.729999999999837, :y 8.210944124521943} {:x 9.739999999999837, :y 2.3877854138821513} {:x 9.749999999999837, :y 5.661405837265113} {:x 9.759999999999836, :y 6.169003304604087} {:x 9.769999999999836, :y 8.499110536820224} {:x 9.779999999999836, :y 7.755356005149888} {:x 9.789999999999836, :y 5.008714179272278} {:x 9.799999999999836, :y 1.8892014497304808} {:x 9.809999999999835, :y 9.737591846524689} {:x 9.819999999999835, :y 2.267696723125725} {:x 9.829999999999835, :y 5.643215974574373} {:x 9.839999999999835, :y 2.13325516448791} {:x 9.849999999999834, :y 1.3855769162824358} {:x 9.859999999999834, :y 1.0829376951594958} {:x 9.869999999999834, :y 5.897899517671505} {:x 9.879999999999834, :y 8.724502490573142} {:x 9.889999999999834, :y 8.467592262957647} {:x 9.899999999999833, :y 2.0816308894956026} {:x 9.909999999999833, :y 8.170096376501508} {:x 9.919999999999833, :y 9.856269431242795} {:x 9.929999999999833, :y 0.8549354864604157} {:x 9.939999999999833, :y 8.452776971645347} {:x 9.949999999999832, :y 0.3192934817032741} {:x 9.959999999999832, :y 6.461941011123203} {:x 9.969999999999832, :y 3.979129787901795} {:x 9.979999999999832, :y 1.1015236616972586} {:x 9.989999999999831, :y 4.247598769573557} {:x 9.999999999999831, :y 2.5095700122491915})} {:name \"af9febcd-43c7-4018-acdf-ef02a96c8669\", :values ({:x 0, :y 0.0} {:x 0.10000000149011612, :y 9.966711523935131E-4} {:x 0.20000000298023224, :y 0.0078939009494512} {:x 0.30000000447034836, :y 0.026199658911198156} {:x 0.4000000059604645, :y 0.06065866074476158} {:x 0.5000000074505806, :y 0.11492442838019619} {:x 0.6000000089406967, :y 0.19129268150732823} {:x 0.7000000104308128, :y 0.2905115115092036} {:x 0.800000011920929, :y 0.41167982458769947} {:x 0.9000000134110451, :y 0.5522409625952036} {:x 1.0000000149011612, :y 0.7080734423742749} {:x 1.1000000163912773, :y 0.8736756420867389} {:x 1.2000000178813934, :y 1.042436259352125} {:x 1.3000000193715096, :y 1.2069777206570362} {:x 1.4000000208616257, :y 1.3595556685107744} {:x 1.5000000223517418, :y 1.4924943994216495} {:x 1.600000023841858, :y 1.5986358422305347} {:x 1.700000025331974, :y 1.6717784775992683} {:x 1.8000000268220901, :y 1.7070825787734616} {:x 1.9000000283122063, :y 1.701419318758029} {:x 2.0000000298023224, :y 1.653643600395876} {:x 2.1000000312924385, :y 1.5647738284498447} {:x 2.2000000327825546, :y 1.4380661097740322} {:x 2.3000000342726707, :y 1.2789753467037075} {:x 2.400000035762787, :y 1.0950011506879587} {:x 2.500000037252903, :y 0.8954221922070159} {:x 2.600000038743019, :y 0.6909282486131232} {:x 2.7000000402331352, :y 0.49316454088117767} {:x 2.8000000417232513, :y 0.3142077010199231} {:x 2.9000000432133675, :y 0.16599664468526332} {:x 3.0000000447034836, :y 0.05974453344218124} {:x 3.1000000461935997, :y 0.005359737795470119} {:x 3.200000047683716, :y 0.010904147933380758} {:x 3.300000049173832, :y 0.08211627504701455} {:x 3.400000050663948, :y 0.22202435482841815} {:x 3.500000052154064, :y 0.43067118124254894} {:x 3.6000000536441803, :y 0.7049677976191696} {:x 3.7000000551342964, :y 1.0386876427986966} {:x 3.8000000566244125, :y 1.4226065285621616} {:x 3.9000000581145287, :y 1.8447871837088208} {:x 4.000000059604645, :y 2.2910003376371773} {:x 4.100000061094761, :y 2.7452677415667375} {:x 4.200000062584877, :y 3.190506445824144} {:x 4.300000064074993, :y 3.6092483578629087} {:x 4.400000065565109, :y 3.9844048590485874} {:x 4.500000067055225, :y 4.300043277672282} {:x 4.6000000685453415, :y 4.542140467191822} {:x 4.700000070035458, :y 4.699278726962665} {:x 4.800000071525574, :y 4.7632508650334975} {:x 4.90000007301569, :y 4.729544306014563} {:x 5.000000074505806, :y 4.597678688538202} {:x 5.100000075995922, :y 4.3713772065510605} {:x 5.200000077486038, :y 4.05855879623405} {:x 5.300000078976154, :y 3.671145874000224} {:x 5.4000000804662704, :y 3.224690369246717} {:x 5.500000081956387, :y 2.7378289205739783} {:x 5.600000083446503, :y 2.2315859569905676} {:x 5.700000084936619, :y 1.7285506115833968} {:x 5.800000086426735, :y 1.2519596798650976} {:x 5.900000087916851, :y 0.8247238395284076} {:x 6.000000089406967, :y 0.46843784294268836} {:x 6.100000090897083, :y 0.20241718996047256} {:x 6.200000092387199, :y 0.04280376871522386} {:x 6.3000000938773155, :y 0.0017810755840114996} {:x 6.400000095367432, :y 0.08693593236609283} {:x 6.500000096857548, :y 0.30079822929542793} {:x 6.600000098347664, :y 0.6405833336437111} {:x 6.70000009983778, :y 1.0981536817068416} {:x 6.800000101327896, :y 1.6602070423863446} {:x 6.900000102818012, :y 2.308689374977461} {:x 7.000000104308128, :y 3.0214205045940603} {:x 7.1000001057982445, :y 3.77291142228647} {:x 7.200000107288361, :y 4.5353432958990245} {:x 7.300000108778477, :y 5.2796706428663045} {:x 7.400000110268593, :y 5.976804919197644} {:x 7.500000111758709, :y 6.598830316615712} {:x 7.600000113248825, :y 7.120201060180292} {:x 7.700000114738941, :y 7.518869109857416} {:x 7.800000116229057, :y 7.777292951795491} {:x 7.900000117719173, :y 7.883282087047769} {:x 8.00000011920929, :y 7.830637763413025} {:x 8.100000120699406, :y 7.619558237636681} {:x 8.200000122189522, :y 7.256786107003342} {:x 8.300000123679638, :y 6.755485647926689} {:x 8.400000125169754, :y 6.134849225333044} {:x 8.50000012665987, :y 5.419443232426153} {:x 8.600000128149986, :y 4.638315207789705} {:x 8.700000129640102, :y 3.823894277701979} {:x 8.800000131130219, :y 3.0107264283034625} {:x 8.900000132620335, :y 2.2340939073053003} {:x 9.00000013411045, :y 1.528573929242129} {:x 9.100000135600567, :y 0.9265955276366005} {:x 9.200000137090683, :y 0.45705466575180215} {:x 9.300000138580799, :y 0.1440464866800439} {:x 9.400000140070915, :y 0.0057698589043499015} {:x 9.500000141561031, :y 0.05365326597234279} {:x 9.600000143051147, :y 0.2917428116738198} {:x 9.700000144541264, :y 0.7163829732921025} {:x 9.80000014603138, :y 1.316209126913524} {:x 9.900000147521496, :y 2.0724582507409313})}), :marks ({:type \"symbol\", :from {:data \"77df1d3a-c1a9-4268-a3e7-e9a9a58561c6\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 0.3}}, :update {:shape \"circle\", :size {:value 50}, :stroke {:value \"transparent\"}}, :hover {:size {:value 150}, :stroke {:value \"white\"}}}} {:type \"line\", :from {:data \"af9febcd-43c7-4018-acdf-ef02a96c8669\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"#FF29D2\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}})}}"}
;; <=

;; **
;;; There's a histogram plot type as well. As above, it composes with list-plots well.
;; **

;; @@
(compose
  (histogram (repeatedly 1000 #(rand 10)) :color "steelblue" :bins 100 :normalize :probability-density)
  (plot (constantly 0.1) [0 10]))
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50},"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"a32d6636-923b-4134-8a55-b1be49fa410a","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"a32d6636-923b-4134-8a55-b1be49fa410a","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"data":[{"name":"a32d6636-923b-4134-8a55-b1be49fa410a","values":[{"x":0.011659436055234673,"y":0},{"x":0.11145391488828783,"y":0.1002059444263351},{"x":0.211248393721341,"y":0.08016475554106808},{"x":0.31104287255439417,"y":0.13026772775423565},{"x":0.4108373513874473,"y":0.13026772775423565},{"x":0.5106318302205005,"y":0.1002059444263351},{"x":0.6104263090535537,"y":0.08016475554106808},{"x":0.7102207878866068,"y":0.08016475554106808},{"x":0.81001526671966,"y":0.07014416109843458},{"x":0.9098097455527131,"y":0.09018534998370159},{"x":1.0096042243857664,"y":0.05010297221316755},{"x":1.1093987032188195,"y":0.08016475554106808},{"x":1.2091931820518727,"y":0.13026772775423565},{"x":1.3089876608849258,"y":0.12024713331160213},{"x":1.408782139717979,"y":0.11022653886896862},{"x":1.508576618551032,"y":0.12024713331160213},{"x":1.6083710973840852,"y":0.13026772775423565},{"x":1.7081655762171384,"y":0.11022653886896862},{"x":1.8079600550501915,"y":0.11022653886896862},{"x":1.9077545338832447,"y":0.12024713331160213},{"x":2.007549012716298,"y":0.13026772775423565},{"x":2.107343491549351,"y":0.09018534998370159},{"x":2.2071379703824046,"y":0.12024713331160213},{"x":2.306932449215458,"y":0.08016475554106808},{"x":2.4067269280485113,"y":0.15030891663950266},{"x":2.5065214068815647,"y":0.06012356665580106},{"x":2.606315885714618,"y":0.1002059444263351},{"x":2.7061103645476714,"y":0.07014416109843458},{"x":2.8059048433807248,"y":0.11022653886896862},{"x":2.905699322213778,"y":0.07014416109843458},{"x":3.0054938010468315,"y":0.11022653886896862},{"x":3.105288279879885,"y":0.11022653886896862},{"x":3.2050827587129382,"y":0.09018534998370159},{"x":3.3048772375459916,"y":0.1002059444263351},{"x":3.404671716379045,"y":0.11022653886896862},{"x":3.5044661952120983,"y":0.11022653886896862},{"x":3.6042606740451517,"y":0.1002059444263351},{"x":3.704055152878205,"y":0.14028832219686915},{"x":3.8038496317112584,"y":0.1002059444263351},{"x":3.903644110544312,"y":0.08016475554106808},{"x":4.003438589377365,"y":0.09018534998370159},{"x":4.103233068210418,"y":0.1002059444263351},{"x":4.2030275470434715,"y":0.04008237777053404},{"x":4.302822025876525,"y":0.15030891663950266},{"x":4.402616504709578,"y":0.1002059444263351},{"x":4.502410983542632,"y":0.1002059444263351},{"x":4.602205462375685,"y":0.15030891663950266},{"x":4.701999941208738,"y":0.14028832219686915},{"x":4.801794420041792,"y":0.13026772775423565},{"x":4.901588898874845,"y":0.08016475554106808},{"x":5.001383377707898,"y":0.11022653886896862},{"x":5.101177856540952,"y":0.1002059444263351},{"x":5.200972335374005,"y":0.08016475554106808},{"x":5.3007668142070585,"y":0.07014416109843458},{"x":5.400561293040112,"y":0.05010297221316755},{"x":5.500355771873165,"y":0.08016475554106808},{"x":5.600150250706219,"y":0.09018534998370159},{"x":5.699944729539272,"y":0.16032951108213617},{"x":5.799739208372325,"y":0.06012356665580106},{"x":5.899533687205379,"y":0.05010297221316755},{"x":5.999328166038432,"y":0.11022653886896862},{"x":6.0991226448714855,"y":0.07014416109843458},{"x":6.198917123704539,"y":0.11022653886896862},{"x":6.298711602537592,"y":0.12024713331160213},{"x":6.398506081370646,"y":0.11022653886896862},{"x":6.498300560203699,"y":0.04008237777053404},{"x":6.598095039036752,"y":0.06012356665580106},{"x":6.697889517869806,"y":0.15030891663950266},{"x":6.797683996702859,"y":0.14028832219686915},{"x":6.897478475535912,"y":0.14028832219686915},{"x":6.997272954368966,"y":0.13026772775423565},{"x":7.097067433202019,"y":0.17035010552476967},{"x":7.1968619120350725,"y":0.16032951108213617},{"x":7.296656390868126,"y":0.11022653886896862},{"x":7.396450869701179,"y":0.13026772775423565},{"x":7.496245348534233,"y":0.12024713331160213},{"x":7.596039827367286,"y":0.08016475554106808},{"x":7.695834306200339,"y":0.1002059444263351},{"x":7.795628785033393,"y":0.06012356665580106},{"x":7.895423263866446,"y":0.09018534998370159},{"x":7.9952177426994995,"y":0.1002059444263351},{"x":8.095012221532553,"y":0.06012356665580106},{"x":8.194806700365605,"y":0.12024713331160213},{"x":8.294601179198658,"y":0.12024713331160213},{"x":8.39439565803171,"y":0.15030891663950266},{"x":8.494190136864763,"y":0.12024713331160213},{"x":8.593984615697815,"y":0.03006178332790053},{"x":8.693779094530868,"y":0.08016475554106808},{"x":8.79357357336392,"y":0.12024713331160213},{"x":8.893368052196973,"y":0.08016475554106808},{"x":8.993162531030025,"y":0.05010297221316755},{"x":9.092957009863078,"y":0.1002059444263351},{"x":9.19275148869613,"y":0.11022653886896862},{"x":9.292545967529183,"y":0.1002059444263351},{"x":9.392340446362235,"y":0.07014416109843458},{"x":9.492134925195288,"y":0.03006178332790053},{"x":9.59192940402834,"y":0.09018534998370159},{"x":9.691723882861393,"y":0.09018534998370159},{"x":9.791518361694445,"y":0.1002059444263351},{"x":9.891312840527497,"y":0.05010297221316755},{"x":9.99110731936055,"y":0.13026772775423565},{"x":10.090901798193602,"y":0}]},{"name":"e1a54459-be80-4fa8-999f-a8c5c6dc88c9","values":[{"x":0,"y":0.1},{"x":0.10000000149011612,"y":0.1},{"x":0.20000000298023224,"y":0.1},{"x":0.30000000447034836,"y":0.1},{"x":0.4000000059604645,"y":0.1},{"x":0.5000000074505806,"y":0.1},{"x":0.6000000089406967,"y":0.1},{"x":0.7000000104308128,"y":0.1},{"x":0.800000011920929,"y":0.1},{"x":0.9000000134110451,"y":0.1},{"x":1.0000000149011612,"y":0.1},{"x":1.1000000163912773,"y":0.1},{"x":1.2000000178813934,"y":0.1},{"x":1.3000000193715096,"y":0.1},{"x":1.4000000208616257,"y":0.1},{"x":1.5000000223517418,"y":0.1},{"x":1.600000023841858,"y":0.1},{"x":1.700000025331974,"y":0.1},{"x":1.8000000268220901,"y":0.1},{"x":1.9000000283122063,"y":0.1},{"x":2.0000000298023224,"y":0.1},{"x":2.1000000312924385,"y":0.1},{"x":2.2000000327825546,"y":0.1},{"x":2.3000000342726707,"y":0.1},{"x":2.400000035762787,"y":0.1},{"x":2.500000037252903,"y":0.1},{"x":2.600000038743019,"y":0.1},{"x":2.7000000402331352,"y":0.1},{"x":2.8000000417232513,"y":0.1},{"x":2.9000000432133675,"y":0.1},{"x":3.0000000447034836,"y":0.1},{"x":3.1000000461935997,"y":0.1},{"x":3.200000047683716,"y":0.1},{"x":3.300000049173832,"y":0.1},{"x":3.400000050663948,"y":0.1},{"x":3.500000052154064,"y":0.1},{"x":3.6000000536441803,"y":0.1},{"x":3.7000000551342964,"y":0.1},{"x":3.8000000566244125,"y":0.1},{"x":3.9000000581145287,"y":0.1},{"x":4.000000059604645,"y":0.1},{"x":4.100000061094761,"y":0.1},{"x":4.200000062584877,"y":0.1},{"x":4.300000064074993,"y":0.1},{"x":4.400000065565109,"y":0.1},{"x":4.500000067055225,"y":0.1},{"x":4.6000000685453415,"y":0.1},{"x":4.700000070035458,"y":0.1},{"x":4.800000071525574,"y":0.1},{"x":4.90000007301569,"y":0.1},{"x":5.000000074505806,"y":0.1},{"x":5.100000075995922,"y":0.1},{"x":5.200000077486038,"y":0.1},{"x":5.300000078976154,"y":0.1},{"x":5.4000000804662704,"y":0.1},{"x":5.500000081956387,"y":0.1},{"x":5.600000083446503,"y":0.1},{"x":5.700000084936619,"y":0.1},{"x":5.800000086426735,"y":0.1},{"x":5.900000087916851,"y":0.1},{"x":6.000000089406967,"y":0.1},{"x":6.100000090897083,"y":0.1},{"x":6.200000092387199,"y":0.1},{"x":6.3000000938773155,"y":0.1},{"x":6.400000095367432,"y":0.1},{"x":6.500000096857548,"y":0.1},{"x":6.600000098347664,"y":0.1},{"x":6.70000009983778,"y":0.1},{"x":6.800000101327896,"y":0.1},{"x":6.900000102818012,"y":0.1},{"x":7.000000104308128,"y":0.1},{"x":7.1000001057982445,"y":0.1},{"x":7.200000107288361,"y":0.1},{"x":7.300000108778477,"y":0.1},{"x":7.400000110268593,"y":0.1},{"x":7.500000111758709,"y":0.1},{"x":7.600000113248825,"y":0.1},{"x":7.700000114738941,"y":0.1},{"x":7.800000116229057,"y":0.1},{"x":7.900000117719173,"y":0.1},{"x":8.00000011920929,"y":0.1},{"x":8.100000120699406,"y":0.1},{"x":8.200000122189522,"y":0.1},{"x":8.300000123679638,"y":0.1},{"x":8.400000125169754,"y":0.1},{"x":8.50000012665987,"y":0.1},{"x":8.600000128149986,"y":0.1},{"x":8.700000129640102,"y":0.1},{"x":8.800000131130219,"y":0.1},{"x":8.900000132620335,"y":0.1},{"x":9.00000013411045,"y":0.1},{"x":9.100000135600567,"y":0.1},{"x":9.200000137090683,"y":0.1},{"x":9.300000138580799,"y":0.1},{"x":9.400000140070915,"y":0.1},{"x":9.500000141561031,"y":0.1},{"x":9.600000143051147,"y":0.1},{"x":9.700000144541264,"y":0.1},{"x":9.80000014603138,"y":0.1},{"x":9.900000147521496,"y":0.1}]}],"marks":[{"type":"line","from":{"data":"a32d6636-923b-4134-8a55-b1be49fa410a"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"interpolate":{"value":"step-before"},"fill":{"value":"steelblue"},"fillOpacity":{"value":0.4},"stroke":{"value":"steelblue"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}},{"type":"line","from":{"data":"e1a54459-be80-4fa8-999f-a8c5c6dc88c9"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"#FF29D2"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}, :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"a32d6636-923b-4134-8a55-b1be49fa410a\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"a32d6636-923b-4134-8a55-b1be49fa410a\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :data ({:name \"a32d6636-923b-4134-8a55-b1be49fa410a\", :values ({:x 0.011659436055234673, :y 0} {:x 0.11145391488828783, :y 0.1002059444263351} {:x 0.211248393721341, :y 0.08016475554106808} {:x 0.31104287255439417, :y 0.13026772775423565} {:x 0.4108373513874473, :y 0.13026772775423565} {:x 0.5106318302205005, :y 0.1002059444263351} {:x 0.6104263090535537, :y 0.08016475554106808} {:x 0.7102207878866068, :y 0.08016475554106808} {:x 0.81001526671966, :y 0.07014416109843458} {:x 0.9098097455527131, :y 0.09018534998370159} {:x 1.0096042243857664, :y 0.05010297221316755} {:x 1.1093987032188195, :y 0.08016475554106808} {:x 1.2091931820518727, :y 0.13026772775423565} {:x 1.3089876608849258, :y 0.12024713331160213} {:x 1.408782139717979, :y 0.11022653886896862} {:x 1.508576618551032, :y 0.12024713331160213} {:x 1.6083710973840852, :y 0.13026772775423565} {:x 1.7081655762171384, :y 0.11022653886896862} {:x 1.8079600550501915, :y 0.11022653886896862} {:x 1.9077545338832447, :y 0.12024713331160213} {:x 2.007549012716298, :y 0.13026772775423565} {:x 2.107343491549351, :y 0.09018534998370159} {:x 2.2071379703824046, :y 0.12024713331160213} {:x 2.306932449215458, :y 0.08016475554106808} {:x 2.4067269280485113, :y 0.15030891663950266} {:x 2.5065214068815647, :y 0.06012356665580106} {:x 2.606315885714618, :y 0.1002059444263351} {:x 2.7061103645476714, :y 0.07014416109843458} {:x 2.8059048433807248, :y 0.11022653886896862} {:x 2.905699322213778, :y 0.07014416109843458} {:x 3.0054938010468315, :y 0.11022653886896862} {:x 3.105288279879885, :y 0.11022653886896862} {:x 3.2050827587129382, :y 0.09018534998370159} {:x 3.3048772375459916, :y 0.1002059444263351} {:x 3.404671716379045, :y 0.11022653886896862} {:x 3.5044661952120983, :y 0.11022653886896862} {:x 3.6042606740451517, :y 0.1002059444263351} {:x 3.704055152878205, :y 0.14028832219686915} {:x 3.8038496317112584, :y 0.1002059444263351} {:x 3.903644110544312, :y 0.08016475554106808} {:x 4.003438589377365, :y 0.09018534998370159} {:x 4.103233068210418, :y 0.1002059444263351} {:x 4.2030275470434715, :y 0.04008237777053404} {:x 4.302822025876525, :y 0.15030891663950266} {:x 4.402616504709578, :y 0.1002059444263351} {:x 4.502410983542632, :y 0.1002059444263351} {:x 4.602205462375685, :y 0.15030891663950266} {:x 4.701999941208738, :y 0.14028832219686915} {:x 4.801794420041792, :y 0.13026772775423565} {:x 4.901588898874845, :y 0.08016475554106808} {:x 5.001383377707898, :y 0.11022653886896862} {:x 5.101177856540952, :y 0.1002059444263351} {:x 5.200972335374005, :y 0.08016475554106808} {:x 5.3007668142070585, :y 0.07014416109843458} {:x 5.400561293040112, :y 0.05010297221316755} {:x 5.500355771873165, :y 0.08016475554106808} {:x 5.600150250706219, :y 0.09018534998370159} {:x 5.699944729539272, :y 0.16032951108213617} {:x 5.799739208372325, :y 0.06012356665580106} {:x 5.899533687205379, :y 0.05010297221316755} {:x 5.999328166038432, :y 0.11022653886896862} {:x 6.0991226448714855, :y 0.07014416109843458} {:x 6.198917123704539, :y 0.11022653886896862} {:x 6.298711602537592, :y 0.12024713331160213} {:x 6.398506081370646, :y 0.11022653886896862} {:x 6.498300560203699, :y 0.04008237777053404} {:x 6.598095039036752, :y 0.06012356665580106} {:x 6.697889517869806, :y 0.15030891663950266} {:x 6.797683996702859, :y 0.14028832219686915} {:x 6.897478475535912, :y 0.14028832219686915} {:x 6.997272954368966, :y 0.13026772775423565} {:x 7.097067433202019, :y 0.17035010552476967} {:x 7.1968619120350725, :y 0.16032951108213617} {:x 7.296656390868126, :y 0.11022653886896862} {:x 7.396450869701179, :y 0.13026772775423565} {:x 7.496245348534233, :y 0.12024713331160213} {:x 7.596039827367286, :y 0.08016475554106808} {:x 7.695834306200339, :y 0.1002059444263351} {:x 7.795628785033393, :y 0.06012356665580106} {:x 7.895423263866446, :y 0.09018534998370159} {:x 7.9952177426994995, :y 0.1002059444263351} {:x 8.095012221532553, :y 0.06012356665580106} {:x 8.194806700365605, :y 0.12024713331160213} {:x 8.294601179198658, :y 0.12024713331160213} {:x 8.39439565803171, :y 0.15030891663950266} {:x 8.494190136864763, :y 0.12024713331160213} {:x 8.593984615697815, :y 0.03006178332790053} {:x 8.693779094530868, :y 0.08016475554106808} {:x 8.79357357336392, :y 0.12024713331160213} {:x 8.893368052196973, :y 0.08016475554106808} {:x 8.993162531030025, :y 0.05010297221316755} {:x 9.092957009863078, :y 0.1002059444263351} {:x 9.19275148869613, :y 0.11022653886896862} {:x 9.292545967529183, :y 0.1002059444263351} {:x 9.392340446362235, :y 0.07014416109843458} {:x 9.492134925195288, :y 0.03006178332790053} {:x 9.59192940402834, :y 0.09018534998370159} {:x 9.691723882861393, :y 0.09018534998370159} {:x 9.791518361694445, :y 0.1002059444263351} {:x 9.891312840527497, :y 0.05010297221316755} {:x 9.99110731936055, :y 0.13026772775423565} {:x 10.090901798193602, :y 0})} {:name \"e1a54459-be80-4fa8-999f-a8c5c6dc88c9\", :values ({:x 0, :y 0.1} {:x 0.10000000149011612, :y 0.1} {:x 0.20000000298023224, :y 0.1} {:x 0.30000000447034836, :y 0.1} {:x 0.4000000059604645, :y 0.1} {:x 0.5000000074505806, :y 0.1} {:x 0.6000000089406967, :y 0.1} {:x 0.7000000104308128, :y 0.1} {:x 0.800000011920929, :y 0.1} {:x 0.9000000134110451, :y 0.1} {:x 1.0000000149011612, :y 0.1} {:x 1.1000000163912773, :y 0.1} {:x 1.2000000178813934, :y 0.1} {:x 1.3000000193715096, :y 0.1} {:x 1.4000000208616257, :y 0.1} {:x 1.5000000223517418, :y 0.1} {:x 1.600000023841858, :y 0.1} {:x 1.700000025331974, :y 0.1} {:x 1.8000000268220901, :y 0.1} {:x 1.9000000283122063, :y 0.1} {:x 2.0000000298023224, :y 0.1} {:x 2.1000000312924385, :y 0.1} {:x 2.2000000327825546, :y 0.1} {:x 2.3000000342726707, :y 0.1} {:x 2.400000035762787, :y 0.1} {:x 2.500000037252903, :y 0.1} {:x 2.600000038743019, :y 0.1} {:x 2.7000000402331352, :y 0.1} {:x 2.8000000417232513, :y 0.1} {:x 2.9000000432133675, :y 0.1} {:x 3.0000000447034836, :y 0.1} {:x 3.1000000461935997, :y 0.1} {:x 3.200000047683716, :y 0.1} {:x 3.300000049173832, :y 0.1} {:x 3.400000050663948, :y 0.1} {:x 3.500000052154064, :y 0.1} {:x 3.6000000536441803, :y 0.1} {:x 3.7000000551342964, :y 0.1} {:x 3.8000000566244125, :y 0.1} {:x 3.9000000581145287, :y 0.1} {:x 4.000000059604645, :y 0.1} {:x 4.100000061094761, :y 0.1} {:x 4.200000062584877, :y 0.1} {:x 4.300000064074993, :y 0.1} {:x 4.400000065565109, :y 0.1} {:x 4.500000067055225, :y 0.1} {:x 4.6000000685453415, :y 0.1} {:x 4.700000070035458, :y 0.1} {:x 4.800000071525574, :y 0.1} {:x 4.90000007301569, :y 0.1} {:x 5.000000074505806, :y 0.1} {:x 5.100000075995922, :y 0.1} {:x 5.200000077486038, :y 0.1} {:x 5.300000078976154, :y 0.1} {:x 5.4000000804662704, :y 0.1} {:x 5.500000081956387, :y 0.1} {:x 5.600000083446503, :y 0.1} {:x 5.700000084936619, :y 0.1} {:x 5.800000086426735, :y 0.1} {:x 5.900000087916851, :y 0.1} {:x 6.000000089406967, :y 0.1} {:x 6.100000090897083, :y 0.1} {:x 6.200000092387199, :y 0.1} {:x 6.3000000938773155, :y 0.1} {:x 6.400000095367432, :y 0.1} {:x 6.500000096857548, :y 0.1} {:x 6.600000098347664, :y 0.1} {:x 6.70000009983778, :y 0.1} {:x 6.800000101327896, :y 0.1} {:x 6.900000102818012, :y 0.1} {:x 7.000000104308128, :y 0.1} {:x 7.1000001057982445, :y 0.1} {:x 7.200000107288361, :y 0.1} {:x 7.300000108778477, :y 0.1} {:x 7.400000110268593, :y 0.1} {:x 7.500000111758709, :y 0.1} {:x 7.600000113248825, :y 0.1} {:x 7.700000114738941, :y 0.1} {:x 7.800000116229057, :y 0.1} {:x 7.900000117719173, :y 0.1} {:x 8.00000011920929, :y 0.1} {:x 8.100000120699406, :y 0.1} {:x 8.200000122189522, :y 0.1} {:x 8.300000123679638, :y 0.1} {:x 8.400000125169754, :y 0.1} {:x 8.50000012665987, :y 0.1} {:x 8.600000128149986, :y 0.1} {:x 8.700000129640102, :y 0.1} {:x 8.800000131130219, :y 0.1} {:x 8.900000132620335, :y 0.1} {:x 9.00000013411045, :y 0.1} {:x 9.100000135600567, :y 0.1} {:x 9.200000137090683, :y 0.1} {:x 9.300000138580799, :y 0.1} {:x 9.400000140070915, :y 0.1} {:x 9.500000141561031, :y 0.1} {:x 9.600000143051147, :y 0.1} {:x 9.700000144541264, :y 0.1} {:x 9.80000014603138, :y 0.1} {:x 9.900000147521496, :y 0.1})}), :marks ({:type \"line\", :from {:data \"a32d6636-923b-4134-8a55-b1be49fa410a\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :interpolate {:value \"step-before\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 0.4}, :stroke {:value \"steelblue\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}} {:type \"line\", :from {:data \"e1a54459-be80-4fa8-999f-a8c5c6dc88c9\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"#FF29D2\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}})}}"}
;; <=

;; **
;;; Something a little fancier!
;; **

;; @@
(compose
  (histogram (repeatedly 10000 #(- (apply + (repeatedly 50 rand)) 25))
             :bins 100
             :normalise :probability-density)
  (plot #(gaussian % 2 0) [-10 10]))
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50},"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"faec6672-dc35-4c6e-be36-5dba334402cc","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"faec6672-dc35-4c6e-be36-5dba334402cc","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"data":[{"name":"faec6672-dc35-4c6e-be36-5dba334402cc","values":[{"x":-7.457974737400555,"y":0},{"x":-7.312168126387523,"y":6.858399581831196E-4},{"x":-7.166361515374492,"y":6.858399581831196E-4},{"x":-7.02055490436146,"y":0.0},{"x":-6.8747482933484285,"y":6.858399581831196E-4},{"x":-6.728941682335397,"y":0.0},{"x":-6.583135071322365,"y":0.0},{"x":-6.437328460309334,"y":6.858399581831196E-4},{"x":-6.291521849296302,"y":0.0027433598327324784},{"x":-6.1457152382832705,"y":0.0027433598327324784},{"x":-5.999908627270239,"y":6.858399581831196E-4},{"x":-5.854102016257207,"y":0.003429199790915598},{"x":-5.708295405244176,"y":0.0027433598327324784},{"x":-5.562488794231144,"y":0.005486719665464957},{"x":-5.4166821832181125,"y":0.006858399581831196},{"x":-5.270875572205081,"y":0.004800879707281837},{"x":-5.125068961192049,"y":0.0027433598327324784},{"x":-4.979262350179018,"y":0.007544239540014316},{"x":-4.833455739165986,"y":0.011659279289113034},{"x":-4.687649128152954,"y":0.013716799163662393},{"x":-4.541842517139923,"y":0.02057519874549359},{"x":-4.396035906126891,"y":0.024004398536409188},{"x":-4.25022929511386,"y":0.02057519874549359},{"x":-4.104422684100828,"y":0.024690238494592308},{"x":-3.9586160730877964,"y":0.0315486380764235},{"x":-3.812809462074765,"y":0.028119438285507906},{"x":-3.667002851061733,"y":0.03909287761643782},{"x":-3.5211962400487016,"y":0.041836237449170297},{"x":-3.37538962903567,"y":0.03703535774188846},{"x":-3.2295830180226384,"y":0.05829639644556517},{"x":-3.0837764070096068,"y":0.05486719665464957},{"x":-2.937969795996575,"y":0.06789815586012884},{"x":-2.7921631849835435,"y":0.06926983577649508},{"x":-2.646356573970512,"y":0.07612823535832627},{"x":-2.5005499629574803,"y":0.09053087448017179},{"x":-2.3547433519444487,"y":0.09807511402018611},{"x":-2.208936740931417,"y":0.10013263389473546},{"x":-2.0631301299183855,"y":0.10973439330929914},{"x":-1.917323518905354,"y":0.1357963117202577},{"x":-1.7715169078923223,"y":0.1392255115111733},{"x":-1.6257102968792907,"y":0.1364821516784408},{"x":-1.479903685866259,"y":0.15157063075846944},{"x":-1.3340970748532275,"y":0.16322991004758247},{"x":-1.1882904638401959,"y":0.16597326988031494},{"x":-1.0424838528271643,"y":0.1707741495875968},{"x":-0.8966772418141327,"y":0.16460158996394872},{"x":-0.7508706308011011,"y":0.17694670921124486},{"x":-0.6050640197880695,"y":0.17351750942032926},{"x":-0.45925740877503785,"y":0.1879201485421748},{"x":-0.31345079776200624,"y":0.189291828458541},{"x":-0.1676441867489746,"y":0.18174758891852671},{"x":-0.02183757573594297,"y":0.19615022804037222},{"x":0.12396903527708866,"y":0.19409270816582286},{"x":0.2697756462901203,"y":0.2023227876640203},{"x":0.4155822573031519,"y":0.19409270816582286},{"x":0.5613888683161835,"y":0.17763254916942797},{"x":0.7071954793292151,"y":0.1886059885003579},{"x":0.8530020903422467,"y":0.17283166946214615},{"x":0.9988087013552783,"y":0.16322991004758247},{"x":1.14461531236831,"y":0.1742033493785124},{"x":1.2904219233813417,"y":0.16117239017303311},{"x":1.4362285343943733,"y":0.14608391109300448},{"x":1.5820351454074049,"y":0.1399113514693564},{"x":1.7278417564204365,"y":0.1392255115111733},{"x":1.873648367433468,"y":0.13236711192934209},{"x":2.0194549784465,"y":0.12962375209660962},{"x":2.1652615894595315,"y":0.11933615272386282},{"x":2.311068200472563,"y":0.10767687343474978},{"x":2.4568748114855947,"y":0.1152211129747641},{"x":2.6026814224986263,"y":0.08298663494015747},{"x":2.748488033511658,"y":0.08367247489834059},{"x":2.8942946445246895,"y":0.07887159519105875},{"x":3.040101255537721,"y":0.07612823535832627},{"x":3.1859078665507528,"y":0.06103975627829765},{"x":3.3317144775637844,"y":0.050752156905550856},{"x":3.477521088576816,"y":0.043893757323719654},{"x":3.6233276995898476,"y":0.050752156905550856},{"x":3.769134310602879,"y":0.03566367782552222},{"x":3.914940921615911,"y":0.0384070376582547},{"x":4.060747532628943,"y":0.030862798118240383},{"x":4.206554143641974,"y":0.030862798118240383},{"x":4.352360754655006,"y":0.01988935878731047},{"x":4.498167365668038,"y":0.025376078452775428},{"x":4.643973976681069,"y":0.022632718620042947},{"x":4.789780587694101,"y":0.015088479080028631},{"x":4.9355871987071325,"y":0.010973439330929913},{"x":5.081393809720164,"y":0.010287599372746795},{"x":5.227200420733196,"y":0.007544239540014316},{"x":5.373007031746227,"y":0.008915919456380554},{"x":5.518813642759259,"y":0.006172559623648077},{"x":5.6646202537722905,"y":0.004115039749098718},{"x":5.810426864785322,"y":0.0027433598327324784},{"x":5.956233475798354,"y":6.858399581831196E-4},{"x":6.102040086811385,"y":0.004115039749098718},{"x":6.247846697824417,"y":0.0013716799163662392},{"x":6.3936533088374485,"y":6.858399581831196E-4},{"x":6.53945991985048,"y":6.858399581831196E-4},{"x":6.685266530863512,"y":0.0},{"x":6.831073141876543,"y":6.858399581831196E-4},{"x":6.976879752889575,"y":6.858399581831196E-4},{"x":7.122686363902607,"y":0.0},{"x":7.268492974915638,"y":6.858399581831196E-4},{"x":7.41429958592867,"y":0}]},{"name":"47562395-7f2d-4223-a03e-7be897608dde","values":[{"x":-10,"y":7.453306344157342E-7},{"x":-9.799999997019768,"y":1.2227136022020082E-6},{"x":-9.599999994039536,"y":1.9859008895787535E-6},{"x":-9.399999991059303,"y":3.1933568466554097E-6},{"x":-9.199999988079071,"y":5.083869442630077E-6},{"x":-8.999999985098839,"y":8.013059747248974E-6},{"x":-8.799999982118607,"y":1.2504301246871565E-5},{"x":-8.599999979138374,"y":1.9318683610927946E-5},{"x":-8.399999976158142,"y":2.9549673525896843E-5},{"x":-8.19999997317791,"y":4.474916120466938E-5},{"x":-7.999999970197678,"y":6.709252957952864E-5},{"x":-7.799999967217445,"y":9.959109066711276E-5},{"x":-7.599999964237213,"y":1.463604937211866E-4},{"x":-7.399999961256981,"y":2.129532625969202E-4},{"x":-7.199999958276749,"y":3.0676215890329826E-4},{"x":-6.999999955296516,"y":4.3749825786254407E-4},{"x":-6.799999952316284,"y":6.177431317230414E-4},{"x":-6.599999949336052,"y":8.635680737170301E-4},{"x":-6.39999994635582,"y":0.0011952046815864247},{"x":-6.1999999433755875,"y":0.0016377403466157404},{"x":-5.999999940395355,"y":0.002221799506292808},{"x":-5.799999937415123,"y":0.002984157484620092},{"x":-5.599999934434891,"y":0.003968219313121465},{"x":-5.3999999314546585,"y":0.005224282454218924},{"x":-5.199999928474426,"y":0.006809491580088464},{"x":-4.999999925494194,"y":0.008787387543070678},{"x":-4.799999922513962,"y":0.011226953610745277},{"x":-4.5999999195337296,"y":0.014201072062040727},{"x":-4.399999916553497,"y":0.017784325124320887},{"x":-4.199999913573265,"y":0.022050107061901632},{"x":-3.999999910593033,"y":0.027067059067306068},{"x":-3.7999999076128006,"y":0.032894894202544614},{"x":-3.5999999046325684,"y":0.03957974321387936},{"x":-3.399999901652336,"y":0.04714921925263572},{"x":-3.199999898672104,"y":0.0556074645983085},{"x":-2.9999998956918716,"y":0.06493049855125375},{"x":-2.7999998927116394,"y":0.07506222540759178},{"x":-2.599999889731407,"y":0.08591147779981703},{"x":-2.399999886751175,"y":0.09735045780688895},{"x":-2.1999998837709427,"y":0.10921489230961066},{"x":-1.9999998807907104,"y":0.12130613917293559},{"x":-1.7999998778104782,"y":0.13339536950647682},{"x":-1.599999874830246,"y":0.1452298146860898},{"x":-1.3999998718500137,"y":0.1565409146696238},{"x":-1.1999998688697815,"y":0.1670540488540041},{"x":-0.9999998658895493,"y":0.17649938643452168},{"x":-0.799999862909317,"y":0.18462327433935283},{"x":-0.5999998599290848,"y":0.1911995003838428},{"x":-0.39999985694885254,"y":0.19603973746572148},{"x":-0.1999998539686203,"y":0.19900249729156638},{"x":1.4901161193847656E-7,"y":0.19999999999999946},{"x":0.20000015199184418,"y":0.1990024943261981},{"x":0.4000001549720764,"y":0.19603973162328203},{"x":0.6000001579523087,"y":0.19119949183655915},{"x":0.8000001609325409,"y":0.1846232633349483},{"x":1.0000001639127731,"y":0.1764993732842929},{"x":1.2000001668930054,"y":0.1670540339182087},{"x":1.4000001698732376,"y":0.15654089834113458},{"x":1.6000001728534698,"y":0.14522979737334754},{"x":1.800000175833702,"y":0.1333953516167646},{"x":2.0000001788139343,"y":0.12130612109691333},{"x":2.2000001817941666,"y":0.109214874407896},{"x":2.400000184774399,"y":0.09735044039927186},{"x":2.600000187754631,"y":0.08591146115746827},{"x":2.8000001907348633,"y":0.07506220974839269},{"x":3.0000001937150955,"y":0.06493048403815777},{"x":3.2000001966953278,"y":0.055607451340457185},{"x":3.40000019967556,"y":0.04714920730880908},{"x":3.6000002026557922,"y":0.03957973259776622},{"x":3.8000002056360245,"y":0.03289488488927549},{"x":4.000000208616257,"y":0.02706705100069495},{"x":4.200000211596489,"y":0.02205010016188641},{"x":4.400000214576721,"y":0.017784319294165653},{"x":4.600000217556953,"y":0.014201067194954814},{"x":4.800000220537186,"y":0.011226949595674444},{"x":5.000000223517418,"y":0.008787384269514283},{"x":5.20000022649765,"y":0.006809488941886312},{"x":5.400000229477882,"y":0.0052242803523266925},{"x":5.600000232458115,"y":0.003968217657451668},{"x":5.800000235438347,"y":0.002984156195065413},{"x":6.000000238418579,"y":0.0022217985130712366},{"x":6.200000241398811,"y":0.001637739590084685},{"x":6.400000244379044,"y":0.0011952041116685484},{"x":6.600000247359276,"y":8.635676490676143E-4},{"x":6.800000250339508,"y":6.177428187500568E-4},{"x":7.00000025331974,"y":4.3749802968947785E-4},{"x":7.2000002562999725,"y":3.0676199434329435E-4},{"x":7.400000259280205,"y":2.1295314518666783E-4},{"x":7.600000262260437,"y":1.46360410845439E-4},{"x":7.800000265240669,"y":9.959103279023577E-5},{"x":8.000000268220901,"y":6.709248958927605E-5},{"x":8.200000271201134,"y":4.47491338652843E-5},{"x":8.400000274181366,"y":2.9549655032275533E-5},{"x":8.600000277161598,"y":1.9318671232486508E-5},{"x":8.80000028014183,"y":1.2504293048415349E-5},{"x":9.000000283122063,"y":8.013054374075417E-6},{"x":9.200000286102295,"y":5.083865957875547E-6},{"x":9.400000289082527,"y":3.193354610174078E-6},{"x":9.60000029206276,"y":1.9858994691522355E-6},{"x":9.800000295042992,"y":1.2227127094295497E-6}]}],"marks":[{"type":"line","from":{"data":"faec6672-dc35-4c6e-be36-5dba334402cc"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"interpolate":{"value":"step-before"},"fill":{"value":"steelblue"},"fillOpacity":{"value":0.4},"stroke":{"value":"steelblue"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}},{"type":"line","from":{"data":"47562395-7f2d-4223-a03e-7be897608dde"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"#FF29D2"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}, :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"faec6672-dc35-4c6e-be36-5dba334402cc\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"faec6672-dc35-4c6e-be36-5dba334402cc\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :data ({:name \"faec6672-dc35-4c6e-be36-5dba334402cc\", :values ({:x -7.457974737400555, :y 0} {:x -7.312168126387523, :y 6.858399581831196E-4} {:x -7.166361515374492, :y 6.858399581831196E-4} {:x -7.02055490436146, :y 0.0} {:x -6.8747482933484285, :y 6.858399581831196E-4} {:x -6.728941682335397, :y 0.0} {:x -6.583135071322365, :y 0.0} {:x -6.437328460309334, :y 6.858399581831196E-4} {:x -6.291521849296302, :y 0.0027433598327324784} {:x -6.1457152382832705, :y 0.0027433598327324784} {:x -5.999908627270239, :y 6.858399581831196E-4} {:x -5.854102016257207, :y 0.003429199790915598} {:x -5.708295405244176, :y 0.0027433598327324784} {:x -5.562488794231144, :y 0.005486719665464957} {:x -5.4166821832181125, :y 0.006858399581831196} {:x -5.270875572205081, :y 0.004800879707281837} {:x -5.125068961192049, :y 0.0027433598327324784} {:x -4.979262350179018, :y 0.007544239540014316} {:x -4.833455739165986, :y 0.011659279289113034} {:x -4.687649128152954, :y 0.013716799163662393} {:x -4.541842517139923, :y 0.02057519874549359} {:x -4.396035906126891, :y 0.024004398536409188} {:x -4.25022929511386, :y 0.02057519874549359} {:x -4.104422684100828, :y 0.024690238494592308} {:x -3.9586160730877964, :y 0.0315486380764235} {:x -3.812809462074765, :y 0.028119438285507906} {:x -3.667002851061733, :y 0.03909287761643782} {:x -3.5211962400487016, :y 0.041836237449170297} {:x -3.37538962903567, :y 0.03703535774188846} {:x -3.2295830180226384, :y 0.05829639644556517} {:x -3.0837764070096068, :y 0.05486719665464957} {:x -2.937969795996575, :y 0.06789815586012884} {:x -2.7921631849835435, :y 0.06926983577649508} {:x -2.646356573970512, :y 0.07612823535832627} {:x -2.5005499629574803, :y 0.09053087448017179} {:x -2.3547433519444487, :y 0.09807511402018611} {:x -2.208936740931417, :y 0.10013263389473546} {:x -2.0631301299183855, :y 0.10973439330929914} {:x -1.917323518905354, :y 0.1357963117202577} {:x -1.7715169078923223, :y 0.1392255115111733} {:x -1.6257102968792907, :y 0.1364821516784408} {:x -1.479903685866259, :y 0.15157063075846944} {:x -1.3340970748532275, :y 0.16322991004758247} {:x -1.1882904638401959, :y 0.16597326988031494} {:x -1.0424838528271643, :y 0.1707741495875968} {:x -0.8966772418141327, :y 0.16460158996394872} {:x -0.7508706308011011, :y 0.17694670921124486} {:x -0.6050640197880695, :y 0.17351750942032926} {:x -0.45925740877503785, :y 0.1879201485421748} {:x -0.31345079776200624, :y 0.189291828458541} {:x -0.1676441867489746, :y 0.18174758891852671} {:x -0.02183757573594297, :y 0.19615022804037222} {:x 0.12396903527708866, :y 0.19409270816582286} {:x 0.2697756462901203, :y 0.2023227876640203} {:x 0.4155822573031519, :y 0.19409270816582286} {:x 0.5613888683161835, :y 0.17763254916942797} {:x 0.7071954793292151, :y 0.1886059885003579} {:x 0.8530020903422467, :y 0.17283166946214615} {:x 0.9988087013552783, :y 0.16322991004758247} {:x 1.14461531236831, :y 0.1742033493785124} {:x 1.2904219233813417, :y 0.16117239017303311} {:x 1.4362285343943733, :y 0.14608391109300448} {:x 1.5820351454074049, :y 0.1399113514693564} {:x 1.7278417564204365, :y 0.1392255115111733} {:x 1.873648367433468, :y 0.13236711192934209} {:x 2.0194549784465, :y 0.12962375209660962} {:x 2.1652615894595315, :y 0.11933615272386282} {:x 2.311068200472563, :y 0.10767687343474978} {:x 2.4568748114855947, :y 0.1152211129747641} {:x 2.6026814224986263, :y 0.08298663494015747} {:x 2.748488033511658, :y 0.08367247489834059} {:x 2.8942946445246895, :y 0.07887159519105875} {:x 3.040101255537721, :y 0.07612823535832627} {:x 3.1859078665507528, :y 0.06103975627829765} {:x 3.3317144775637844, :y 0.050752156905550856} {:x 3.477521088576816, :y 0.043893757323719654} {:x 3.6233276995898476, :y 0.050752156905550856} {:x 3.769134310602879, :y 0.03566367782552222} {:x 3.914940921615911, :y 0.0384070376582547} {:x 4.060747532628943, :y 0.030862798118240383} {:x 4.206554143641974, :y 0.030862798118240383} {:x 4.352360754655006, :y 0.01988935878731047} {:x 4.498167365668038, :y 0.025376078452775428} {:x 4.643973976681069, :y 0.022632718620042947} {:x 4.789780587694101, :y 0.015088479080028631} {:x 4.9355871987071325, :y 0.010973439330929913} {:x 5.081393809720164, :y 0.010287599372746795} {:x 5.227200420733196, :y 0.007544239540014316} {:x 5.373007031746227, :y 0.008915919456380554} {:x 5.518813642759259, :y 0.006172559623648077} {:x 5.6646202537722905, :y 0.004115039749098718} {:x 5.810426864785322, :y 0.0027433598327324784} {:x 5.956233475798354, :y 6.858399581831196E-4} {:x 6.102040086811385, :y 0.004115039749098718} {:x 6.247846697824417, :y 0.0013716799163662392} {:x 6.3936533088374485, :y 6.858399581831196E-4} {:x 6.53945991985048, :y 6.858399581831196E-4} {:x 6.685266530863512, :y 0.0} {:x 6.831073141876543, :y 6.858399581831196E-4} {:x 6.976879752889575, :y 6.858399581831196E-4} {:x 7.122686363902607, :y 0.0} {:x 7.268492974915638, :y 6.858399581831196E-4} {:x 7.41429958592867, :y 0})} {:name \"47562395-7f2d-4223-a03e-7be897608dde\", :values ({:x -10, :y 7.453306344157342E-7} {:x -9.799999997019768, :y 1.2227136022020082E-6} {:x -9.599999994039536, :y 1.9859008895787535E-6} {:x -9.399999991059303, :y 3.1933568466554097E-6} {:x -9.199999988079071, :y 5.083869442630077E-6} {:x -8.999999985098839, :y 8.013059747248974E-6} {:x -8.799999982118607, :y 1.2504301246871565E-5} {:x -8.599999979138374, :y 1.9318683610927946E-5} {:x -8.399999976158142, :y 2.9549673525896843E-5} {:x -8.19999997317791, :y 4.474916120466938E-5} {:x -7.999999970197678, :y 6.709252957952864E-5} {:x -7.799999967217445, :y 9.959109066711276E-5} {:x -7.599999964237213, :y 1.463604937211866E-4} {:x -7.399999961256981, :y 2.129532625969202E-4} {:x -7.199999958276749, :y 3.0676215890329826E-4} {:x -6.999999955296516, :y 4.3749825786254407E-4} {:x -6.799999952316284, :y 6.177431317230414E-4} {:x -6.599999949336052, :y 8.635680737170301E-4} {:x -6.39999994635582, :y 0.0011952046815864247} {:x -6.1999999433755875, :y 0.0016377403466157404} {:x -5.999999940395355, :y 0.002221799506292808} {:x -5.799999937415123, :y 0.002984157484620092} {:x -5.599999934434891, :y 0.003968219313121465} {:x -5.3999999314546585, :y 0.005224282454218924} {:x -5.199999928474426, :y 0.006809491580088464} {:x -4.999999925494194, :y 0.008787387543070678} {:x -4.799999922513962, :y 0.011226953610745277} {:x -4.5999999195337296, :y 0.014201072062040727} {:x -4.399999916553497, :y 0.017784325124320887} {:x -4.199999913573265, :y 0.022050107061901632} {:x -3.999999910593033, :y 0.027067059067306068} {:x -3.7999999076128006, :y 0.032894894202544614} {:x -3.5999999046325684, :y 0.03957974321387936} {:x -3.399999901652336, :y 0.04714921925263572} {:x -3.199999898672104, :y 0.0556074645983085} {:x -2.9999998956918716, :y 0.06493049855125375} {:x -2.7999998927116394, :y 0.07506222540759178} {:x -2.599999889731407, :y 0.08591147779981703} {:x -2.399999886751175, :y 0.09735045780688895} {:x -2.1999998837709427, :y 0.10921489230961066} {:x -1.9999998807907104, :y 0.12130613917293559} {:x -1.7999998778104782, :y 0.13339536950647682} {:x -1.599999874830246, :y 0.1452298146860898} {:x -1.3999998718500137, :y 0.1565409146696238} {:x -1.1999998688697815, :y 0.1670540488540041} {:x -0.9999998658895493, :y 0.17649938643452168} {:x -0.799999862909317, :y 0.18462327433935283} {:x -0.5999998599290848, :y 0.1911995003838428} {:x -0.39999985694885254, :y 0.19603973746572148} {:x -0.1999998539686203, :y 0.19900249729156638} {:x 1.4901161193847656E-7, :y 0.19999999999999946} {:x 0.20000015199184418, :y 0.1990024943261981} {:x 0.4000001549720764, :y 0.19603973162328203} {:x 0.6000001579523087, :y 0.19119949183655915} {:x 0.8000001609325409, :y 0.1846232633349483} {:x 1.0000001639127731, :y 0.1764993732842929} {:x 1.2000001668930054, :y 0.1670540339182087} {:x 1.4000001698732376, :y 0.15654089834113458} {:x 1.6000001728534698, :y 0.14522979737334754} {:x 1.800000175833702, :y 0.1333953516167646} {:x 2.0000001788139343, :y 0.12130612109691333} {:x 2.2000001817941666, :y 0.109214874407896} {:x 2.400000184774399, :y 0.09735044039927186} {:x 2.600000187754631, :y 0.08591146115746827} {:x 2.8000001907348633, :y 0.07506220974839269} {:x 3.0000001937150955, :y 0.06493048403815777} {:x 3.2000001966953278, :y 0.055607451340457185} {:x 3.40000019967556, :y 0.04714920730880908} {:x 3.6000002026557922, :y 0.03957973259776622} {:x 3.8000002056360245, :y 0.03289488488927549} {:x 4.000000208616257, :y 0.02706705100069495} {:x 4.200000211596489, :y 0.02205010016188641} {:x 4.400000214576721, :y 0.017784319294165653} {:x 4.600000217556953, :y 0.014201067194954814} {:x 4.800000220537186, :y 0.011226949595674444} {:x 5.000000223517418, :y 0.008787384269514283} {:x 5.20000022649765, :y 0.006809488941886312} {:x 5.400000229477882, :y 0.0052242803523266925} {:x 5.600000232458115, :y 0.003968217657451668} {:x 5.800000235438347, :y 0.002984156195065413} {:x 6.000000238418579, :y 0.0022217985130712366} {:x 6.200000241398811, :y 0.001637739590084685} {:x 6.400000244379044, :y 0.0011952041116685484} {:x 6.600000247359276, :y 8.635676490676143E-4} {:x 6.800000250339508, :y 6.177428187500568E-4} {:x 7.00000025331974, :y 4.3749802968947785E-4} {:x 7.2000002562999725, :y 3.0676199434329435E-4} {:x 7.400000259280205, :y 2.1295314518666783E-4} {:x 7.600000262260437, :y 1.46360410845439E-4} {:x 7.800000265240669, :y 9.959103279023577E-5} {:x 8.000000268220901, :y 6.709248958927605E-5} {:x 8.200000271201134, :y 4.47491338652843E-5} {:x 8.400000274181366, :y 2.9549655032275533E-5} {:x 8.600000277161598, :y 1.9318671232486508E-5} {:x 8.80000028014183, :y 1.2504293048415349E-5} {:x 9.000000283122063, :y 8.013054374075417E-6} {:x 9.200000286102295, :y 5.083865957875547E-6} {:x 9.400000289082527, :y 3.193354610174078E-6} {:x 9.60000029206276, :y 1.9858994691522355E-6} {:x 9.800000295042992, :y 1.2227127094295497E-6})}), :marks ({:type \"line\", :from {:data \"faec6672-dc35-4c6e-be36-5dba334402cc\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :interpolate {:value \"step-before\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 0.4}, :stroke {:value \"steelblue\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}} {:type \"line\", :from {:data \"47562395-7f2d-4223-a03e-7be897608dde\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"#FF29D2\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}})}}"}
;; <=

;; @@
(compose
  (histogram (repeatedly 10000 #(- (apply + (repeatedly 50 rand)) 25))
             :bins 100
             :normalise :probability-density)
  (histogram (repeatedly 10000 #(- (apply + (repeatedly 50 rand)) 23))
             :bins 100
             :normalise :probability-density
             :colour "green")
  (plot #(gaussian % 2 0) [-10 10])
  (plot #(gaussian % 2 2) [-10 10] :colour "orange"))
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"bottom":20,"top":10,"right":10,"left":50},"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"2d937b55-f67f-4692-bfea-3247cff48f99","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"2d937b55-f67f-4692-bfea-3247cff48f99","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"data":[{"name":"2d937b55-f67f-4692-bfea-3247cff48f99","values":[{"x":-7.2253856492203035,"y":0},{"x":-7.07563338075625,"y":0.0013355390342417947},{"x":-6.925881112292197,"y":0.0},{"x":-6.776128843828144,"y":6.677695171208973E-4},{"x":-6.626376575364091,"y":0.0},{"x":-6.476624306900038,"y":0.0},{"x":-6.326872038435985,"y":6.677695171208973E-4},{"x":-6.177119769971932,"y":0.0013355390342417947},{"x":-6.027367501507879,"y":0.004006617102725384},{"x":-5.877615233043826,"y":0.002003308551362692},{"x":-5.7278629645797725,"y":0.0026710780684835894},{"x":-5.578110696115719,"y":0.003338847585604487},{"x":-5.428358427651666,"y":0.004006617102725384},{"x":-5.278606159187613,"y":0.004006617102725384},{"x":-5.12885389072356,"y":0.01001654275681346},{"x":-4.979101622259507,"y":0.01001654275681346},{"x":-4.829349353795454,"y":0.012019851308176153},{"x":-4.679597085331401,"y":0.01268762082529705},{"x":-4.529844816867348,"y":0.01736200744514333},{"x":-4.380092548403295,"y":0.021368624547868715},{"x":-4.2303402799392416,"y":0.016694237928022434},{"x":-4.0805880114751885,"y":0.020700855030747818},{"x":-3.930835743011135,"y":0.027378550201956793},{"x":-3.7810834745470814,"y":0.032052936821803074},{"x":-3.631331206083028,"y":0.031385167304682174},{"x":-3.4815789376189743,"y":0.03539178440740756},{"x":-3.3318266691549208,"y":0.04407278812997922},{"x":-3.182074400690867,"y":0.06143479557512256},{"x":-3.0323221322268137,"y":0.062102565092243454},{"x":-2.88256986376276,"y":0.06076702605800166},{"x":-2.7328175952987066,"y":0.08146788108874949},{"x":-2.583065326834653,"y":0.08814557625995845},{"x":-2.4333130583705995,"y":0.078129033503145},{"x":-2.283560789906546,"y":0.12220182163312422},{"x":-2.1338085214424924,"y":0.09882988853389281},{"x":-1.984056252978439,"y":0.13021505583857498},{"x":-1.8343039845143854,"y":0.13488944245842127},{"x":-1.6845517160503318,"y":0.13021505583857498},{"x":-1.5347994475862783,"y":0.1275439777700914},{"x":-1.3850471791222247,"y":0.16093245362613626},{"x":-1.2352949106581712,"y":0.14423821569811382},{"x":-1.0855426421941177,"y":0.16226799266037806},{"x":-0.9357903737300641,"y":0.1769589220370378},{"x":-0.7860381052660106,"y":0.18764323431097216},{"x":-0.636285836801957,"y":0.18497215624248858},{"x":-0.4865335683379035,"y":0.19565646851642293},{"x":-0.33678129987384997,"y":0.1742878439685542},{"x":-0.18702903140979646,"y":0.18296884769112587},{"x":-0.03727676294574295,"y":0.19432092948218113},{"x":0.11247550551831056,"y":0.21301847596156626},{"x":0.2622277739823641,"y":0.19632423803354382},{"x":0.4119800424464176,"y":0.18630769527673036},{"x":0.5617323109104712,"y":0.18096553913976318},{"x":0.7114845793745247,"y":0.19031431237945576},{"x":0.8612368478385782,"y":0.1742878439685542},{"x":1.0109891163026317,"y":0.1769589220370378},{"x":1.1607413847666852,"y":0.18096553913976318},{"x":1.3104936532307387,"y":0.16694237928022435},{"x":1.4602459216947923,"y":0.14290267666387205},{"x":1.6099981901588458,"y":0.13422167294130039},{"x":1.7597504586228994,"y":0.1489126023179601},{"x":1.909502727086953,"y":0.13622498149266307},{"x":2.059254995551006,"y":0.11084973984206896},{"x":2.2090072640150598,"y":0.1015009666023764},{"x":2.3587595324791133,"y":0.11352081791055256},{"x":2.508511800943167,"y":0.10016542756813461},{"x":2.6582640694072204,"y":0.08881334577707935},{"x":2.808016337871274,"y":0.08213565060587037},{"x":2.9577686063353275,"y":0.08347118964011217},{"x":3.107520874799381,"y":0.06744472122921064},{"x":3.2572731432634345,"y":0.05742817847239717},{"x":3.407025411727488,"y":0.050082713784067305},{"x":3.5567776801915416,"y":0.05342156136967179},{"x":3.706529948655595,"y":0.04073394054437474},{"x":3.8562822171196487,"y":0.036059553924528455},{"x":4.006034485583702,"y":0.029381858753319483},{"x":4.155786754047755,"y":0.03071739778756128},{"x":4.305539022511808,"y":0.024707472133473202},{"x":4.455291290975861,"y":0.02003308551362692},{"x":4.605043559439914,"y":0.018697546479385128},{"x":4.754795827903967,"y":0.011352081791055256},{"x":4.90454809636802,"y":0.009348773239692564},{"x":5.0543003648320735,"y":0.014690929376659742},{"x":5.2040526332961266,"y":0.006009925654088076},{"x":5.35380490176018,"y":0.006677695171208974},{"x":5.503557170224233,"y":0.009348773239692564},{"x":5.653309438688286,"y":0.007345464688329871},{"x":5.803061707152339,"y":0.005342156136967179},{"x":5.952813975616392,"y":0.0026710780684835894},{"x":6.102566244080445,"y":0.003338847585604487},{"x":6.252318512544498,"y":0.0013355390342417947},{"x":6.402070781008551,"y":0.0},{"x":6.551823049472604,"y":0.0026710780684835894},{"x":6.7015753179366575,"y":6.677695171208973E-4},{"x":6.851327586400711,"y":0.0013355390342417947},{"x":7.001079854864764,"y":6.677695171208973E-4},{"x":7.150832123328817,"y":6.677695171208973E-4},{"x":7.30058439179287,"y":0.0},{"x":7.450336660256923,"y":6.677695171208973E-4},{"x":7.600088928720976,"y":0.0},{"x":7.749841197185029,"y":0.0},{"x":7.899593465649082,"y":6.677695171208973E-4},{"x":8.049345734113135,"y":0}]},{"name":"27215750-f8ba-46a9-8a74-52544d2e6620","values":[{"x":-5.572605651764029,"y":0},{"x":-5.426994521629237,"y":6.867606886055368E-4},{"x":-5.281383391494446,"y":0.0},{"x":-5.135772261359654,"y":0.0},{"x":-4.990161131224863,"y":6.867606886055368E-4},{"x":-4.844550001090071,"y":0.0013735213772110735},{"x":-4.6989388709552795,"y":6.867606886055368E-4},{"x":-4.553327740820488,"y":0.0020602820658166105},{"x":-4.407716610685696,"y":6.867606886055368E-4},{"x":-4.262105480550905,"y":0.0013735213772110735},{"x":-4.116494350416113,"y":6.867606886055368E-4},{"x":-3.970883220281322,"y":0.0020602820658166105},{"x":-3.8252720901465302,"y":0.004807324820238758},{"x":-3.6796609600117387,"y":6.867606886055368E-4},{"x":-3.534049829876947,"y":0.003433803443027684},{"x":-3.3884386997421556,"y":0.003433803443027684},{"x":-3.242827569607364,"y":0.006180846197449831},{"x":-3.0972164394725725,"y":0.005494085508844294},{"x":-2.951605309337781,"y":0.009614649640477516},{"x":-2.8059941792029894,"y":0.009614649640477516},{"x":-2.660383049068198,"y":0.016482256526532884},{"x":-2.5147719189334063,"y":0.010301410329083051},{"x":-2.369160788798615,"y":0.021976342035377176},{"x":-2.2235496586638233,"y":0.019229299280955032},{"x":-2.0779385285290317,"y":0.028843948921432546},{"x":-1.93232739839424,"y":0.037085077184698986},{"x":-1.7867162682594482,"y":0.03639831649609345},{"x":-1.6411051381246564,"y":0.03159099167585469},{"x":-1.4954940079898646,"y":0.04944676957959865},{"x":-1.3498828778550729,"y":0.05288057302262633},{"x":-1.2042717477202811,"y":0.06455550472892045},{"x":-1.0586606175854893,"y":0.054940855088442944},{"x":-0.9130494874506976,"y":0.07073635092637029},{"x":-0.7674383573159058,"y":0.07760395781242566},{"x":-0.621827227181114,"y":0.07623043643521459},{"x":-0.4762160970463223,"y":0.09477297502756407},{"x":-0.3306049669115305,"y":0.09408621433895854},{"x":-0.18499383677673878,"y":0.11400227430851911},{"x":-0.03938270664194704,"y":0.12086988119457448},{"x":0.1062284234928447,"y":0.12499044532620769},{"x":0.25183955362763644,"y":0.12155664188318001},{"x":0.3974506837624282,"y":0.15452115493624577},{"x":0.54306181389722,"y":0.15040059080461254},{"x":0.6886729440320117,"y":0.15795495837927345},{"x":0.8342840741668035,"y":0.1490270694274015},{"x":0.9798952043015953,"y":0.18130482179186172},{"x":1.125506334436387,"y":0.1565814370020624},{"x":1.2711174645711787,"y":0.1881724286779171},{"x":1.4167285947059705,"y":0.19366651418676137},{"x":1.5623397248407622,"y":0.17787101834883404},{"x":1.707950854975554,"y":0.19916059969560568},{"x":1.8535619851103458,"y":0.1922929928095503},{"x":1.9991731152451375,"y":0.1771842576602285},{"x":2.1447842453799293,"y":0.18267834316907278},{"x":2.290395375514721,"y":0.19916059969560568},{"x":2.4360065056495124,"y":0.19160623212094477},{"x":2.581617635784304,"y":0.1737504542172008},{"x":2.7272287659190955,"y":0.19366651418676137},{"x":2.872839896053887,"y":0.15177411218182363},{"x":3.0184510261886786,"y":0.1565814370020624},{"x":3.16406215632347,"y":0.18336510385767832},{"x":3.3096732864582616,"y":0.18199158248046726},{"x":3.455284416593053,"y":0.158641719067879},{"x":3.6008955467278447,"y":0.1696298900855676},{"x":3.7465066768626363,"y":0.12979777014644645},{"x":3.892117806997428,"y":0.13872565909831844},{"x":4.037728937132219,"y":0.12224340257178555},{"x":4.183340067267011,"y":0.12155664188318001},{"x":4.3289511974018025,"y":0.09065241089593086},{"x":4.474562327536594,"y":0.10232734260222498},{"x":4.6201734576713855,"y":0.09614649640477516},{"x":4.765784587806177,"y":0.093399453650353},{"x":4.911395717940969,"y":0.06867606886055368},{"x":5.05700684807576,"y":0.05906141922007616},{"x":5.202617978210552,"y":0.04876000889099311},{"x":5.348229108345343,"y":0.06386874404031492},{"x":5.493840238480135,"y":0.0466997268251765},{"x":5.639451368614926,"y":0.04395268407075435},{"x":5.785062498749718,"y":0.047386487513782036},{"x":5.930673628884509,"y":0.0336512737416713},{"x":6.076284759019301,"y":0.03021747029864362},{"x":6.2218958891540925,"y":0.03021747029864362},{"x":6.367507019288884,"y":0.022663102723982715},{"x":6.513118149423676,"y":0.016482256526532884},{"x":6.658729279558467,"y":0.007554367574660905},{"x":6.804340409693259,"y":0.010988171017688588},{"x":6.94995153982805,"y":0.010301410329083051},{"x":7.095562669962842,"y":0.005494085508844294},{"x":7.241173800097633,"y":0.005494085508844294},{"x":7.386784930232425,"y":0.006867606886055368},{"x":7.532396060367216,"y":0.004120564131633221},{"x":7.678007190502008,"y":0.005494085508844294},{"x":7.8236183206367995,"y":0.0020602820658166105},{"x":7.969229450771591,"y":0.002747042754422147},{"x":8.114840580906383,"y":0.0013735213772110735},{"x":8.260451711041176,"y":6.867606886055368E-4},{"x":8.406062841175968,"y":6.867606886055368E-4},{"x":8.55167397131076,"y":6.867606886055368E-4},{"x":8.697285101445553,"y":0.0013735213772110735},{"x":8.842896231580346,"y":0.0013735213772110735},{"x":8.988507361715138,"y":0.0},{"x":9.13411849184993,"y":6.867606886055368E-4},{"x":9.279729621984723,"y":0}]},{"name":"98361f49-97b3-4a0f-b7eb-e8d0a5fe0da7","values":[{"x":-10,"y":7.453306344157342E-7},{"x":-9.799999997019768,"y":1.2227136022020082E-6},{"x":-9.599999994039536,"y":1.9859008895787535E-6},{"x":-9.399999991059303,"y":3.1933568466554097E-6},{"x":-9.199999988079071,"y":5.083869442630077E-6},{"x":-8.999999985098839,"y":8.013059747248974E-6},{"x":-8.799999982118607,"y":1.2504301246871565E-5},{"x":-8.599999979138374,"y":1.9318683610927946E-5},{"x":-8.399999976158142,"y":2.9549673525896843E-5},{"x":-8.19999997317791,"y":4.474916120466938E-5},{"x":-7.999999970197678,"y":6.709252957952864E-5},{"x":-7.799999967217445,"y":9.959109066711276E-5},{"x":-7.599999964237213,"y":1.463604937211866E-4},{"x":-7.399999961256981,"y":2.129532625969202E-4},{"x":-7.199999958276749,"y":3.0676215890329826E-4},{"x":-6.999999955296516,"y":4.3749825786254407E-4},{"x":-6.799999952316284,"y":6.177431317230414E-4},{"x":-6.599999949336052,"y":8.635680737170301E-4},{"x":-6.39999994635582,"y":0.0011952046815864247},{"x":-6.1999999433755875,"y":0.0016377403466157404},{"x":-5.999999940395355,"y":0.002221799506292808},{"x":-5.799999937415123,"y":0.002984157484620092},{"x":-5.599999934434891,"y":0.003968219313121465},{"x":-5.3999999314546585,"y":0.005224282454218924},{"x":-5.199999928474426,"y":0.006809491580088464},{"x":-4.999999925494194,"y":0.008787387543070678},{"x":-4.799999922513962,"y":0.011226953610745277},{"x":-4.5999999195337296,"y":0.014201072062040727},{"x":-4.399999916553497,"y":0.017784325124320887},{"x":-4.199999913573265,"y":0.022050107061901632},{"x":-3.999999910593033,"y":0.027067059067306068},{"x":-3.7999999076128006,"y":0.032894894202544614},{"x":-3.5999999046325684,"y":0.03957974321387936},{"x":-3.399999901652336,"y":0.04714921925263572},{"x":-3.199999898672104,"y":0.0556074645983085},{"x":-2.9999998956918716,"y":0.06493049855125375},{"x":-2.7999998927116394,"y":0.07506222540759178},{"x":-2.599999889731407,"y":0.08591147779981703},{"x":-2.399999886751175,"y":0.09735045780688895},{"x":-2.1999998837709427,"y":0.10921489230961066},{"x":-1.9999998807907104,"y":0.12130613917293559},{"x":-1.7999998778104782,"y":0.13339536950647682},{"x":-1.599999874830246,"y":0.1452298146860898},{"x":-1.3999998718500137,"y":0.1565409146696238},{"x":-1.1999998688697815,"y":0.1670540488540041},{"x":-0.9999998658895493,"y":0.17649938643452168},{"x":-0.799999862909317,"y":0.18462327433935283},{"x":-0.5999998599290848,"y":0.1911995003838428},{"x":-0.39999985694885254,"y":0.19603973746572148},{"x":-0.1999998539686203,"y":0.19900249729156638},{"x":1.4901161193847656E-7,"y":0.19999999999999946},{"x":0.20000015199184418,"y":0.1990024943261981},{"x":0.4000001549720764,"y":0.19603973162328203},{"x":0.6000001579523087,"y":0.19119949183655915},{"x":0.8000001609325409,"y":0.1846232633349483},{"x":1.0000001639127731,"y":0.1764993732842929},{"x":1.2000001668930054,"y":0.1670540339182087},{"x":1.4000001698732376,"y":0.15654089834113458},{"x":1.6000001728534698,"y":0.14522979737334754},{"x":1.800000175833702,"y":0.1333953516167646},{"x":2.0000001788139343,"y":0.12130612109691333},{"x":2.2000001817941666,"y":0.109214874407896},{"x":2.400000184774399,"y":0.09735044039927186},{"x":2.600000187754631,"y":0.08591146115746827},{"x":2.8000001907348633,"y":0.07506220974839269},{"x":3.0000001937150955,"y":0.06493048403815777},{"x":3.2000001966953278,"y":0.055607451340457185},{"x":3.40000019967556,"y":0.04714920730880908},{"x":3.6000002026557922,"y":0.03957973259776622},{"x":3.8000002056360245,"y":0.03289488488927549},{"x":4.000000208616257,"y":0.02706705100069495},{"x":4.200000211596489,"y":0.02205010016188641},{"x":4.400000214576721,"y":0.017784319294165653},{"x":4.600000217556953,"y":0.014201067194954814},{"x":4.800000220537186,"y":0.011226949595674444},{"x":5.000000223517418,"y":0.008787384269514283},{"x":5.20000022649765,"y":0.006809488941886312},{"x":5.400000229477882,"y":0.0052242803523266925},{"x":5.600000232458115,"y":0.003968217657451668},{"x":5.800000235438347,"y":0.002984156195065413},{"x":6.000000238418579,"y":0.0022217985130712366},{"x":6.200000241398811,"y":0.001637739590084685},{"x":6.400000244379044,"y":0.0011952041116685484},{"x":6.600000247359276,"y":8.635676490676143E-4},{"x":6.800000250339508,"y":6.177428187500568E-4},{"x":7.00000025331974,"y":4.3749802968947785E-4},{"x":7.2000002562999725,"y":3.0676199434329435E-4},{"x":7.400000259280205,"y":2.1295314518666783E-4},{"x":7.600000262260437,"y":1.46360410845439E-4},{"x":7.800000265240669,"y":9.959103279023577E-5},{"x":8.000000268220901,"y":6.709248958927605E-5},{"x":8.200000271201134,"y":4.47491338652843E-5},{"x":8.400000274181366,"y":2.9549655032275533E-5},{"x":8.600000277161598,"y":1.9318671232486508E-5},{"x":8.80000028014183,"y":1.2504293048415349E-5},{"x":9.000000283122063,"y":8.013054374075417E-6},{"x":9.200000286102295,"y":5.083865957875547E-6},{"x":9.400000289082527,"y":3.193354610174078E-6},{"x":9.60000029206276,"y":1.9858994691522355E-6},{"x":9.800000295042992,"y":1.2227127094295497E-6}]},{"name":"d625cdcf-c0f7-46b2-9f39-b7d7d63a8056","values":[{"x":-10,"y":3.045995948942526E-9},{"x":-9.799999997019768,"y":5.522484962208031E-9},{"x":-9.599999994039536,"y":9.912810809691382E-9},{"x":-9.399999991059303,"y":1.7616358841803305E-8},{"x":-9.199999988079071,"y":3.099506374862979E-8},{"x":-8.999999985098839,"y":5.399157227973733E-8},{"x":-8.799999982118607,"y":9.31143188111988E-8},{"x":-8.599999979138374,"y":1.5898788109635867E-7},{"x":-8.399999976158142,"y":2.68762472186575E-7},{"x":-8.19999997317791,"y":4.4981122417190456E-7},{"x":-7.999999970197678,"y":7.45330689947196E-7},{"x":-7.799999967217445,"y":1.2227136914792885E-6},{"x":-7.599999964237213,"y":1.9859010316214617E-6},{"x":-7.399999961256981,"y":3.1933570703036276E-6},{"x":-7.199999958276749,"y":5.083869791105649E-6},{"x":-6.999999955296516,"y":8.01306028456652E-6},{"x":-6.799999952316284,"y":1.2504302066717465E-5},{"x":-6.599999949336052,"y":1.931868484877249E-5},{"x":-6.39999994635582,"y":2.9549675375259554E-5},{"x":-6.1999999433755875,"y":4.474916393860878E-5},{"x":-5.999999940395355,"y":6.709253357855516E-5},{"x":-5.799999937415123,"y":9.959109645480218E-5},{"x":-5.599999934434891,"y":1.463605020087637E-4},{"x":-5.3999999314546585,"y":2.1295327433794878E-4},{"x":-5.199999928474426,"y":3.067621753593031E-4},{"x":-4.999999925494194,"y":4.3749828067985645E-4},{"x":-4.799999922513962,"y":6.177431630203478E-4},{"x":-4.5999999195337296,"y":8.635681161819818E-4},{"x":-4.399999916553497,"y":0.0011952047385782264},{"x":-4.199999913573265,"y":0.001637740422268863},{"x":-3.999999910593033,"y":0.002221799605614986},{"x":-3.7999999076128006,"y":0.002984157613575588},{"x":-3.5999999046325684,"y":0.003968219478688477},{"x":-3.399999901652336,"y":0.0052242826644081876},{"x":-3.199999898672104,"y":0.0068094918439087285},{"x":-2.9999998956918716,"y":0.008787387870426376},{"x":-2.7999998927116394,"y":0.011226954012252426},{"x":-2.599999889731407,"y":0.014201072548749392},{"x":-2.399999886751175,"y":0.017784325707336495},{"x":-2.1999998837709427,"y":0.02205010775190324},{"x":-1.9999998807907104,"y":0.027067059873967278},{"x":-1.7999998778104782,"y":0.03289489513387163},{"x":-1.599999874830246,"y":0.03957974427549078},{"x":-1.3999998718500137,"y":0.0471492204470185},{"x":-1.1999998688697815,"y":0.05560746592409373},{"x":-0.9999998658895493,"y":0.06493050000256344},{"x":-0.799999862909317,"y":0.07506222697351177},{"x":-0.5999998599290848,"y":0.08591147946405198},{"x":-0.39999985694885254,"y":0.0973504595476507},{"x":-0.1999998539686203,"y":0.10921489409978216},{"x":1.4901161193847656E-7,"y":0.12130614098053782},{"x":0.20000015199184418,"y":0.133395371295448},{"x":0.4000001549720764,"y":0.14522981641736396},{"x":0.6000001579523087,"y":0.1565409163024726},{"x":0.8000001609325409,"y":0.16705405034758353},{"x":1.0000001639127731,"y":0.17649938774954438},{"x":1.2000001668930054,"y":0.18462327543979307},{"x":1.4000001698732376,"y":0.191199501238571},{"x":1.6000001728534698,"y":0.19603973804996522},{"x":1.800000175833702,"y":0.19900249758810298},{"x":2.0000001788139343,"y":0.1999999999999992},{"x":2.2000001817941666,"y":0.19900249402966103},{"x":2.400000184774399,"y":0.19603973103903785},{"x":2.600000187754631,"y":0.1911994909818306},{"x":2.8000001907348633,"y":0.18462326223450765},{"x":3.0000001937150955,"y":0.1764993719692699},{"x":3.2000001966953278,"y":0.16705403242462902},{"x":3.40000019967556,"y":0.15654089670828555},{"x":3.6000002026557922,"y":0.14522979564207325},{"x":3.8000002056360245,"y":0.13339534982779339},{"x":4.000000208616257,"y":0.1213061192893111},{"x":4.200000211596489,"y":0.10921487261772456},{"x":4.400000214576721,"y":0.09735043865851022},{"x":4.600000217556953,"y":0.08591145949323346},{"x":4.800000220537186,"y":0.07506220818247286},{"x":5.000000223517418,"y":0.06493048258684828},{"x":5.20000022649765,"y":0.055607450014672155},{"x":5.400000229477882,"y":0.04714920611442652},{"x":5.600000232458115,"y":0.039579731536155016},{"x":5.800000235438347,"y":0.032894883957948685},{"x":6.000000238418579,"y":0.027067050194033934},{"x":6.200000241398811,"y":0.02205009947188498},{"x":6.400000244379044,"y":0.017784318711150215},{"x":6.600000247359276,"y":0.0142010667082463},{"x":6.800000250339508,"y":0.011226949194167425},{"x":7.00000025331974,"y":0.0087873839421587},{"x":7.2000002562999725,"y":0.0068094886780661465},{"x":7.400000259280205,"y":0.005224280142137508},{"x":7.600000262260437,"y":0.003968217491884721},{"x":7.800000265240669,"y":0.0029841560661099708},{"x":8.000000268220901,"y":0.0022217984137491015},{"x":8.200000271201134,"y":0.0016377395144315963},{"x":8.400000274181366,"y":0.0011952040546767745},{"x":8.600000277161598,"y":8.635676066026832E-4},{"x":8.80000028014183,"y":6.177427874527661E-4},{"x":9.000000283122063,"y":4.374980068721774E-4},{"x":9.200000286102295,"y":3.067619778872984E-4},{"x":9.400000289082527,"y":2.1295313344564596E-4},{"x":9.60000029206276,"y":1.4636040255786668E-4},{"x":9.800000295042992,"y":9.959102700254978E-5}]}],"marks":[{"type":"line","from":{"data":"2d937b55-f67f-4692-bfea-3247cff48f99"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"interpolate":{"value":"step-before"},"fill":{"value":"steelblue"},"fillOpacity":{"value":0.4},"stroke":{"value":"steelblue"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}},{"type":"line","from":{"data":"27215750-f8ba-46a9-8a74-52544d2e6620"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"interpolate":{"value":"step-before"},"fill":{"value":"green"},"fillOpacity":{"value":0.4},"stroke":{"value":"green"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}},{"type":"line","from":{"data":"98361f49-97b3-4a0f-b7eb-e8d0a5fe0da7"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"#FF29D2"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}},{"type":"line","from":{"data":"d625cdcf-c0f7-46b2-9f39-b7d7d63a8056"},"properties":{"enter":{"x":{"field":"data.x","scale":"x"},"y":{"field":"data.y","scale":"y"},"stroke":{"value":"orange"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:bottom 20, :top 10, :right 10, :left 50}, :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"2d937b55-f67f-4692-bfea-3247cff48f99\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"2d937b55-f67f-4692-bfea-3247cff48f99\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :data ({:name \"2d937b55-f67f-4692-bfea-3247cff48f99\", :values ({:x -7.2253856492203035, :y 0} {:x -7.07563338075625, :y 0.0013355390342417947} {:x -6.925881112292197, :y 0.0} {:x -6.776128843828144, :y 6.677695171208973E-4} {:x -6.626376575364091, :y 0.0} {:x -6.476624306900038, :y 0.0} {:x -6.326872038435985, :y 6.677695171208973E-4} {:x -6.177119769971932, :y 0.0013355390342417947} {:x -6.027367501507879, :y 0.004006617102725384} {:x -5.877615233043826, :y 0.002003308551362692} {:x -5.7278629645797725, :y 0.0026710780684835894} {:x -5.578110696115719, :y 0.003338847585604487} {:x -5.428358427651666, :y 0.004006617102725384} {:x -5.278606159187613, :y 0.004006617102725384} {:x -5.12885389072356, :y 0.01001654275681346} {:x -4.979101622259507, :y 0.01001654275681346} {:x -4.829349353795454, :y 0.012019851308176153} {:x -4.679597085331401, :y 0.01268762082529705} {:x -4.529844816867348, :y 0.01736200744514333} {:x -4.380092548403295, :y 0.021368624547868715} {:x -4.2303402799392416, :y 0.016694237928022434} {:x -4.0805880114751885, :y 0.020700855030747818} {:x -3.930835743011135, :y 0.027378550201956793} {:x -3.7810834745470814, :y 0.032052936821803074} {:x -3.631331206083028, :y 0.031385167304682174} {:x -3.4815789376189743, :y 0.03539178440740756} {:x -3.3318266691549208, :y 0.04407278812997922} {:x -3.182074400690867, :y 0.06143479557512256} {:x -3.0323221322268137, :y 0.062102565092243454} {:x -2.88256986376276, :y 0.06076702605800166} {:x -2.7328175952987066, :y 0.08146788108874949} {:x -2.583065326834653, :y 0.08814557625995845} {:x -2.4333130583705995, :y 0.078129033503145} {:x -2.283560789906546, :y 0.12220182163312422} {:x -2.1338085214424924, :y 0.09882988853389281} {:x -1.984056252978439, :y 0.13021505583857498} {:x -1.8343039845143854, :y 0.13488944245842127} {:x -1.6845517160503318, :y 0.13021505583857498} {:x -1.5347994475862783, :y 0.1275439777700914} {:x -1.3850471791222247, :y 0.16093245362613626} {:x -1.2352949106581712, :y 0.14423821569811382} {:x -1.0855426421941177, :y 0.16226799266037806} {:x -0.9357903737300641, :y 0.1769589220370378} {:x -0.7860381052660106, :y 0.18764323431097216} {:x -0.636285836801957, :y 0.18497215624248858} {:x -0.4865335683379035, :y 0.19565646851642293} {:x -0.33678129987384997, :y 0.1742878439685542} {:x -0.18702903140979646, :y 0.18296884769112587} {:x -0.03727676294574295, :y 0.19432092948218113} {:x 0.11247550551831056, :y 0.21301847596156626} {:x 0.2622277739823641, :y 0.19632423803354382} {:x 0.4119800424464176, :y 0.18630769527673036} {:x 0.5617323109104712, :y 0.18096553913976318} {:x 0.7114845793745247, :y 0.19031431237945576} {:x 0.8612368478385782, :y 0.1742878439685542} {:x 1.0109891163026317, :y 0.1769589220370378} {:x 1.1607413847666852, :y 0.18096553913976318} {:x 1.3104936532307387, :y 0.16694237928022435} {:x 1.4602459216947923, :y 0.14290267666387205} {:x 1.6099981901588458, :y 0.13422167294130039} {:x 1.7597504586228994, :y 0.1489126023179601} {:x 1.909502727086953, :y 0.13622498149266307} {:x 2.059254995551006, :y 0.11084973984206896} {:x 2.2090072640150598, :y 0.1015009666023764} {:x 2.3587595324791133, :y 0.11352081791055256} {:x 2.508511800943167, :y 0.10016542756813461} {:x 2.6582640694072204, :y 0.08881334577707935} {:x 2.808016337871274, :y 0.08213565060587037} {:x 2.9577686063353275, :y 0.08347118964011217} {:x 3.107520874799381, :y 0.06744472122921064} {:x 3.2572731432634345, :y 0.05742817847239717} {:x 3.407025411727488, :y 0.050082713784067305} {:x 3.5567776801915416, :y 0.05342156136967179} {:x 3.706529948655595, :y 0.04073394054437474} {:x 3.8562822171196487, :y 0.036059553924528455} {:x 4.006034485583702, :y 0.029381858753319483} {:x 4.155786754047755, :y 0.03071739778756128} {:x 4.305539022511808, :y 0.024707472133473202} {:x 4.455291290975861, :y 0.02003308551362692} {:x 4.605043559439914, :y 0.018697546479385128} {:x 4.754795827903967, :y 0.011352081791055256} {:x 4.90454809636802, :y 0.009348773239692564} {:x 5.0543003648320735, :y 0.014690929376659742} {:x 5.2040526332961266, :y 0.006009925654088076} {:x 5.35380490176018, :y 0.006677695171208974} {:x 5.503557170224233, :y 0.009348773239692564} {:x 5.653309438688286, :y 0.007345464688329871} {:x 5.803061707152339, :y 0.005342156136967179} {:x 5.952813975616392, :y 0.0026710780684835894} {:x 6.102566244080445, :y 0.003338847585604487} {:x 6.252318512544498, :y 0.0013355390342417947} {:x 6.402070781008551, :y 0.0} {:x 6.551823049472604, :y 0.0026710780684835894} {:x 6.7015753179366575, :y 6.677695171208973E-4} {:x 6.851327586400711, :y 0.0013355390342417947} {:x 7.001079854864764, :y 6.677695171208973E-4} {:x 7.150832123328817, :y 6.677695171208973E-4} {:x 7.30058439179287, :y 0.0} {:x 7.450336660256923, :y 6.677695171208973E-4} {:x 7.600088928720976, :y 0.0} {:x 7.749841197185029, :y 0.0} {:x 7.899593465649082, :y 6.677695171208973E-4} {:x 8.049345734113135, :y 0})} {:name \"27215750-f8ba-46a9-8a74-52544d2e6620\", :values ({:x -5.572605651764029, :y 0} {:x -5.426994521629237, :y 6.867606886055368E-4} {:x -5.281383391494446, :y 0.0} {:x -5.135772261359654, :y 0.0} {:x -4.990161131224863, :y 6.867606886055368E-4} {:x -4.844550001090071, :y 0.0013735213772110735} {:x -4.6989388709552795, :y 6.867606886055368E-4} {:x -4.553327740820488, :y 0.0020602820658166105} {:x -4.407716610685696, :y 6.867606886055368E-4} {:x -4.262105480550905, :y 0.0013735213772110735} {:x -4.116494350416113, :y 6.867606886055368E-4} {:x -3.970883220281322, :y 0.0020602820658166105} {:x -3.8252720901465302, :y 0.004807324820238758} {:x -3.6796609600117387, :y 6.867606886055368E-4} {:x -3.534049829876947, :y 0.003433803443027684} {:x -3.3884386997421556, :y 0.003433803443027684} {:x -3.242827569607364, :y 0.006180846197449831} {:x -3.0972164394725725, :y 0.005494085508844294} {:x -2.951605309337781, :y 0.009614649640477516} {:x -2.8059941792029894, :y 0.009614649640477516} {:x -2.660383049068198, :y 0.016482256526532884} {:x -2.5147719189334063, :y 0.010301410329083051} {:x -2.369160788798615, :y 0.021976342035377176} {:x -2.2235496586638233, :y 0.019229299280955032} {:x -2.0779385285290317, :y 0.028843948921432546} {:x -1.93232739839424, :y 0.037085077184698986} {:x -1.7867162682594482, :y 0.03639831649609345} {:x -1.6411051381246564, :y 0.03159099167585469} {:x -1.4954940079898646, :y 0.04944676957959865} {:x -1.3498828778550729, :y 0.05288057302262633} {:x -1.2042717477202811, :y 0.06455550472892045} {:x -1.0586606175854893, :y 0.054940855088442944} {:x -0.9130494874506976, :y 0.07073635092637029} {:x -0.7674383573159058, :y 0.07760395781242566} {:x -0.621827227181114, :y 0.07623043643521459} {:x -0.4762160970463223, :y 0.09477297502756407} {:x -0.3306049669115305, :y 0.09408621433895854} {:x -0.18499383677673878, :y 0.11400227430851911} {:x -0.03938270664194704, :y 0.12086988119457448} {:x 0.1062284234928447, :y 0.12499044532620769} {:x 0.25183955362763644, :y 0.12155664188318001} {:x 0.3974506837624282, :y 0.15452115493624577} {:x 0.54306181389722, :y 0.15040059080461254} {:x 0.6886729440320117, :y 0.15795495837927345} {:x 0.8342840741668035, :y 0.1490270694274015} {:x 0.9798952043015953, :y 0.18130482179186172} {:x 1.125506334436387, :y 0.1565814370020624} {:x 1.2711174645711787, :y 0.1881724286779171} {:x 1.4167285947059705, :y 0.19366651418676137} {:x 1.5623397248407622, :y 0.17787101834883404} {:x 1.707950854975554, :y 0.19916059969560568} {:x 1.8535619851103458, :y 0.1922929928095503} {:x 1.9991731152451375, :y 0.1771842576602285} {:x 2.1447842453799293, :y 0.18267834316907278} {:x 2.290395375514721, :y 0.19916059969560568} {:x 2.4360065056495124, :y 0.19160623212094477} {:x 2.581617635784304, :y 0.1737504542172008} {:x 2.7272287659190955, :y 0.19366651418676137} {:x 2.872839896053887, :y 0.15177411218182363} {:x 3.0184510261886786, :y 0.1565814370020624} {:x 3.16406215632347, :y 0.18336510385767832} {:x 3.3096732864582616, :y 0.18199158248046726} {:x 3.455284416593053, :y 0.158641719067879} {:x 3.6008955467278447, :y 0.1696298900855676} {:x 3.7465066768626363, :y 0.12979777014644645} {:x 3.892117806997428, :y 0.13872565909831844} {:x 4.037728937132219, :y 0.12224340257178555} {:x 4.183340067267011, :y 0.12155664188318001} {:x 4.3289511974018025, :y 0.09065241089593086} {:x 4.474562327536594, :y 0.10232734260222498} {:x 4.6201734576713855, :y 0.09614649640477516} {:x 4.765784587806177, :y 0.093399453650353} {:x 4.911395717940969, :y 0.06867606886055368} {:x 5.05700684807576, :y 0.05906141922007616} {:x 5.202617978210552, :y 0.04876000889099311} {:x 5.348229108345343, :y 0.06386874404031492} {:x 5.493840238480135, :y 0.0466997268251765} {:x 5.639451368614926, :y 0.04395268407075435} {:x 5.785062498749718, :y 0.047386487513782036} {:x 5.930673628884509, :y 0.0336512737416713} {:x 6.076284759019301, :y 0.03021747029864362} {:x 6.2218958891540925, :y 0.03021747029864362} {:x 6.367507019288884, :y 0.022663102723982715} {:x 6.513118149423676, :y 0.016482256526532884} {:x 6.658729279558467, :y 0.007554367574660905} {:x 6.804340409693259, :y 0.010988171017688588} {:x 6.94995153982805, :y 0.010301410329083051} {:x 7.095562669962842, :y 0.005494085508844294} {:x 7.241173800097633, :y 0.005494085508844294} {:x 7.386784930232425, :y 0.006867606886055368} {:x 7.532396060367216, :y 0.004120564131633221} {:x 7.678007190502008, :y 0.005494085508844294} {:x 7.8236183206367995, :y 0.0020602820658166105} {:x 7.969229450771591, :y 0.002747042754422147} {:x 8.114840580906383, :y 0.0013735213772110735} {:x 8.260451711041176, :y 6.867606886055368E-4} {:x 8.406062841175968, :y 6.867606886055368E-4} {:x 8.55167397131076, :y 6.867606886055368E-4} {:x 8.697285101445553, :y 0.0013735213772110735} {:x 8.842896231580346, :y 0.0013735213772110735} {:x 8.988507361715138, :y 0.0} {:x 9.13411849184993, :y 6.867606886055368E-4} {:x 9.279729621984723, :y 0})} {:name \"98361f49-97b3-4a0f-b7eb-e8d0a5fe0da7\", :values ({:x -10, :y 7.453306344157342E-7} {:x -9.799999997019768, :y 1.2227136022020082E-6} {:x -9.599999994039536, :y 1.9859008895787535E-6} {:x -9.399999991059303, :y 3.1933568466554097E-6} {:x -9.199999988079071, :y 5.083869442630077E-6} {:x -8.999999985098839, :y 8.013059747248974E-6} {:x -8.799999982118607, :y 1.2504301246871565E-5} {:x -8.599999979138374, :y 1.9318683610927946E-5} {:x -8.399999976158142, :y 2.9549673525896843E-5} {:x -8.19999997317791, :y 4.474916120466938E-5} {:x -7.999999970197678, :y 6.709252957952864E-5} {:x -7.799999967217445, :y 9.959109066711276E-5} {:x -7.599999964237213, :y 1.463604937211866E-4} {:x -7.399999961256981, :y 2.129532625969202E-4} {:x -7.199999958276749, :y 3.0676215890329826E-4} {:x -6.999999955296516, :y 4.3749825786254407E-4} {:x -6.799999952316284, :y 6.177431317230414E-4} {:x -6.599999949336052, :y 8.635680737170301E-4} {:x -6.39999994635582, :y 0.0011952046815864247} {:x -6.1999999433755875, :y 0.0016377403466157404} {:x -5.999999940395355, :y 0.002221799506292808} {:x -5.799999937415123, :y 0.002984157484620092} {:x -5.599999934434891, :y 0.003968219313121465} {:x -5.3999999314546585, :y 0.005224282454218924} {:x -5.199999928474426, :y 0.006809491580088464} {:x -4.999999925494194, :y 0.008787387543070678} {:x -4.799999922513962, :y 0.011226953610745277} {:x -4.5999999195337296, :y 0.014201072062040727} {:x -4.399999916553497, :y 0.017784325124320887} {:x -4.199999913573265, :y 0.022050107061901632} {:x -3.999999910593033, :y 0.027067059067306068} {:x -3.7999999076128006, :y 0.032894894202544614} {:x -3.5999999046325684, :y 0.03957974321387936} {:x -3.399999901652336, :y 0.04714921925263572} {:x -3.199999898672104, :y 0.0556074645983085} {:x -2.9999998956918716, :y 0.06493049855125375} {:x -2.7999998927116394, :y 0.07506222540759178} {:x -2.599999889731407, :y 0.08591147779981703} {:x -2.399999886751175, :y 0.09735045780688895} {:x -2.1999998837709427, :y 0.10921489230961066} {:x -1.9999998807907104, :y 0.12130613917293559} {:x -1.7999998778104782, :y 0.13339536950647682} {:x -1.599999874830246, :y 0.1452298146860898} {:x -1.3999998718500137, :y 0.1565409146696238} {:x -1.1999998688697815, :y 0.1670540488540041} {:x -0.9999998658895493, :y 0.17649938643452168} {:x -0.799999862909317, :y 0.18462327433935283} {:x -0.5999998599290848, :y 0.1911995003838428} {:x -0.39999985694885254, :y 0.19603973746572148} {:x -0.1999998539686203, :y 0.19900249729156638} {:x 1.4901161193847656E-7, :y 0.19999999999999946} {:x 0.20000015199184418, :y 0.1990024943261981} {:x 0.4000001549720764, :y 0.19603973162328203} {:x 0.6000001579523087, :y 0.19119949183655915} {:x 0.8000001609325409, :y 0.1846232633349483} {:x 1.0000001639127731, :y 0.1764993732842929} {:x 1.2000001668930054, :y 0.1670540339182087} {:x 1.4000001698732376, :y 0.15654089834113458} {:x 1.6000001728534698, :y 0.14522979737334754} {:x 1.800000175833702, :y 0.1333953516167646} {:x 2.0000001788139343, :y 0.12130612109691333} {:x 2.2000001817941666, :y 0.109214874407896} {:x 2.400000184774399, :y 0.09735044039927186} {:x 2.600000187754631, :y 0.08591146115746827} {:x 2.8000001907348633, :y 0.07506220974839269} {:x 3.0000001937150955, :y 0.06493048403815777} {:x 3.2000001966953278, :y 0.055607451340457185} {:x 3.40000019967556, :y 0.04714920730880908} {:x 3.6000002026557922, :y 0.03957973259776622} {:x 3.8000002056360245, :y 0.03289488488927549} {:x 4.000000208616257, :y 0.02706705100069495} {:x 4.200000211596489, :y 0.02205010016188641} {:x 4.400000214576721, :y 0.017784319294165653} {:x 4.600000217556953, :y 0.014201067194954814} {:x 4.800000220537186, :y 0.011226949595674444} {:x 5.000000223517418, :y 0.008787384269514283} {:x 5.20000022649765, :y 0.006809488941886312} {:x 5.400000229477882, :y 0.0052242803523266925} {:x 5.600000232458115, :y 0.003968217657451668} {:x 5.800000235438347, :y 0.002984156195065413} {:x 6.000000238418579, :y 0.0022217985130712366} {:x 6.200000241398811, :y 0.001637739590084685} {:x 6.400000244379044, :y 0.0011952041116685484} {:x 6.600000247359276, :y 8.635676490676143E-4} {:x 6.800000250339508, :y 6.177428187500568E-4} {:x 7.00000025331974, :y 4.3749802968947785E-4} {:x 7.2000002562999725, :y 3.0676199434329435E-4} {:x 7.400000259280205, :y 2.1295314518666783E-4} {:x 7.600000262260437, :y 1.46360410845439E-4} {:x 7.800000265240669, :y 9.959103279023577E-5} {:x 8.000000268220901, :y 6.709248958927605E-5} {:x 8.200000271201134, :y 4.47491338652843E-5} {:x 8.400000274181366, :y 2.9549655032275533E-5} {:x 8.600000277161598, :y 1.9318671232486508E-5} {:x 8.80000028014183, :y 1.2504293048415349E-5} {:x 9.000000283122063, :y 8.013054374075417E-6} {:x 9.200000286102295, :y 5.083865957875547E-6} {:x 9.400000289082527, :y 3.193354610174078E-6} {:x 9.60000029206276, :y 1.9858994691522355E-6} {:x 9.800000295042992, :y 1.2227127094295497E-6})} {:name \"d625cdcf-c0f7-46b2-9f39-b7d7d63a8056\", :values ({:x -10, :y 3.045995948942526E-9} {:x -9.799999997019768, :y 5.522484962208031E-9} {:x -9.599999994039536, :y 9.912810809691382E-9} {:x -9.399999991059303, :y 1.7616358841803305E-8} {:x -9.199999988079071, :y 3.099506374862979E-8} {:x -8.999999985098839, :y 5.399157227973733E-8} {:x -8.799999982118607, :y 9.31143188111988E-8} {:x -8.599999979138374, :y 1.5898788109635867E-7} {:x -8.399999976158142, :y 2.68762472186575E-7} {:x -8.19999997317791, :y 4.4981122417190456E-7} {:x -7.999999970197678, :y 7.45330689947196E-7} {:x -7.799999967217445, :y 1.2227136914792885E-6} {:x -7.599999964237213, :y 1.9859010316214617E-6} {:x -7.399999961256981, :y 3.1933570703036276E-6} {:x -7.199999958276749, :y 5.083869791105649E-6} {:x -6.999999955296516, :y 8.01306028456652E-6} {:x -6.799999952316284, :y 1.2504302066717465E-5} {:x -6.599999949336052, :y 1.931868484877249E-5} {:x -6.39999994635582, :y 2.9549675375259554E-5} {:x -6.1999999433755875, :y 4.474916393860878E-5} {:x -5.999999940395355, :y 6.709253357855516E-5} {:x -5.799999937415123, :y 9.959109645480218E-5} {:x -5.599999934434891, :y 1.463605020087637E-4} {:x -5.3999999314546585, :y 2.1295327433794878E-4} {:x -5.199999928474426, :y 3.067621753593031E-4} {:x -4.999999925494194, :y 4.3749828067985645E-4} {:x -4.799999922513962, :y 6.177431630203478E-4} {:x -4.5999999195337296, :y 8.635681161819818E-4} {:x -4.399999916553497, :y 0.0011952047385782264} {:x -4.199999913573265, :y 0.001637740422268863} {:x -3.999999910593033, :y 0.002221799605614986} {:x -3.7999999076128006, :y 0.002984157613575588} {:x -3.5999999046325684, :y 0.003968219478688477} {:x -3.399999901652336, :y 0.0052242826644081876} {:x -3.199999898672104, :y 0.0068094918439087285} {:x -2.9999998956918716, :y 0.008787387870426376} {:x -2.7999998927116394, :y 0.011226954012252426} {:x -2.599999889731407, :y 0.014201072548749392} {:x -2.399999886751175, :y 0.017784325707336495} {:x -2.1999998837709427, :y 0.02205010775190324} {:x -1.9999998807907104, :y 0.027067059873967278} {:x -1.7999998778104782, :y 0.03289489513387163} {:x -1.599999874830246, :y 0.03957974427549078} {:x -1.3999998718500137, :y 0.0471492204470185} {:x -1.1999998688697815, :y 0.05560746592409373} {:x -0.9999998658895493, :y 0.06493050000256344} {:x -0.799999862909317, :y 0.07506222697351177} {:x -0.5999998599290848, :y 0.08591147946405198} {:x -0.39999985694885254, :y 0.0973504595476507} {:x -0.1999998539686203, :y 0.10921489409978216} {:x 1.4901161193847656E-7, :y 0.12130614098053782} {:x 0.20000015199184418, :y 0.133395371295448} {:x 0.4000001549720764, :y 0.14522981641736396} {:x 0.6000001579523087, :y 0.1565409163024726} {:x 0.8000001609325409, :y 0.16705405034758353} {:x 1.0000001639127731, :y 0.17649938774954438} {:x 1.2000001668930054, :y 0.18462327543979307} {:x 1.4000001698732376, :y 0.191199501238571} {:x 1.6000001728534698, :y 0.19603973804996522} {:x 1.800000175833702, :y 0.19900249758810298} {:x 2.0000001788139343, :y 0.1999999999999992} {:x 2.2000001817941666, :y 0.19900249402966103} {:x 2.400000184774399, :y 0.19603973103903785} {:x 2.600000187754631, :y 0.1911994909818306} {:x 2.8000001907348633, :y 0.18462326223450765} {:x 3.0000001937150955, :y 0.1764993719692699} {:x 3.2000001966953278, :y 0.16705403242462902} {:x 3.40000019967556, :y 0.15654089670828555} {:x 3.6000002026557922, :y 0.14522979564207325} {:x 3.8000002056360245, :y 0.13339534982779339} {:x 4.000000208616257, :y 0.1213061192893111} {:x 4.200000211596489, :y 0.10921487261772456} {:x 4.400000214576721, :y 0.09735043865851022} {:x 4.600000217556953, :y 0.08591145949323346} {:x 4.800000220537186, :y 0.07506220818247286} {:x 5.000000223517418, :y 0.06493048258684828} {:x 5.20000022649765, :y 0.055607450014672155} {:x 5.400000229477882, :y 0.04714920611442652} {:x 5.600000232458115, :y 0.039579731536155016} {:x 5.800000235438347, :y 0.032894883957948685} {:x 6.000000238418579, :y 0.027067050194033934} {:x 6.200000241398811, :y 0.02205009947188498} {:x 6.400000244379044, :y 0.017784318711150215} {:x 6.600000247359276, :y 0.0142010667082463} {:x 6.800000250339508, :y 0.011226949194167425} {:x 7.00000025331974, :y 0.0087873839421587} {:x 7.2000002562999725, :y 0.0068094886780661465} {:x 7.400000259280205, :y 0.005224280142137508} {:x 7.600000262260437, :y 0.003968217491884721} {:x 7.800000265240669, :y 0.0029841560661099708} {:x 8.000000268220901, :y 0.0022217984137491015} {:x 8.200000271201134, :y 0.0016377395144315963} {:x 8.400000274181366, :y 0.0011952040546767745} {:x 8.600000277161598, :y 8.635676066026832E-4} {:x 8.80000028014183, :y 6.177427874527661E-4} {:x 9.000000283122063, :y 4.374980068721774E-4} {:x 9.200000286102295, :y 3.067619778872984E-4} {:x 9.400000289082527, :y 2.1295313344564596E-4} {:x 9.60000029206276, :y 1.4636040255786668E-4} {:x 9.800000295042992, :y 9.959102700254978E-5})}), :marks ({:type \"line\", :from {:data \"2d937b55-f67f-4692-bfea-3247cff48f99\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :interpolate {:value \"step-before\"}, :fill {:value \"steelblue\"}, :fillOpacity {:value 0.4}, :stroke {:value \"steelblue\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}} {:type \"line\", :from {:data \"27215750-f8ba-46a9-8a74-52544d2e6620\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :interpolate {:value \"step-before\"}, :fill {:value \"green\"}, :fillOpacity {:value 0.4}, :stroke {:value \"green\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}} {:type \"line\", :from {:data \"98361f49-97b3-4a0f-b7eb-e8d0a5fe0da7\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"#FF29D2\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}} {:type \"line\", :from {:data \"d625cdcf-c0f7-46b2-9f39-b7d7d63a8056\"}, :properties {:enter {:x {:field \"data.x\", :scale \"x\"}, :y {:field \"data.y\", :scale \"y\"}, :stroke {:value \"orange\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}})}}"}
;; <=

;; @@

;; @@
