(ns com.yetanalytics.gen-openapi.generate.schema
  (:require [clojure.spec.alpha :as sp]
            [spec-tools.json-schema :as json-schema]
            [com.yetanalytics.gen-openapi.defaults :as defaults]))

(defn reference [kw-or-str]
  {"$ref" (str (defaults/get-default :schema-loc) (name kw-or-str))})

(defn owrap [pairs]
  {:type :object
   :properties pairs
   :required (mapv name (keys pairs))})

(defn clean-prefix [kw]
  (->> kw name rest rest (apply str) keyword))

(defn map-vals [f m]
  (reduce (fn [acc [k v :as item]]
            (assoc acc k (f v)))
          {}
          m))

(defn optional? [kw] 
  (= '(\? \#) (take 2 (name kw))))

(defn o
  "Yields a map compatible with json-schema that describes a json-schema \"object\".  

  (o {:key1 {:type :string} :key2 {:type :string}}) 

  yields

  {:type :object,
  :properties {:key2 {:type :string}, :key1 {:type :string}},
  :required [\"key2\" \"key1\"]}

  Add a ?# at the beginning of a property name to mark it as optional:

  (o {:?#key1 {:type :string}}) 

  yields

  {:type :object,
  :properties {:key1 {:type :string}},
  :required []}
  "
  [pairs]
  (let [ks (keys pairs)
        {optional-pairs true
         required-pairs false} (group-by #(optional? (key %)) pairs)

        required-pairs (apply hash-map (apply concat required-pairs))
        optional-pairs (reduce (fn [acc [k v]]
                                 (assoc acc (clean-prefix k) v)) {} optional-pairs)]

    {:type :object
     :properties (merge required-pairs optional-pairs)
     :required (mapv name (keys required-pairs))}))

(defn a
  "Takes a schema and returns a json-schema of an array of items of the input schema:
  (a {:some :schema}) 

  yields

  {:type :array
   :items {:some :schema}}"
  [schema]
  {:type :array
   :items schema})

(def transform-spec json-schema/transform)

(defn transform-kw [kw]
  (let [[pre rem] (split-at 2 (name kw))]
    (cond (= pre '(\? \#))
          kw
          (= pre '(\t \#))
          {:type (apply str rem)}
          (= pre '(\r \#))
          (reference (apply str rem))
          :else
          kw)))

(defn dsl [overall]
  (clojure.walk/prewalk (fn [form]
                          (cond
                            (sp/spec? form)
                            (transform-spec form)

                            (keyword? form)
                            (cond (sp/get-spec form)
                                  (transform-spec form)
                                  :else
                                  (transform-kw form))
                            :else form))
                        overall))

(def test-m {:allOf [(o {:?#username :t#string
                         :password {:type :string}
                         :id :r#ID})
                     {:oneOf [(o {:password :r#Password})
                              (o {:id :r#ID} )
                              (o {:height
                                  {:type :number :lessThan 5}})
                              (o {:height :t#number})
                              ::id]}]})
