(ns com.yetanalytics.gen-openapi.generate.params
  (:require
   [clojure.spec.alpha :as sp]
   [com.yetanalytics.gen-openapi.generate.schema :as schema]
   [com.yetanalytics.gen-openapi.defaults :as defaults]))

(defn process-name [kw]
  (let [[pre rem] (split-at 2 (name kw))]
    (if (= pre '(\? \#))
      {:required false
       :name (keyword (apply str rem))}
      {:required true
       :name kw})))

(defn decode-schema-keyword [kw];primitive type, and ref
  (if (sp/get-spec kw)
    (schema/transform-spec kw)
    (let [[pre rem] (split-at 2 (name kw))]
      (cond (= pre '(\t \#))
            {:type (apply str rem)}
            (= pre '(\r \#))
            (schema/reference (apply str rem))
            :else
            (let [message  "Keyword arguments in param vectors need to either be primitive types (like :t#string) or references (like :r#SomeSchema)"]
              #?(:clj (throw (Error. message))
                 :cljs (js/throw message)))))))

(defn process-non-name-arg [m item]
  (cond (sp/spec? item)
        (assoc m :schema (schema/transform-spec item))

        (keyword? item)
        (assoc m :schema
               (decode-schema-keyword item))
        
        (string? item)
        (assoc m :description item)

        (map? item)
        (merge item m)
        (nil? item)
        m))

(defn interpret-param-vec [v]
  (let [[nom & items] v
        m (process-name nom)
        m (reduce process-non-name-arg
                             m
                             items)
        m (if (:in m) m
              (assoc m :in (defaults/get-default :param-in)))]
    m))

(defn interpret-pair-to-json-param-map [[k v]]
  (let [base (process-name k)
        v (if (vector? v)
            v
            [v])
        with-value (reduce process-non-name-arg
                           base
                           v)
        with-location (if (:in with-value) with-value
                          (assoc with-value :in (defaults/get-default :param-in)))]
    with-location))

(defn params
  "takes a map from <param-name> to param descriptions.  a param description can be a schema shorthand keyword, an arbitrary map, a string, or a vector of any of these in any order "
  [params-map]
  (cond (map? params-map)
        (mapv interpret-pair-to-json-param-map params-map)
        :else
        (let [message "values to :params must be a map"]
          #?(:clj (throw (Error. message))
             :cljs (js/throw message)))))

(defn process-params [m]
  (clojure.walk/prewalk
   (fn [form]
     (cond (and (map? form)
                (seq (:params form)))
           (-> form (assoc :parameters (params (:params form)))
               (dissoc :params))
           :else form))
   m))
