(ns gen-openapi.test
  (:require
   [cheshire.core :as json]
   [com.yetanalytics.gen-openapi.splice :as splice]
   [com.yetanalytics.gen-openapi.generate :as g]
   [com.yetanalytics.gen-openapi.core :as goa]
   [com.yetanalytics.gen-openapi.generate.schema :as gs]))

(def err-400 {"$ref" "#/components/responses/error-400"})
(def err-401 {"$ref" "#/components/responses/error-401"})


(def components
  {:securitySchemes
   {:bearerAuth {:type :http
                 :scheme :bearer
                 :bearerFormat :JWT}}
   :responses
   {:error-400 (g/response "Bad Request" :r#Error)
    :error-401 (g/response "Unauthorized" :r#Error)}
   :schemas
   {:Account (gs/o {:homePage :r#IRL
                    :name :t#string})
    :Activity {:type :object
               :required [:id]
               :properties {:objectType {:type :string :pattern "String"}
                            :id :r#IRI
                            :definition {:type :object
                                         :properties {:name {}
                                                      :description {}
                                                      :type {}
                                                      :moreinfo {}
                                                      :extensions {}}}}}
    :Agent                              ;maybe important
    {:allOf [{:type :object
              :properties  {:name :t#string
                            :objectType :t#string}}
             :r#IFI]}
    :Group {:oneOf [{:properties {:objectType {:type :string :pattern "Group"}
                                  :name :t#string
                                  :member (gs/a :r#Agent)}
                     :required [:objectType :member]}
                    {:allOf [{:properties {:objectType {:type :string :pattern "Group"}
                                           :name {:type :string}
                                           :member (gs/a :r#Agent)}
                              :required [:objectType]}
                             :r#IFI]}]}
    :Actor {:oneOf [:r#Group
                    :r#Agent]}
    
    :Error (gs/o {:error :t#string})

    :IFI {:oneOf [(gs/o {:mbox :r#MailToIRI})
                  (gs/o {:mbox_sha1sum :t#string})
                  (gs/o {:openid :r#URI})
                  (gs/o {:account :r#Account})]}
    
    :IRI {:type :string :format :iri}
    :IRL :t#string
    :MailToIRI {:type :string :format :email}
    :KeyPair (gs/o {:api-key :t#string
                    :secret-key :t#string})

    :Person {:type :object
             :properties {:objectType {:type :string :pattern "Person"}
                          :name (gs/a :t#string)
                          :mbox (gs/a :r#MailToIRI)
                          :mbox_sha1sum (gs/a :t#string)
                          :openid* (gs/a :r#URI)
                          :account* (gs/a :r#Account)}
             :required [:objectType]}
    :Scopes (gs/o {:scopes (gs/a :t#string)})
    :ScopedKeyPair {:allOf [:r#KeyPair
                            :r#Scopes]}

    :statementId {:type :string}
    :Statement {:type :object :description "https://github.com/adlnet/xAPI-Spec/blob/master/xAPI-Data.md#20-statements"}

    :Timestamp {:type :string :format :date-time}

    :StatementResult {:type :object
                      :required [:statements]
                      :properties {:statements (gs/a :r#Statement)
                                   :more :r#IRL}}
    :URI {:type :string :format :uri}
    :UUID {:type :string :format :uuid}}})


(def lrsql-routes
  #{
    (splice/annotate-short
     ["/admin/account/login" :post []
      :route-name :lrsql.admin.account/login]
     {:description "Log into an existing account"
      :requestBody (g/request (gs/o {:username :t#string
                                     :password :t#string}))
      :operationId :login
      :responses {200 (g/response "Account ID and JWT"
                                  (gs/o {:account-id :t#string
                                         :json-web-token :t#string}))
                  400 err-400
                  401 err-401}})

    (splice/annotate-short
     ["/admin/account/create" :post []
      :route-name :lrsql.admin.account/create]
     {:description "Create new account"
      :requestBody (g/request (gs/o {:username :t#string 
                                     :password :t#string}))
      :operationId :create-account
      :security [{:bearerAuth []}]
      :responses {200 (g/response "ID of new account"
                                  (gs/o {:account-id {:type "string"}}))
                  400 err-401
                  401 err-401}})

    (splice/annotate-short
     ["/admin/account/password" :put []]
     {:description "Update account password"
      :requestBody (g/request (gs/o {:old-password :t#string
                                     :new-password :t#string}))
      :operationId :update-password
      :security [{:bearerAuth []}]
      :responses {200 (g/response "ID of updated account"
                                  (gs/o {:account-id :t#string}))
                  400 err-400
                  401 err-401}})

    (splice/annotate-short
     ["/admin/account" :get []
      :route-name :lrsql.admin.account/get]
     {:description "Get all accounts"
      :operationId :get-admin-accounts
      :security [{:bearerAuth []}]
      :responses {200 (g/response "Array of account objects"
                                  (gs/a (gs/o {:account-id :t#string 
                                               :username :t#string})))
                  401 err-401}})

    (splice/annotate-short
     ["/admin/me" :get []
      :route-name :lrsql.admin.me/get]
     {:description "Get account of querying account"
      :operationId :get-own-account
      :security [{:bearerAuth []}]
      :responses {200 (g/response  "Account object referring to own account"
                                   (gs/o {:account-id :t#string
                                          :username :t#string}))
                  401 err-401}})
    ;; Delete account (and associated credentials)
    (splice/annotate-short
     ["/admin/account" :delete []
      :route-name :lrsql.admin.account/delete]
     {:description "Delete account (and associated credentials)"
      :requestBody (g/request (gs/o {:account-id  :t#string}))
      :operationId :delete-admin-account
      :security [{:bearerAuth []}]
      :responses {200 (g/response  "ID of deleted account"
                                   (gs/o {:account-id :t#string}))
                  400 err-400
                  401 err-401}})


    (splice/annotate-short
     ["/admin/creds" :post []
      :route-name :lrsql.admin.creds/put]
     {:description "Create new API key pair w/scope set"
      :requestBody (g/request :r#Scopes)
      :operationId :create-api-keys
      :security [{:bearerAuth []}]
      :responses {400 err-400
                  401 err-401
                  200 (g/response "Object containing key, secret key, and array of scopes"
                                  :r#ScopedKeyPair)}})
    ;; Create or update new keys w/ scope set
    (splice/annotate-short
     ["/admin/creds" :put []
      :route-name :lrsql.admin.creds/post]
     {:description "Create or update new keys w/scope set"
      :requestBody  (g/request :r#ScopedKeyPair)
      :operationId :update-api-keys
      :security [{:bearerAuth []}]
      :responses {400 err-400
                  401 err-401
                  200 (g/response "Key, secret key, and scopes of updated account"
                                  :r#ScopedKeyPair)}})
    ;; Get current keys + scopes associated w/ account
    (splice/annotate-short
     ["/admin/creds" :get []
      :route-name :lrsql.admin.creds/get]
     {:description "Get current keys + scopes associated w/account"
      :operationId :get-api-keys
      :security [{:bearerAuth []}]
      :responses {200 (g/response "Array of scoped key pairs"
                                  (gs/a :r#ScopedKeyPair))
                  401 err-401}})
    ;; Delete API key pair and associated scopes
    (splice/annotate-short
     ["/admin/creds" :delete []
      :route-name :lrsql.admin.creds/delete]
     {:description  "Delete API key pair and associated scopes"
      :requestBody (g/request :r#KeyPair)
      :operationId :delete-api-key
      :security [{:bearerAuth []}]
      :responses {200 (g/response "Empty body" {})
                  400 err-400
                  401 err-401}})

    ["/admin/status" :get  []
     :route-name :lrsql.admin.status/get]})


(comment
  (require '[com.stuartsierra.component :as component]
           '[lrsql.system :as system]
           '[next.jdbc :as jdbc]
           '[com.yetanalytics.lrs.protocol :as lrsp]
           '[lrsql.admin.protocol :as adp]
           '[lrsql.util :as u]
           '[lrsql.util.actor :as a-util])
  (require
   '[lrsql.sqlite.record :as r]
   '[lrsql.lrs-test :refer [stmt-1 stmt-2 auth-ident auth-ident-oauth]])

  (def sys (system/system (r/map->SQLiteBackend {}) :test-sqlite-mem))

  (def sys' (component/start sys))

  (def real-lrsql-routes
    (get-in sys' [:webserver :service :io.pedestal.http/routes]))

  (->>
   (splice/final-process
    {:openapi "3.0.0"
     :info {:title "LRSQL"
            :version "0.7.2"}
     :externalDocs {:url "https://github.com/yetanalytics/lrsql/blob/main/doc/endpoints.md"}
     :components (gs/dsl components)}
    real-lrsql-routes)
   
   json/generate-string
   (spit "/home/daniel/Desktop/openapi.json")))


(splice/final-process
 {:openapi "3.1.0"
  :info {:meta "blank"}}
 lrsql-routes)
