(ns conndo.core
  (:require [datomic.api :as d]
            [clojure.core.async :as a]
            [clojure.tools.logging :as log]
            [io.rkn.conformity :as c]
            [clojure.string :as cs]
            )
  (:import [datomic Connection]
           [datomic.peer LocalConnection]))


(defprotocol IConnManager
  "Conn manager lifecycle and admin"
  (shutdown [this] [this force?]
    "Stop taking input and let conns expire. If force? is true, it will
    Immediately kill all conns.")
  (get-conn-map [this]
    "Get the current state map of all conns/usages in a promise channel.")
  (tx-unsub-all [this]
    "Unsubscribe all tx report queue subs."))

(defprotocol IConnLender
  "Manages conn lending"
  (get-conn [this db-uri] [this db-uri keepalive] [this db-uri keepalive opts]
    "Connects/yields an existing conn promise channel, registers the usage.
     A keepalive of LT zero means the conn will never be released.")
  (return-conn [this conn]
    "Registers that the caller is done with the conn.
    After all of a conn's usages are returned, it will wait for keepalive
   (ms) before release, or persist if there is no keepalive.")
  (tx-sub [this conn] [this conn chan]
    "Given a conn that is managed, returns a channel that will receive tx results.
    If a channel is provided, this channel will be subbed.")
  (tx-unsub [this conn] [this conn chan]
    "Given a conn, unsub all subs watching that topic. Given a conn and a (sub)
     channel, unsub that channel."))


(defn- split-cm
  "Split a conn map into conns to release and conns to keep"
  [cm & [force?]]
  (reduce
   (fn [out-map [conn {:keys [keepalive
                              used
                              last-return
                              before-release]
                       :as info}]]
     (update out-map
             (if (or force?
                     (and
                      (<= 0 keepalive)
                      (<= used 0) ;; no one is using it
                      (< keepalive ;; keepalive reached.
                         (- (System/currentTimeMillis)
                            last-return))))
               :release-cm
               :keep-cm)
             assoc conn info))
   {:keep-cm {}
    :release-cm {}}
   cm))

(defn- release-conns!
  "Release all unused/timed-out conns in the conn map, unsub any subs for their
   tx updates,  and return the map with those conns removed. Setting ?force to
   true will delete all conns."
  [cm tx-pub & {:keys [release-locals?
                force?]}]
  (let [{:keys [keep-cm
                release-cm]} (split-cm cm force?)]
    (doseq [[conn {:keys [before-release
                          stop-queue-watcher]}] release-cm
            ;; run a before-release callback if present
            :let [_ (when before-release (before-release conn))]
            :when (or release-locals? ;; release all
                      (not (instance? LocalConnection conn)))]
      (try
        ;; Remove any subs for that conn
        (a/unsub-all tx-pub conn)
        ;; remove any tx queue
        (d/remove-tx-report-queue conn)
        ;; Release
        ;; release is async, so we can't really do it. Ever.
        ;; (d/release conn)
        (catch Exception _
          (log/warnf "Couldn't release conn: %s" conn))))

    keep-cm))

(defn- poll [^java.util.concurrent.LinkedBlockingQueue q]
  (.poll q))

(defn new-conn-manager [& {:keys [conn-map
                                  poll-ms
                                  release-locals?
                                  norms
                                  default-keepalive]
                           :or {conn-map {}
                                poll-ms 1000
                                release-locals? true
                                default-keepalive 60000}}]
  (log/info "Starting connection manager...")
  (let [use-chan (a/chan)
        return-chan (a/chan)
        control-chan (a/chan)
        error-chan (a/chan)
        ;; tx report chan receives a tuple of [conn tx-result]
        tx-report-chan (a/chan)
        ;; Published with the conn as topic
        tx-pub (a/pub tx-report-chan first)
        event-loop
        (a/go-loop [cm conn-map
                    shutdown? false]
          ;; Before anything else, make sure to check any queues,
          ;; drain and publish.
          (doseq [conn (keys cm)
                  :let [q (d/tx-report-queue conn)]
                  ;; drain the queue
                  tx-result (take-while identity
                                        (repeatedly #(poll q)))]
            (log/debugf "Publishing tx result: %s for conn: %s" tx-result conn)
            (a/>! tx-report-chan [conn tx-result]))

          ;; Wait for one of the control channels, or execute cleanup if timeout
          ;; is reached.
          (let [timeout-chan (a/timeout poll-ms)
                [v p] (a/alts!
                       [use-chan
                        return-chan
                        control-chan
                        timeout-chan])]
            (log/debugf "state: %s" cm)

            (condp = p
              timeout-chan
              ;; When the poll time passes, old conns are cleared!
              (let [keep-cm (release-conns! cm
                                            tx-pub
                                            :release-locals? release-locals?)]
                (if (and (empty? keep-cm) shutdown?)
                  (do
                    ;; Unsub anything else...
                    (a/unsub-all tx-pub)
                    (log/info "Shutdown complete.")
                    ::shutdown)
                  (recur keep-cm shutdown?)))
              use-chan
              (let [{:keys [conn-promise db-uri keepalive opts]} v]
                (if shutdown?
                  (do
                    (log/warn "Can't get conn, manager shutting down.")
                    (a/>! conn-promise
                          ::conn-mgr-shutdown)
                    (recur cm true))
                  (let [;; if this is an in-memory conn and has been released,
                        ;; we need to ensure it is created before connection
                        _ (when (cs/starts-with? db-uri "datomic:mem://")
                            (d/create-database db-uri))
                        conn
                        (try (d/connect db-uri)
                             (catch Exception ex
                               (log/warnf "Connection error db-uri: %s" db-uri)
                               ::connection-error))
                        new-conn? (not (get cm conn))
                        {before-release-fn :before-release
                         on-new-fn :on-new
                         conn-norms :norms} opts]

                    ;; when we make a new connection to the conn
                    (when new-conn?
                      ;; ensure norms when they are present, and the conn is new
                      (when-let [norms (or conn-norms norms)]
                        (c/ensure-conforms conn norms))
                      ;; when an on-new fn is defined, run it
                      (when on-new-fn
                        (on-new-fn conn)))
                    (a/>! conn-promise
                          conn)
                    (if (= ::connection-error
                           conn)
                      (recur cm false)
                      (do (log/debugf "Granting conn for event: %s" v)
                        (recur
                         (update cm
                                 conn
                                 #(cond-> (-> %
                                              (assoc :keepalive keepalive)
                                              (update :used (fnil inc 0)))
                                    before-release-fn
                                    (assoc :before-release
                                           before-release-fn)))
                         false))))))
              return-chan
              (let [{:keys [conn]} v]
                (log/debug "Returning conn.")
                (recur (update cm
                               conn
                               #(-> %
                                    (assoc :last-return (System/currentTimeMillis))
                                    (update :used dec))) shutdown?))
              control-chan
              (let [{:keys [command options]} v]
                (case command
                  ::shutdown (let [{:keys [force?]} options]
                               (log/debugf "Shutdown initiated, options: %s" options)
                               (if force?
                                 (do
                                   ;; release everything
                                   (release-conns! cm
                                                   tx-pub
                                                   :release-locals? release-locals?
                                                   :force? true)
                                   ;; Unsub anything else
                                   (a/unsub-all tx-pub)
                                   ;; end the loop
                                   ::forced-shutdown)
                                 ;; Setting all keepalives to zero
                                 ;; will have the effect of waiting
                                 ;; for users to return them.
                                 (recur (into {}
                                              (for [[conn info] cm]
                                                [conn (assoc info
                                                             :keepalive 0)]))
                                        true)))
                  ::get-conn-map
                  (let [{:keys [conn-map-promise]} options]
                    (a/>! conn-map-promise cm)
                    (recur cm shutdown?))
                  (do (a/>! error-chan {:type ::unknown-command
                                        :command command})
                      (recur cm shutdown?)))))))]


    (reify
      IConnManager
      (get-conn-map [_]
        (let [conn-map-promise (a/promise-chan)]
          (a/go (a/>! control-chan
                      {:command ::get-conn-map
                       :options {:conn-map-promise
                                 conn-map-promise}}))
          conn-map-promise))
      (shutdown [_]
        (a/go
          (a/>! control-chan
                {:command ::shutdown})
          (a/<! event-loop)))
      (shutdown [_ force?]
        (a/>!! control-chan
               {:command ::shutdown
                :options {:force? true}})
        (a/<!! event-loop))
      (tx-unsub-all [_]
        (a/unsub-all tx-pub))
      IConnLender
      (get-conn [this db-uri]
        (get-conn this db-uri default-keepalive))
      (get-conn [this db-uri keepalive]
        (get-conn this db-uri (or keepalive
                                  default-keepalive) {}))
      (get-conn [_ db-uri keepalive opts]
        (let [conn-promise (a/promise-chan)]
          (a/go (a/>! use-chan
                      {:conn-promise conn-promise
                       :db-uri db-uri
                       :keepalive (or keepalive
                                      default-keepalive)
                       :opts opts}))
          conn-promise))
      (return-conn [_ conn]
        (a/go
          (a/>! return-chan
                {:conn conn})))
      (tx-sub [this conn]
        (tx-sub this conn (a/chan))) ;; TODO: buffer for default sub chans
      (tx-sub [_ conn chan]
        (a/sub tx-pub conn chan))
      (tx-unsub [_ conn]
        (a/unsub-all tx-pub conn))
      (tx-unsub [_ conn chan]
        (a/unsub tx-pub conn chan)))))
